package ru.yandex.direct.core.entity.banner.repository;

import java.time.LocalDateTime;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.Table;
import org.jooq.impl.DSL;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.banner.container.BannerRepositoryContainer;
import ru.yandex.direct.core.entity.banner.model.Banner;
import ru.yandex.direct.core.entity.banner.model.InternalBanner;
import ru.yandex.direct.core.entity.banner.repository.filter.BannerFilterFactory;
import ru.yandex.direct.core.entity.banner.repository.type.BannerRepositoryTypeSupportFacade;
import ru.yandex.direct.dbschema.ppc.enums.BannersBannerType;
import ru.yandex.direct.dbschema.ppc.enums.BannersStatusarch;
import ru.yandex.direct.dbschema.ppc.tables.Banners;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.multitype.entity.LimitOffset;
import ru.yandex.direct.multitype.repository.TypedRepository;
import ru.yandex.direct.multitype.repository.filter.ConditionFilter;
import ru.yandex.direct.multitype.repository.filter.Filter;

import static java.util.Collections.emptyList;
import static java.util.Collections.emptyMap;
import static java.util.Collections.emptySet;
import static ru.yandex.direct.core.entity.banner.repository.filter.BannerFilterFactory.bannerAdGroupIdFilter;
import static ru.yandex.direct.core.entity.banner.repository.filter.BannerFilterFactory.bannerBannerTypesFilter;
import static ru.yandex.direct.core.entity.banner.repository.filter.BannerFilterFactory.bannerCampaignIdFilter;
import static ru.yandex.direct.core.entity.banner.repository.filter.BannerFilterFactory.bannerIdFilter;
import static ru.yandex.direct.core.entity.banner.repository.filter.BannerFilterFactory.bannerStatusArchFilter;
import static ru.yandex.direct.core.entity.banner.repository.filter.BannerFilterFactory.clientIdAndAdGroupIdsAndBannerTypesFilter;
import static ru.yandex.direct.core.entity.banner.repository.filter.BannerFilterFactory.clientIdAndCreativeIdsFilter;
import static ru.yandex.direct.core.entity.banner.repository.filter.BannerFilterFactory.internalBannersWaitingForModerationTill;
import static ru.yandex.direct.core.entity.banner.repository.filter.BannerFilterFactory.notArchivedInternalBannerWithStatusShowYesTemplateIdFilter;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNERS;
import static ru.yandex.direct.dbschema.ppc.Tables.CAMPAIGNS;
import static ru.yandex.direct.multitype.repository.filter.ConditionFilterFactory.multipleConditionFilter;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.utils.FunctionalUtils.selectList;

@Repository
@ParametersAreNonnullByDefault
public class BannerTypedRepository extends TypedRepository<Banner, BannersBannerType,
        BannerRepositoryContainer, BannerRepositoryContainer> {

    @Autowired
    public BannerTypedRepository(BannerRepositoryTypeSupportFacade typeSupportFacade,
                                 DslContextProvider dslContextProvider) {
        super(dslContextProvider, typeSupportFacade);
    }

    @Override
    protected ConditionFilter getIdFilter(Collection<Long> modelIds) {
        return bannerIdFilter(modelIds);
    }

    @Override
    protected Table<?> getBaseTable() {
        return BANNERS;
    }

    /**
     * Получает список баннеров по списку id их групп
     *
     * @param adGroupIds список id групп извлекаемых баннеров
     * @param shard      шард
     * @return список баннеров
     */
    public List<Banner> getBannersByGroupIds(int shard, Collection<Long> adGroupIds) {
        return getTypedByFilter(shard, bannerAdGroupIdFilter(adGroupIds));
    }

    public <T> List<T> getBannersByGroupIds(int shard, Collection<Long> adGroupIds, Class<T> tClass) {
        List<Banner> banners = getBannersByGroupIds(shard, adGroupIds);
        return selectList(banners, tClass);
    }

    public <T extends Banner> List<T> getBannersByGroupIds(int shard,
                                                           Collection<Long> adGroupIds,
                                                           ClientId clientId,
                                                           Class<T> tClass) {
        //noinspection unchecked
        return (List<T>) getTyped(dslContextProvider.ppc(shard),
                clientIdAndAdGroupIdsAndBannerTypesFilter(clientId, adGroupIds, tClass));
    }

    /**
     * Этот метод (и похожие) скорее для интерфейсов, т.к. вернёт полностью заполненные модели, в отличии от getSafely.
     * А для классов более эффективно будет делать так:
     * getSafely(shard, bannerCampaignIdFilter(campaignIds), NewTextBanner.class)
     * поскольку тогда не будем join-инть неприменимые таблицы.
     */
    public <T extends Banner> List<T> getBannersByCampaignIdsAndClass(
            int shard,
            Collection<Long> campaignIds,
            Class<T> tClass) {
        //noinspection unchecked
        return (List<T>) getTyped(dslContextProvider.ppc(shard),
                multipleConditionFilter(
                        bannerBannerTypesFilter(tClass),
                        bannerCampaignIdFilter(campaignIds)));
    }

    public <T> List<T> getBannersByCreativeIds(int shard,
                                               ClientId clientId,
                                               Collection<Long> creativeIds,
                                               Class<T> tClass) {
        Filter filter = clientIdAndCreativeIdsFilter(clientId, creativeIds);
        List<Banner> banners = getTyped(dslContextProvider.ppc(shard), filter);
        return selectList(banners, tClass);
    }

    public List<Banner> getBannersByCampaignIds(int shard,
                                                Collection<Long> campaignIds) {
        return getBannersByCampaignIds(shard, campaignIds, null);
    }

    public List<Banner> getNonArchivedBannersByCampaignIds(int shard,
                                                           Collection<Long> campaignIds,
                                                           @Nullable LimitOffset limitOffset) {
        return getTyped(dslContextProvider.ppc(shard),
                multipleConditionFilter(bannerCampaignIdFilter(campaignIds),
                        bannerStatusArchFilter(BannersStatusarch.No)),
                limitOffset);
    }

    public List<Banner> getBannersByCampaignIds(int shard,
                                                Collection<Long> campaignIds,
                                                @Nullable LimitOffset limitOffset) {
        return getTyped(dslContextProvider.ppc(shard), bannerCampaignIdFilter(campaignIds), limitOffset);
    }

    public List<InternalBanner> getNoArchivedInternalBannersWithStatusShowYesByTemplateIds(
            int shard, Collection<Long> templateIds) {
        return mapList(getTypedByFilter(shard, notArchivedInternalBannerWithStatusShowYesTemplateIdFilter(templateIds)),
                InternalBanner.class::cast);
    }

    /**
     * Получить id и статусы всех баннеров, по которым не получили ответ от модерации за указанное время
     *
     * @param borderDateTime - максимальное время отправки баннера
     * @return map id банера -> статус модерации
     */
    public List<InternalBanner> getInternalBannersOnModerationOlderThan(int shard,
                                                                        LocalDateTime borderDateTime) {

        return mapList(getTypedByFilter(shard, internalBannersWaitingForModerationTill(borderDateTime)),
                InternalBanner.class::cast);
    }

    public List<InternalBanner> getNoArchivedInternalBannersStoppedByUrlMonitoring(int shard) {
        return mapList(getTypedByFilter(shard,
                        BannerFilterFactory.notArchivedInternalBannersStoppedByUrlMonitoringFilter()),
                InternalBanner.class::cast
        );
    }

    public List<Banner> getBanners(int shard,
                                   Collection<Long> bannerIds,
                                   @Nullable LimitOffset limitOffset) {
        return getTyped(dslContextProvider.ppc(shard), bannerIdFilter(bannerIds), limitOffset);
    }

    public List<InternalBanner> getInternalBannersByIds(int shard, Collection<Long> bannerIds) {
        return getSafely(shard, bannerIdFilter(bannerIds), InternalBanner.class);
    }

    public Map<Long, BannersBannerType> getClientBannerIdsWithType(int shard, ClientId clientId,
                                                                   Collection<Long> bannerIds) {
        if (bannerIds.isEmpty()) {
            return emptyMap();
        }
        return dslContextProvider.ppc(shard)
                .select(BANNERS.BID, BANNERS.BANNER_TYPE)
                .from(BANNERS)
                .join(CAMPAIGNS).on(CAMPAIGNS.CID.eq(BANNERS.CID))
                .where(CAMPAIGNS.CLIENT_ID.eq(clientId.asLong()))
                .and(BANNERS.BID.in(bannerIds))
                .fetch()
                .intoMap(BANNERS.BID, BANNERS.BANNER_TYPE);
    }

    public Set<Long> getClientBannerIds(int shard, ClientId clientId, Collection<Long> bannerIds) {
        if (bannerIds.isEmpty()) {
            return emptySet();
        }
        return dslContextProvider.ppc(shard)
                .select(BANNERS.BID)
                .from(BANNERS)
                .join(CAMPAIGNS).on(CAMPAIGNS.CID.eq(BANNERS.CID))
                .where(CAMPAIGNS.CLIENT_ID.eq(clientId.asLong()))
                .and(BANNERS.BID.in(bannerIds))
                .fetchSet(BANNERS.BID);
    }

    /**
     * Получает map id банера -> id группы по списку id баннеров
     *
     * @param bannerIds список id баннеров
     * @return map id банера -> id группы
     */
    public Map<Long, Long> getBannerIdToAdGroupId(int shard, Collection<Long> bannerIds) {
        if (bannerIds.isEmpty()) {
            return emptyMap();
        }

        return dslContextProvider.ppc(shard)
                .select(Banners.BANNERS.BID, Banners.BANNERS.PID)
                .from(Banners.BANNERS)
                .where(Banners.BANNERS.BID.in(bannerIds))
                .fetchMap(Banners.BANNERS.BID, Banners.BANNERS.PID);
    }

    public static class BannerData {
        public long bannerId;
        public long campaignId;
        public long adgroupId;
        public BannersBannerType bannerType;

        public BannerData(long bannerId, long campaignId, long adgroupId, BannersBannerType bannerType) {
            this.bannerId = bannerId;
            this.campaignId = campaignId;
            this.adgroupId = adgroupId;
            this.bannerType = bannerType;
        }
    }

    public List<BannerData> getOrderedBannersByCampaignIds(int shard,
                                                           Collection<Long> campaignIds,
                                                           long limit,
                                                           @Nullable Long minBidExclusive) {
        if (campaignIds.isEmpty()) {
            return emptyList();
        }
        return dslContextProvider.ppc(shard)
                .select(BANNERS.BID, BANNERS.CID, BANNERS.PID, BANNERS.BANNER_TYPE)
                .from(BANNERS)
                .where(BANNERS.CID.in(campaignIds))
                .and(minBidExclusive != null ? BANNERS.BID.gt(minBidExclusive) : DSL.trueCondition())
                .orderBy(BANNERS.BID)
                .limit(limit)
                .fetch(r -> new BannerData(
                        r.get(BANNERS.BID),
                        r.get(BANNERS.CID),
                        r.get(BANNERS.PID),
                        r.get(BANNERS.BANNER_TYPE)
                ));
    }
}
