package ru.yandex.direct.core.entity.banner.repository;

import java.util.Collection;
import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.DSLContext;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.dbutil.QueryWithoutIndex;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;

import static ru.yandex.direct.dbschema.ppc.Tables.BANNERS_TO_FILL_LANGUAGE_QUEUE;

/**
 * Репозиторий для работы с таблицей ppc.banners_to_fill_language_queue
 */
@Repository
@ParametersAreNonnullByDefault
public class BannersToFillLanguageQueueRepository {

    private final DslContextProvider dslContextProvider;

    @Autowired
    public BannersToFillLanguageQueueRepository(DslContextProvider dslContextProvider) {
        this.dslContextProvider = dslContextProvider;
    }

    /**
     * Добавляет объекты в очередь.
     *
     * @param shard шард
     * @param bids  id баннеров, добавляемых в очередь
     */
    public void addItems(int shard, Collection<Long> bids) {
        addItems(dslContextProvider.ppc(shard), bids);
    }

    public void addItems(DSLContext context, Collection<Long> bids) {
        var insertStep = context
                .insertInto(BANNERS_TO_FILL_LANGUAGE_QUEUE)
                .columns(BANNERS_TO_FILL_LANGUAGE_QUEUE.BID);

        bids.forEach(insertStep::values);

        insertStep
                .onDuplicateKeyIgnore()
                .execute();
    }

    /**
     * Возвращает объекты из очереди.
     *
     * @param shard шард
     * @param count ограничение на количество возвращаемых объектов
     * @return объекты из очереди (не более count штук)
     */
    @QueryWithoutIndex(value = "В запросе есть константный limit")
    public List<Long> peekItems(int shard, int count) {
        return dslContextProvider.ppc(shard)
                .select(BANNERS_TO_FILL_LANGUAGE_QUEUE.BID)
                .from(BANNERS_TO_FILL_LANGUAGE_QUEUE)
                .limit(count)
                .fetch(BANNERS_TO_FILL_LANGUAGE_QUEUE.BID);
    }

    /**
     * Удаляет объекты из очереди.
     *
     * @param shard     шард
     * @param bannerIds id объектов
     */
    public void removeItems(int shard, Collection<Long> bannerIds) {
        dslContextProvider.ppc(shard)
                .deleteFrom(BANNERS_TO_FILL_LANGUAGE_QUEUE)
                .where(BANNERS_TO_FILL_LANGUAGE_QUEUE.BID.in(bannerIds))
                .execute();
    }
}
