package ru.yandex.direct.core.entity.banner.repository;

import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.atomic.AtomicLong;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.commons.lang3.tuple.Triple;
import org.jooq.Condition;
import org.jooq.DSLContext;
import org.jooq.Row2;
import org.jooq.impl.DSL;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.common.util.RepositoryUtils;
import ru.yandex.direct.core.entity.banner.container.ModerateBannerPagesSyncResult;
import ru.yandex.direct.core.entity.banner.model.ModerateBannerPage;
import ru.yandex.direct.core.entity.banner.model.StatusModerateBannerPage;
import ru.yandex.direct.core.entity.banner.model.StatusModerateOperator;
import ru.yandex.direct.dbschema.ppc.enums.ModerateBannerPagesStatusmoderate;
import ru.yandex.direct.dbschema.ppc.tables.ModerateBannerPages;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;

import static java.util.Collections.emptyMap;
import static java.util.stream.Collectors.toList;
import static org.jooq.impl.DSL.noCondition;
import static org.jooq.impl.DSL.row;
import static ru.yandex.direct.common.jooqmapperex.ReaderWriterBuildersEx.booleanProperty;
import static ru.yandex.direct.dbschema.ppc.Tables.MODERATE_BANNER_PAGES;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.convertibleProperty;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Component
@ParametersAreNonnullByDefault
public class ModerateBannerPagesRepository {

    public static final int MAX_BANNERS_PER_REQUEST = 100;
    private final ShardHelper shardHelper;
    private final DslContextProvider dslContextProvider;

    private final JooqMapperWithSupplier<ModerateBannerPage> moderateBannerPageMapper =
            createModerateBannerPageMapper();

    @Autowired
    public ModerateBannerPagesRepository(ShardHelper shardHelper, DslContextProvider dslContextProvider) {
        this.shardHelper = shardHelper;
        this.dslContextProvider = dslContextProvider;
    }

    /**
     * Синхронизирует записи в moderate_banner_pages.
     *
     * @param dslContext                    контекст бд
     * @param bannerIdToModerateBannerPages привязки bannerId -> page, которые должны остаться в таблице
     *                                      moderate_banner_pages
     * @return результат синхронизации
     */
    public ModerateBannerPagesSyncResult syncModerateBannerPages(DSLContext dslContext,
                                                                 Map<Long, List<ModerateBannerPage>> bannerIdToModerateBannerPages) {
        ModerateBannerPagesSyncResult result = new ModerateBannerPagesSyncResult();
        markDeletedModerateBannerPagesNotListedIn(dslContext, bannerIdToModerateBannerPages, result);
        addModerateBannerPages(dslContext, bannerIdToModerateBannerPages, result);
        return result;
    }

    /**
     * Пометить удаленными записи moderate_banner_pages, которые не содержатся в bannerIdToModerateBannerPages
     *
     * @param dslContext                    контекст бд
     * @param bannerIdToModerateBannerPages привязки bannerId -> page, которые должны остаться в таблице
     *                                      moderate_banner_pages
     */
    private void markDeletedModerateBannerPagesNotListedIn(DSLContext dslContext,
                                                           Map<Long, List<ModerateBannerPage>> bannerIdToModerateBannerPages,
                                                           ModerateBannerPagesSyncResult result) {
        Condition rowsToDeleteCondition = EntryStream.of(bannerIdToModerateBannerPages)
                .mapKeyValue((bannerId, moderateBannerPages) -> {
                    List<Long> pageIds = mapList(moderateBannerPages, ModerateBannerPage::getPageId);
                    return MODERATE_BANNER_PAGES.BID.eq(bannerId)
                            .and(MODERATE_BANNER_PAGES.PAGE_ID.notIn(pageIds));
                })
                .reduce(Condition::or)
                .orElse(DSL.falseCondition());

        dslContext
                .select(MODERATE_BANNER_PAGES.BID, MODERATE_BANNER_PAGES.PAGE_ID,
                        MODERATE_BANNER_PAGES.MODERATE_BANNER_PAGE_ID)
                .from(MODERATE_BANNER_PAGES)
                .where(rowsToDeleteCondition)
                .and(MODERATE_BANNER_PAGES.IS_REMOVED.eq(RepositoryUtils.FALSE))
                .fetch()
                .forEach(record -> {
                    long rowId = record.get(MODERATE_BANNER_PAGES.MODERATE_BANNER_PAGE_ID);
                    long bannerId = record.get(MODERATE_BANNER_PAGES.BID);
                    long pageId = record.get(MODERATE_BANNER_PAGES.PAGE_ID);
                    result.getDeletedBannerIdToPageIds()
                            .computeIfAbsent(bannerId, id -> new ArrayList<>())
                            .add(pageId);
                    result.getDeletedModerateBannerPageIds()
                            .add(rowId);
                });
        dslContext.update(MODERATE_BANNER_PAGES)
                .set(MODERATE_BANNER_PAGES.IS_REMOVED, RepositoryUtils.TRUE)
                .where(MODERATE_BANNER_PAGES.MODERATE_BANNER_PAGE_ID.in(result.getDeletedModerateBannerPageIds()))
                .execute();
    }

    /**
     * Добавляет переданные записи moderate_banner_pages для переданных привязок
     *
     * @param dslContext                    контекст бд
     * @param bannerIdToModerateBannerPages привязки bannerId -> page, которые надо добавить
     */
    private void addModerateBannerPages(DSLContext dslContext,
                                        Map<Long, List<ModerateBannerPage>> bannerIdToModerateBannerPages,
                                        ModerateBannerPagesSyncResult result) {
        List<Row2<Long, Long>> rowsToAdd = bannerIdToModerateBannerPages.values().stream()
                .flatMap(Collection::stream)
                .map(moderateBannerPage -> DSL.row(moderateBannerPage.getBannerId(), moderateBannerPage.getPageId()))
                .collect(toList());
        if (rowsToAdd.isEmpty()) {
            return;
        }

        Set<Triple<Long, Long, Long>> rowsExisted = dslContext
                .select(MODERATE_BANNER_PAGES.BID, MODERATE_BANNER_PAGES.PAGE_ID, MODERATE_BANNER_PAGES.IS_REMOVED)
                .from(MODERATE_BANNER_PAGES)
                .where(row(MODERATE_BANNER_PAGES.BID, MODERATE_BANNER_PAGES.PAGE_ID).in(rowsToAdd))
                .fetch()
                .intoSet(record -> Triple.of(record.getValue(MODERATE_BANNER_PAGES.BID),
                        record.getValue(MODERATE_BANNER_PAGES.PAGE_ID),
                        record.getValue(MODERATE_BANNER_PAGES.IS_REMOVED)));

        Iterator<Long> ids = shardHelper.generateModerateBannerPageIds(rowsToAdd.size() - rowsExisted.size())
                .iterator();

        List<ModerateBannerPage> moderateBannerPagesToAdd = bannerIdToModerateBannerPages.values().stream()
                .flatMap(Collection::stream)
                .filter(moderateBannerPage -> {
                    Triple<Long, Long, Long> existingKey = Triple.of(moderateBannerPage.getBannerId(), moderateBannerPage.getPageId(), 0L);
                    Triple<Long, Long, Long> deletedKey = Triple.of(moderateBannerPage.getBannerId(), moderateBannerPage.getPageId(), 1L);
                    return !rowsExisted.contains(existingKey) && !rowsExisted.contains(deletedKey);
                })
                .map(moderateBannerPage -> moderateBannerPage
                        .withId(ids.next())
                        .withStatusModerate(StatusModerateBannerPage.READY)
                        .withStatusModerateOperator(StatusModerateOperator.NONE)
                        .withIsRemoved(false)
                        .withCreateTime(LocalDateTime.now()))
                .collect(toList());

        List<Row2<Long, Long>> moderateBannerPagesToRestore = bannerIdToModerateBannerPages.values().stream()
                .flatMap(Collection::stream)
                .filter(moderateBannerPage -> {
                    Triple<Long, Long, Long> key = Triple.of(moderateBannerPage.getBannerId(), moderateBannerPage.getPageId(), 1L);
                    return rowsExisted.contains(key);
                })
                .map(r -> DSL.row(r.getBannerId(), r.getPageId()))
                .collect(toList());

        new InsertHelper<>(dslContext, MODERATE_BANNER_PAGES)
                .addAll(moderateBannerPageMapper, moderateBannerPagesToAdd)
                .executeIfRecordsAdded();

        if (!moderateBannerPagesToRestore.isEmpty()) {
            dslContext
                    .update(ModerateBannerPages.MODERATE_BANNER_PAGES)
                    .set(ModerateBannerPages.MODERATE_BANNER_PAGES.IS_REMOVED, RepositoryUtils.FALSE)
                    .where(row(MODERATE_BANNER_PAGES.BID, MODERATE_BANNER_PAGES.PAGE_ID).in(moderateBannerPagesToRestore))
                    .execute();
        }

        StreamEx.of(moderateBannerPagesToAdd)
                .forEach(moderateBannerPage -> {
                    long bannerId = moderateBannerPage.getBannerId();
                    long pageId = moderateBannerPage.getPageId();
                    result.getAddedBannerIdToPageIds()
                            .computeIfAbsent(bannerId, id -> new ArrayList<>())
                            .add(pageId);
                });
    }

    /**
     * Найти список существующих модерируемых операторов щитов по bannerId + pageIds.
     * <p>
     * Используется для проверки того что в таблице moderate_banner_pages есть запись о модерации баннера
     * определенных операторов щитов
     *
     * @param bannerId - идентификатор баннера
     * @param pageIds  - идентификаторы операторов щитов
     * @return список существующих модерируемых операторов щитов (page_id)
     */
    public Set<Long> getExistingModeratePageIds(int shard, Long bannerId, Collection<Long> pageIds) {

        return dslContextProvider.ppc(shard)
                .select(MODERATE_BANNER_PAGES.PAGE_ID)
                .from(MODERATE_BANNER_PAGES)
                .where(MODERATE_BANNER_PAGES.BID.eq(bannerId))
                .and(MODERATE_BANNER_PAGES.PAGE_ID.in(pageIds))
                .and(MODERATE_BANNER_PAGES.IS_REMOVED.eq(RepositoryUtils.FALSE))
                .fetchSet(MODERATE_BANNER_PAGES.PAGE_ID);
    }

    /**
     * Получить список bannerPageId (moderate_banner_pages.moderate_banner_page_id)
     *
     * @param bannerId - идентификатор баннера
     * @param pageIds  - идентификаторы операторов щитов
     * @return - список bannerPageId (moderate_banner_page_id)
     */
    public Set<Long> getModerateBannerPageIds(int shard, Long bannerId, Collection<Long> pageIds) {
        return dslContextProvider.ppc(shard)
                .select(MODERATE_BANNER_PAGES.MODERATE_BANNER_PAGE_ID)
                .from(MODERATE_BANNER_PAGES)
                .where(MODERATE_BANNER_PAGES.BID.eq(bannerId))
                .and(MODERATE_BANNER_PAGES.PAGE_ID.in(pageIds))
                .and(MODERATE_BANNER_PAGES.IS_REMOVED.eq(RepositoryUtils.FALSE))
                .fetchSet(MODERATE_BANNER_PAGES.MODERATE_BANNER_PAGE_ID);
    }


    /**
     * Получить список ModerateBannerPage (moderate_banner_pages)
     *
     * @param dslContext
     * @param bannerId   - идентификатор баннера
     * @param pageIds    - идентификаторы операторов щитов
     * @return - список ModerateBannerPage
     */
    public List<ModerateBannerPage> getModerateBannerPages(DSLContext dslContext, Long bannerId,
                                                           Collection<Long> pageIds) {
        return dslContext
                .select(moderateBannerPageMapper.getFieldsToRead())
                .from(MODERATE_BANNER_PAGES)
                .where(MODERATE_BANNER_PAGES.BID.eq(bannerId))
                .and(MODERATE_BANNER_PAGES.PAGE_ID.in(pageIds))
                .and(MODERATE_BANNER_PAGES.IS_REMOVED.eq(RepositoryUtils.FALSE))
                .fetch(moderateBannerPageMapper::fromDb);
    }


    /**
     * Получить список ModerateBannerPage (moderate_banner_pages) по каждому переданному ID банера
     *
     * @param shard     - номер шарда
     * @param bannerIds - идентификаторы баннеров
     * @return - список ModerateBannerPage, разбитый по баннерам
     */
    public Map<Long, List<ModerateBannerPage>> getModerateBannerPages(int shard, List<Long> bannerIds) {
        return dslContextProvider.ppc(shard)
                .select(moderateBannerPageMapper.getFieldsToRead())
                .from(MODERATE_BANNER_PAGES)
                .where(MODERATE_BANNER_PAGES.BID.in(bannerIds))
                .and(MODERATE_BANNER_PAGES.IS_REMOVED.eq(RepositoryUtils.FALSE))
                .fetchGroups(MODERATE_BANNER_PAGES.BID, moderateBannerPageMapper::fromDb);
    }


    /**
     * Получить список ModerateBannerPage (moderate_banner_pages)
     *
     * @param shard              Шард
     * @param pageIdsByBannerIds списки page id, сгруппированные по баннерам
     * @return - список ModerateBannerPage
     */
    public Map<Long, List<ModerateBannerPage>> getModerateBannerPagesByBannerId(
            int shard,
            Map<Long, Set<Long>> pageIdsByBannerIds) {
        if (pageIdsByBannerIds.isEmpty()) {
            return emptyMap();
        }

        var result = new HashMap<Long, List<ModerateBannerPage>>();

        AtomicLong counter = new AtomicLong(0);
        StreamEx.of(pageIdsByBannerIds.entrySet())
                .groupRuns((a, b) -> counter.incrementAndGet() % MAX_BANNERS_PER_REQUEST != 0)
                .forEach(entries -> {
                    Condition condition = EntryStream.of(pageIdsByBannerIds)
                            .mapKeyValue((bannerId, pageIds) ->
                                    MODERATE_BANNER_PAGES.BID.eq(bannerId)
                                            .and(MODERATE_BANNER_PAGES.PAGE_ID.in(pageIds)))
                            .reduce(noCondition(), Condition::or);

                    result.putAll(dslContextProvider.ppc(shard)
                            .select(moderateBannerPageMapper.getFieldsToRead())
                            .from(MODERATE_BANNER_PAGES)
                            .where(condition)
                            .and(MODERATE_BANNER_PAGES.IS_REMOVED.eq(RepositoryUtils.FALSE))
                            .fetchGroups(MODERATE_BANNER_PAGES.BID, moderateBannerPageMapper::fromDb));
                });

        return result;
    }

    /**
     * Получить список ModerateBannerPage (moderate_banner_pages)
     * аналог getModerateBannerPagesByBannerIds только без параноии с получением таргета на группу, в этом методе
     * верим что лишних пейджей, которые не соответствуют таргету на группу, на объявлении нет
     * @param shard     шард
     * @param bannerIds список id объявлений
     * @return - список ModerateBannerPage
     */
    public Map<Long, List<ModerateBannerPage>> getModerateBannerPagesByBannerIds(int shard, Set<Long> bannerIds) {
        if (bannerIds.isEmpty()) {
            return emptyMap();
        }

        return dslContextProvider.ppc(shard)
                .select(moderateBannerPageMapper.getFieldsToRead())
                .from(MODERATE_BANNER_PAGES)
                .where(MODERATE_BANNER_PAGES.BID.in(bannerIds))
                .and(MODERATE_BANNER_PAGES.IS_REMOVED.eq(RepositoryUtils.FALSE))
                .fetchGroups(MODERATE_BANNER_PAGES.BID, moderateBannerPageMapper::fromDb);
    }

    /**
     * Сбросить статус модерации на ready
     *
     * @param bannerPageIds - pk moderate_banner_pages
     */
    public void resetModerateStatus(DSLContext dslContext, Collection<Long> bannerPageIds) {
        dslContext
                .update(MODERATE_BANNER_PAGES)
                .set(MODERATE_BANNER_PAGES.STATUS_MODERATE, ModerateBannerPagesStatusmoderate.Ready)
                .set(MODERATE_BANNER_PAGES.VERSION, MODERATE_BANNER_PAGES.VERSION.add(1))
                .set(MODERATE_BANNER_PAGES.COMMENT, (String) null)
                .set(MODERATE_BANNER_PAGES.CREATE_TIME, DSL.currentLocalDateTime())
                .where(MODERATE_BANNER_PAGES.MODERATE_BANNER_PAGE_ID.in(bannerPageIds))
                .and(MODERATE_BANNER_PAGES.IS_REMOVED.eq(RepositoryUtils.FALSE))
                .execute();
    }

    public void updateOperatorStatus(int shard, Long pageId, StatusModerateOperator status,
                                     List<Pair<Long, Long>> bidAndVersionList) {
        Condition whereCondition = DSL.falseCondition();
        for (Pair<Long, Long> pair : bidAndVersionList) {
            Long bid = pair.getLeft();
            Long version = pair.getRight();
            whereCondition = whereCondition.or(
                    MODERATE_BANNER_PAGES.PAGE_ID.eq(pageId)
                            .and(MODERATE_BANNER_PAGES.BID.eq(bid))
                            .and(MODERATE_BANNER_PAGES.VERSION.eq(version)));
        }

        dslContextProvider.ppc(shard)
                .update(ModerateBannerPages.MODERATE_BANNER_PAGES)
                .set(ModerateBannerPages.MODERATE_BANNER_PAGES.STATUS_MODERATE_OPERATOR,
                        StatusModerateOperator.toSource(status))
                .where(whereCondition)
                .and(MODERATE_BANNER_PAGES.IS_REMOVED.eq(RepositoryUtils.FALSE))
                .execute();
    }

    public static JooqMapperWithSupplier<ModerateBannerPage> createModerateBannerPageMapper() {
        return JooqMapperWithSupplierBuilder.builder(ModerateBannerPage::new)
                .map(property(ModerateBannerPage.ID, MODERATE_BANNER_PAGES.MODERATE_BANNER_PAGE_ID))
                .map(property(ModerateBannerPage.BANNER_ID, MODERATE_BANNER_PAGES.BID))
                .map(property(ModerateBannerPage.PAGE_ID, MODERATE_BANNER_PAGES.PAGE_ID))
                .map(property(ModerateBannerPage.VERSION, MODERATE_BANNER_PAGES.VERSION))
                .map(convertibleProperty(ModerateBannerPage.STATUS_MODERATE, MODERATE_BANNER_PAGES.STATUS_MODERATE,
                        StatusModerateBannerPage::fromSource,
                        StatusModerateBannerPage::toSource))
                .map(convertibleProperty(ModerateBannerPage.STATUS_MODERATE_OPERATOR,
                        MODERATE_BANNER_PAGES.STATUS_MODERATE_OPERATOR,
                        StatusModerateOperator::fromSource,
                        StatusModerateOperator::toSource))
                .map(booleanProperty(ModerateBannerPage.IS_REMOVED, MODERATE_BANNER_PAGES.IS_REMOVED))
                .map(property(ModerateBannerPage.CREATE_TIME, MODERATE_BANNER_PAGES.CREATE_TIME))
                .map(property(ModerateBannerPage.COMMENT, MODERATE_BANNER_PAGES.COMMENT))
                .map(property(ModerateBannerPage.TASK_URL, MODERATE_BANNER_PAGES.TASK_URL))
                .build();
    }
}
