package ru.yandex.direct.core.entity.banner.repository.filter;

import java.time.LocalDateTime;
import java.util.Collection;
import java.util.List;
import java.util.Set;

import one.util.streamex.EntryStream;
import org.apache.commons.lang3.StringUtils;

import ru.yandex.direct.core.entity.banner.model.Banner;
import ru.yandex.direct.core.entity.internalads.Constants;
import ru.yandex.direct.dbschema.ppc.enums.BannersBannerType;
import ru.yandex.direct.dbschema.ppc.enums.BannersStatusarch;
import ru.yandex.direct.dbschema.ppc.enums.BannersStatusmoderate;
import ru.yandex.direct.dbschema.ppc.enums.BannersStatusshow;
import ru.yandex.direct.dbschema.ppc.enums.CampaignsArchived;
import ru.yandex.direct.dbschema.ppc.tables.BannersPerformance;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.multitype.repository.filter.ConditionFilter;
import ru.yandex.direct.multitype.repository.filter.Filter;

import static ru.yandex.direct.common.util.RepositoryUtils.booleanToLong;
import static ru.yandex.direct.core.entity.banner.repository.BannerRepositoryConstants.BANNER_CLASS_TO_TYPE;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNERS;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNERS_INTERNAL;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNERS_PERFORMANCE;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNER_MODERATION_VERSIONS;
import static ru.yandex.direct.dbschema.ppc.Tables.CAMPAIGNS;
import static ru.yandex.direct.multitype.repository.filter.ConditionFilterFactory.lessThan;
import static ru.yandex.direct.multitype.repository.filter.ConditionFilterFactory.multipleConditionFilter;
import static ru.yandex.direct.multitype.repository.filter.ConditionFilterFactory.whereEqFilter;
import static ru.yandex.direct.multitype.repository.filter.ConditionFilterFactory.whereInFilter;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

public class BannerFilterFactory {

    private static final Set<BannersStatusmoderate> BANNERS_STATUSES_WAITING_FOR_MODERATION = Set.of(
            BannersStatusmoderate.Ready, BannersStatusmoderate.Sending, BannersStatusmoderate.Sent);

    private BannerFilterFactory() {
    }

    public static ConditionFilter bannerDomainFilter(Collection<String> domains) {
        List<String> reverseDomains = mapList(domains, StringUtils::reverse);
        return bannerReverseDomainFilter(reverseDomains);
    }

    public static ConditionFilter bannerReverseDomainFilter(Collection<String> reverseDomains) {
        return whereInFilter(BANNERS.REVERSE_DOMAIN, reverseDomains);
    }

    public static ConditionFilter bannerIdFilter(Collection<Long> bannerIds) {
        return whereInFilter(BANNERS.BID, bannerIds);
    }

    public static ConditionFilter bannerCampaignIdFilter(Collection<Long> campaignIds) {
        return whereInFilter(BANNERS.CID, campaignIds);
    }

    public static ConditionFilter bannerStatusArchFilter(BannersStatusarch bannersStatusarch) {
        return whereEqFilter(BANNERS.STATUS_ARCH, bannersStatusarch);
    }

    public static ConditionFilter bannerAdGroupIdFilter(Collection<Long> adGroupIds) {
        return whereInFilter(BANNERS.PID, adGroupIds);
    }

    public static ConditionFilter bannerClientIdFilter(ClientId clientId) {
        return whereEqFilter(CAMPAIGNS.CLIENT_ID,
                clientId.asLong(),
                CAMPAIGNS,
                BANNERS.CID.eq(CAMPAIGNS.CID));
    }

    public static ConditionFilter bannerBannerTypesFilter(Class<? extends Banner> tClass) {
        Set<BannersBannerType> bannerTypes = EntryStream.of(BANNER_CLASS_TO_TYPE)
                .filterKeys(tClass::isAssignableFrom)
                .values()
                .toSet();
        return bannerBannerTypesFilter(bannerTypes);
    }

    public static ConditionFilter bannerBannerTypesFilter(Collection<BannersBannerType> bannerTypes) {
        return whereInFilter(BANNERS.BANNER_TYPE, bannerTypes);
    }

    public static Filter clientIdAndCreativeIdsFilter(ClientId clientId, Collection<Long> creativeIds) {
        BannersPerformance bannersPerformanceFilterTable =
                BANNERS_PERFORMANCE.as("BANNERS_PERFORMANCE_FILTER_TABLE");

        return multipleConditionFilter(
                bannerClientIdFilter(clientId),

                whereInFilter(bannersPerformanceFilterTable.CREATIVE_ID,
                        creativeIds,
                        bannersPerformanceFilterTable,
                        BANNERS.BID.eq(bannersPerformanceFilterTable.BID))
        );
    }

    public static Filter clientIdAndAdGroupIdsFilter(ClientId clientId, Collection<Long> adGroupIds) {
        return multipleConditionFilter(
                bannerClientIdFilter(clientId),
                bannerAdGroupIdFilter(adGroupIds));
    }

    public static Filter clientIdAndAdGroupIdsAndBannerTypesFilter(ClientId clientId, Collection<Long> adGroupIds,
                                                                   Class<? extends Banner> tClass) {
        return multipleConditionFilter(
                bannerClientIdFilter(clientId),
                bannerAdGroupIdFilter(adGroupIds),
                bannerBannerTypesFilter(tClass));
    }

    public static Filter notArchivedInternalBannerWithStatusShowYesTemplateIdFilter(Collection<Long> templateIds) {
        return multipleConditionFilter(
                whereInFilter(BANNERS_INTERNAL.TEMPLATE_ID, templateIds),
                whereEqFilter(BANNERS.STATUS_SHOW, BannersStatusshow.Yes),
                whereEqFilter(BANNERS.STATUS_ARCH, BannersStatusarch.No),
                whereEqFilter(CAMPAIGNS.ARCHIVED, CampaignsArchived.No, CAMPAIGNS, BANNERS.CID.eq(CAMPAIGNS.CID))
        );
    }

    public static Filter internalBannersWaitingForModerationTill(LocalDateTime borderDateTime) {
        return multipleConditionFilter(
                whereInFilter(BANNERS_INTERNAL.TEMPLATE_ID, Constants.MODERATED_TEMPLATE_IDS),
                whereInFilter(BANNERS.STATUS_MODERATE, BANNERS_STATUSES_WAITING_FOR_MODERATION),
                whereEqFilter(BANNERS.STATUS_ARCH, BannersStatusarch.No),
                whereEqFilter(CAMPAIGNS.ARCHIVED, CampaignsArchived.No, CAMPAIGNS, BANNERS.CID.eq(CAMPAIGNS.CID)),
                lessThan(BANNER_MODERATION_VERSIONS.CREATE_TIME, borderDateTime,
                        BANNER_MODERATION_VERSIONS, BANNERS.BID.eq(BANNER_MODERATION_VERSIONS.BID)));
    }

    public static Filter notArchivedInternalBannersStoppedByUrlMonitoringFilter() {
        return multipleConditionFilter(
                whereEqFilter(BANNERS_INTERNAL.IS_STOPPED_BY_URL_MONITORING, booleanToLong(true)),
                whereEqFilter(BANNERS.STATUS_SHOW, BannersStatusshow.No),
                whereEqFilter(BANNERS.STATUS_ARCH, BannersStatusarch.No),
                whereEqFilter(CAMPAIGNS.ARCHIVED, CampaignsArchived.No, CAMPAIGNS, BANNERS.CID.eq(CAMPAIGNS.CID))
        );
    }
}
