package ru.yandex.direct.core.entity.banner.repository.old;

import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.jooq.DSLContext;
import org.jooq.Record;
import org.jooq.Result;
import org.jooq.impl.DSL;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.banner.model.old.OldBannerAdditionalHref;
import ru.yandex.direct.core.entity.banner.model.old.OldBannerAdditionalHrefInternal;
import ru.yandex.direct.core.entity.banner.model.old.OldBannerWithAdditionalHrefs;
import ru.yandex.direct.core.entity.banner.repository.old.container.InsertUpdateDeleteContainer;
import ru.yandex.direct.dbschema.ppc.tables.records.BannerAdditionalHrefsRecord;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.Model;

import static java.util.Collections.emptyList;
import static java.util.stream.Collectors.toList;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNER_ADDITIONAL_HREFS;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;
import static ru.yandex.direct.utils.FunctionalUtils.filterAndMapList;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;

/**
 * Новые изменения в этом репозитории скорее всего нужно продублировать в
 * BannerWithAdditionalHrefsRepositoryTypeSupport
 * В новых баннерах реализовано в BannerWithAdditionalHrefsRepositoryTypeSupport
 */
@Repository
@ParametersAreNonnullByDefault
@Deprecated
public class OldBannerAdditionalHrefsRepository {

    private static final JooqMapperWithSupplier<OldBannerAdditionalHref> ADDITIONAL_HREFS_MAPPER =
            JooqMapperWithSupplierBuilder.builder(OldBannerAdditionalHref::new)
                    .map(property(OldBannerAdditionalHref.HREF, BANNER_ADDITIONAL_HREFS.HREF))
                    .build();

    private static final JooqMapperWithSupplier<OldBannerAdditionalHrefInternal> ADDITIONAL_HREFS_INTERNAL_MAPPER =
            JooqMapperWithSupplierBuilder.builder(OldBannerAdditionalHrefInternal::new)
                    .map(property(OldBannerAdditionalHrefInternal.ID, BANNER_ADDITIONAL_HREFS.HREF_ID))
                    .map(property(OldBannerAdditionalHrefInternal.BANNER_ID, BANNER_ADDITIONAL_HREFS.BID))
                    .map(property(OldBannerAdditionalHrefInternal.INDEX, BANNER_ADDITIONAL_HREFS.INDEX))
                    .map(property(OldBannerAdditionalHrefInternal.HREF, BANNER_ADDITIONAL_HREFS.HREF))
                    .build();

    private final ShardHelper shardHelper;
    private final DslContextProvider dslContextProvider;

    @Autowired
    public OldBannerAdditionalHrefsRepository(ShardHelper shardHelper, DslContextProvider dslContextProvider) {
        this.shardHelper = shardHelper;
        this.dslContextProvider = dslContextProvider;
    }

    /**
     * Возвращает список дополнительных ссылок по id баннеров.
     * Если у баннера нет дополнитльных ссылок, то он не попадёт в результат.
     */
    public Map<Long, List<OldBannerAdditionalHref>> getAdditionalHrefs(int shard, Collection<Long> bannerIds) {
        return getAdditionalHrefs(dslContextProvider.ppc(shard), bannerIds);
    }

    /**
     * Возвращает список дополнительных ссылок по id баннеров.
     * Если у баннера нет дополнитльных ссылок, то он не попадёт в результат.
     */
    public Map<Long, List<OldBannerAdditionalHref>> getAdditionalHrefs(DSLContext context, Collection<Long> bannerIds) {
        return getAdditionalHrefsWithMapper(context, bannerIds, ADDITIONAL_HREFS_MAPPER);
    }

    private Map<Long, List<OldBannerAdditionalHrefInternal>> getAdditionalHrefsInternal(DSLContext context,
                                                                                        Collection<Long> bannerIds) {
        return getAdditionalHrefsWithMapper(context, bannerIds, ADDITIONAL_HREFS_INTERNAL_MAPPER);
    }

    private <M extends Model> Map<Long, List<M>> getAdditionalHrefsWithMapper(
            DSLContext context,
            Collection<Long> bannerIds,
            JooqMapperWithSupplier<M> mapper) {
        Map<Long, Result<Record>> bannerIdToAdditionalHrefs = context
                .select(mapper.getFieldsToRead())
                .select(BANNER_ADDITIONAL_HREFS.BID, BANNER_ADDITIONAL_HREFS.INDEX)
                .from(BANNER_ADDITIONAL_HREFS)
                .where(BANNER_ADDITIONAL_HREFS.BID.in(bannerIds))
                .fetchGroups(BANNER_ADDITIONAL_HREFS.BID);

        return EntryStream.of(bannerIdToAdditionalHrefs)
                .mapValues(result -> result
                        .sortAsc(BANNER_ADDITIONAL_HREFS.INDEX)
                        .map(mapper::fromDb))
                .toMap();
    }

    public <B extends OldBannerWithAdditionalHrefs> void addBannerAdditionalHrefs(
            DSLContext context, Collection<B> banners) {
        List<OldBannerAdditionalHrefInternal> allAdditionalHrefs = banners
                .stream()
                .map(this::extractAdditionalHrefsInternal)
                .flatMap(List::stream)
                .collect(toList());
        insertAdditionalHrefs(context, allAdditionalHrefs);
    }

    public <B extends OldBannerWithAdditionalHrefs> void updateBannerAdditionalHrefs(
            DSLContext context, Collection<AppliedChanges<B>> appliedChanges) {
        InsertUpdateDeleteContainer<OldBannerAdditionalHrefInternal> updateContainer =
                prepareAdditionalHrefsUpdateContainer(context, appliedChanges);

        insertAdditionalHrefs(context, updateContainer.getModelsToInsert());
        updateAdditionalHrefs(context, updateContainer.getModelsToUpdate());
        deleteAdditionalHrefs(context, updateContainer.getModelsToDelete());
    }

    private void insertAdditionalHrefs(DSLContext context, List<OldBannerAdditionalHrefInternal> additionalHrefs) {
        if (additionalHrefs.isEmpty()) {
            return;
        }

        List<Long> ids = shardHelper.generateBannerAdditionalHrefIds(additionalHrefs.size());
        StreamEx.of(additionalHrefs).zipWith(ids.stream())
                .forKeyValue(OldBannerAdditionalHrefInternal::setId);

        InsertHelper<BannerAdditionalHrefsRecord> insertHelper = new InsertHelper<>(context, BANNER_ADDITIONAL_HREFS)
                .addAll(ADDITIONAL_HREFS_INTERNAL_MAPPER, additionalHrefs);
        insertHelper.executeIfRecordsAdded();
    }

    private void updateAdditionalHrefs(DSLContext context, List<OldBannerAdditionalHrefInternal> additionalHrefs) {
        if (additionalHrefs.isEmpty()) {
            return;
        }

        Map<Long, String> idToHref =
                listToMap(additionalHrefs, OldBannerAdditionalHrefInternal::getId, OldBannerAdditionalHrefInternal::getHref);
        context.update(BANNER_ADDITIONAL_HREFS)
                .set(BANNER_ADDITIONAL_HREFS.HREF, DSL.choose(BANNER_ADDITIONAL_HREFS.HREF_ID).mapValues(idToHref))
                .where(BANNER_ADDITIONAL_HREFS.HREF_ID.in(idToHref.keySet()))
                .execute();
    }

    private void deleteAdditionalHrefs(DSLContext context, List<OldBannerAdditionalHrefInternal> additionalHrefs) {
        if (additionalHrefs.isEmpty()) {
            return;
        }

        Set<Long> idsToDelete = listToSet(additionalHrefs, OldBannerAdditionalHrefInternal::getId);
        context.delete(BANNER_ADDITIONAL_HREFS)
                .where(BANNER_ADDITIONAL_HREFS.HREF_ID.in(idsToDelete))
                .execute();
    }

    private List<OldBannerAdditionalHrefInternal> extractAdditionalHrefsInternal(OldBannerWithAdditionalHrefs banner) {
        return EntryStream.of(banner.getAdditionalHrefs())
                .mapKeyValue((index, additionalHref) -> new OldBannerAdditionalHrefInternal()
                        .withBannerId(banner.getId())
                        .withIndex((long) index)
                        .withHref(additionalHref.getHref()))
                .toList();
    }

    private <B extends OldBannerWithAdditionalHrefs>
    InsertUpdateDeleteContainer<OldBannerAdditionalHrefInternal> prepareAdditionalHrefsUpdateContainer(
            DSLContext context, Collection<AppliedChanges<B>> appliedChangesCollection) {

        List<Long> bannerWithChangedAdditionalHrefsIds = filterAndMapList(appliedChangesCollection,
                appliedChanges -> appliedChanges.changed(OldBannerWithAdditionalHrefs.ADDITIONAL_HREFS),
                appliedChanges -> appliedChanges.getModel().getId());

        Map<Long, List<OldBannerAdditionalHrefInternal>> currentChangingAdditionalHrefs =
                getAdditionalHrefsInternal(context, bannerWithChangedAdditionalHrefsIds);
        InsertUpdateDeleteContainer<OldBannerAdditionalHrefInternal> additionalHrefsUpdateContainer =
                new InsertUpdateDeleteContainer<>();

        for (AppliedChanges<B> appliedChanges : appliedChangesCollection) {
            if (!appliedChanges.changed(B.ADDITIONAL_HREFS)) {
                continue;
            }
            Long bannerId = appliedChanges.getModel().getId();

            Iterator<OldBannerAdditionalHrefInternal> oldHrefs =
                    currentChangingAdditionalHrefs.getOrDefault(bannerId, emptyList()).iterator();
            Iterator<OldBannerAdditionalHrefInternal> newHrefs =
                    extractAdditionalHrefsInternal(appliedChanges.getModel()).iterator();
            while (oldHrefs.hasNext() || newHrefs.hasNext()) {
                if (!oldHrefs.hasNext()) {
                    additionalHrefsUpdateContainer.markForInsertion(newHrefs.next());
                } else if (!newHrefs.hasNext()) {
                    additionalHrefsUpdateContainer.markForDeletion(oldHrefs.next());
                } else {
                    OldBannerAdditionalHrefInternal oldHref = oldHrefs.next();
                    OldBannerAdditionalHrefInternal newHref = newHrefs.next();
                    if (isHrefChanged(oldHref, newHref)) {
                        additionalHrefsUpdateContainer.markForUpdate(oldHref.copy()
                                .withHref(newHref.getHref())
                                .withIndex(newHref.getIndex()));
                    }
                }
            }
        }

        return additionalHrefsUpdateContainer;
    }

    private boolean isHrefChanged(OldBannerAdditionalHrefInternal oldHref, OldBannerAdditionalHrefInternal newHref) {
        return !oldHref.getHref().equals(newHref.getHref())
                || !oldHref.getIndex().equals(newHref.getIndex());
    }
}
