package ru.yandex.direct.core.entity.banner.repository.old;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.function.Function;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.jooq.Condition;
import org.jooq.DSLContext;
import org.jooq.impl.DSL;
import org.jooq.util.mysql.MySQLDSL;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.banner.model.AdditionType;
import ru.yandex.direct.core.entity.banner.model.BannerAddition;
import ru.yandex.direct.core.entity.banner.model.old.OldBannerWithCallouts;
import ru.yandex.direct.dbschema.ppc.enums.BannersAdditionsAdditionsType;
import ru.yandex.direct.dbschema.ppc.tables.records.BannersAdditionsRecord;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;

import static java.util.Collections.emptyList;
import static java.util.function.Function.identity;
import static java.util.stream.Collectors.collectingAndThen;
import static java.util.stream.Collectors.toList;
import static org.apache.commons.collections4.CollectionUtils.isEmpty;
import static ru.yandex.direct.dbschema.ppc.Tables.ADDITIONS_ITEM_DISCLAIMERS;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNERS;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNERS_ADDITIONS;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.convertibleProperty;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Repository
@Deprecated
public class OldBannerAdditionsRepository {

    private final DslContextProvider dslContextProvider;

    private final JooqMapperWithSupplier<BannerAddition> bannerAdditionsMapper;

    public OldBannerAdditionsRepository(DslContextProvider dslContextProvider) {
        this.dslContextProvider = dslContextProvider;

        bannerAdditionsMapper = createBannerAdditionMapper();
    }

    /**
     * Получение списка id уточнений баннеров. Уточнения сортированы по полю sequenceNum
     *
     * @param shard     шард для запроса
     * @param bannerIds список id баннеров
     * @return мапа bannerId -> список additionId
     */
    public Map<Long, List<Long>> getCalloutIdsByBannerIds(int shard, Collection<Long> bannerIds) {
        Function<List<BannerAddition>, List<BannerAddition>> sortBySequenceNum =
                bannerAdditions -> bannerAdditions.stream()
                        .sorted(Comparator.comparing(BannerAddition::getSequenceNum))
                        .collect(toList());
        List<BannerAddition> calloutsResult = dslContextProvider.ppc(shard)
                .select(BANNERS_ADDITIONS.BID, BANNERS_ADDITIONS.ADDITIONS_ITEM_ID, BANNERS_ADDITIONS.SEQUENCE_NUM)
                .from(BANNERS_ADDITIONS)
                .where(BANNERS_ADDITIONS.BID.in(bannerIds))
                .and(BANNERS_ADDITIONS.ADDITIONS_TYPE.eq(BannersAdditionsAdditionsType.callout))
                .fetch()
                .map(bannerAdditionsMapper::fromDb);

        return StreamEx.of(calloutsResult)
                .mapToEntry(BannerAddition::getBannerId, identity())
                .collapseKeys(collectingAndThen(toList(), sortBySequenceNum))
                .mapValues(bannersAdditions -> mapList(bannersAdditions, BannerAddition::getId))
                .toMap();
    }

    /**
     * Добавление записи в таблицу banners_additions. Если запись уже существует, то обновляем sequebce_num
     *
     * @param context         контекст
     * @param bannerAdditions список записей
     */
    public void addOrUpdateBannerAdditions(DSLContext context, Collection<BannerAddition> bannerAdditions) {
        InsertHelper<BannersAdditionsRecord> insertHelper = new InsertHelper<>(context, BANNERS_ADDITIONS)
                .addAll(bannerAdditionsMapper, bannerAdditions);

        if (insertHelper.hasAddedRecords()) {
            insertHelper.onDuplicateKeyUpdate()
                    .set(BANNERS_ADDITIONS.SEQUENCE_NUM, MySQLDSL.values(BANNERS_ADDITIONS.SEQUENCE_NUM));
        }

        insertHelper.executeIfRecordsAdded();
    }

    /**
     * отвязка уточнений от баннеров
     *
     * @param bannerIdToCalloutIds мапа bannerId -> список уточнений
     * @param context              контекст
     */
    public void deleteBannersAdditions(Map<Long, List<Long>> bannerIdToCalloutIds, DSLContext context) {
        if (bannerIdToCalloutIds.isEmpty()) {
            return;
        }
        Condition condition = EntryStream.of(bannerIdToCalloutIds)
                .filter(entry -> !entry.getValue().isEmpty())
                .map(entry -> BANNERS_ADDITIONS.BID.eq(entry.getKey())
                        .and(BANNERS_ADDITIONS.ADDITIONS_ITEM_ID.in(entry.getValue())))
                .reduce(Condition::or)
                .orElse(DSL.falseCondition());
        context.deleteFrom(BANNERS_ADDITIONS)
                .where(condition)
                .execute();
    }

    /**
     * Получить дисклеймеры к объявлениям
     *
     * @param shard     шард для запроса
     * @param bannerIds коллекция id баннеров
     */
    public Map<Long, String> getAdditionDisclaimerByBannerIds(int shard, Collection<Long> bannerIds) {
        return dslContextProvider.ppc(shard)
                .select(BANNERS.BID, ADDITIONS_ITEM_DISCLAIMERS.DISCLAIMER_TEXT)
                .from(BANNERS_ADDITIONS)
                .join(ADDITIONS_ITEM_DISCLAIMERS).using(ADDITIONS_ITEM_DISCLAIMERS.ADDITIONS_ITEM_ID)
                .join(BANNERS).using(BANNERS.BID)
                .where(BANNERS_ADDITIONS.BID.in(bannerIds))
                .and(BANNERS_ADDITIONS.ADDITIONS_TYPE.eq(BannersAdditionsAdditionsType.disclaimer))
                .fetchMap(BANNERS.BID, ADDITIONS_ITEM_DISCLAIMERS.DISCLAIMER_TEXT);
    }

    /**
     * Создает на основе {@link OldBannerWithCallouts} список {@link BannerAddition}.
     *
     * @param bannerWithCallouts список баннеров с уточнениями
     * @return список {@link BannerAddition}
     */
    public static List<BannerAddition> extractCalloutAdditions(OldBannerWithCallouts bannerWithCallouts) {
        return extractCalloutAdditions(bannerWithCallouts.getId(), bannerWithCallouts.getCalloutIds());
    }

    /**
     * создает на основе id баннера и id уточнений список {@link BannerAddition}.
     * Выставляет поле sequenceNum в порядке следования уточнений в переданном списке
     *
     * @param bannerId   id баннера
     * @param calloutIds список id уточнений
     * @return список {@link BannerAddition}
     */
    public static List<BannerAddition> extractCalloutAdditions(Long bannerId, List<Long> calloutIds) {
        if (isEmpty(calloutIds)) {
            return emptyList();
        }
        //в уточнениях счет начинается с единицы
        long orderNum = 1;
        List<BannerAddition> bannerAdditions = new ArrayList<>();
        for (Long calloutId : calloutIds) {
            bannerAdditions.add(new BannerAddition()
                    .withId(calloutId)
                    .withBannerId(bannerId)
                    .withAdditionType(AdditionType.CALLOUT)
                    .withSequenceNum(orderNum++));
        }
        return bannerAdditions;
    }

    /**
     * Конвертация бизнес объектов в записи в таблицах и обратно
     */
    private static JooqMapperWithSupplier<BannerAddition> createBannerAdditionMapper() {
        return JooqMapperWithSupplierBuilder.builder(BannerAddition::new)
                .map(property(BannerAddition.ID, BANNERS_ADDITIONS.ADDITIONS_ITEM_ID))
                .map(property(BannerAddition.BANNER_ID, BANNERS_ADDITIONS.BID))
                .map(property(BannerAddition.SEQUENCE_NUM, BANNERS_ADDITIONS.SEQUENCE_NUM))
                .map(convertibleProperty(BannerAddition.ADDITION_TYPE, BANNERS_ADDITIONS.ADDITIONS_TYPE,
                        AdditionType::fromSource, AdditionType::toSource))
                .build();
    }
}
