package ru.yandex.direct.core.entity.banner.repository.old;

import java.time.LocalDateTime;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import javax.annotation.ParametersAreNonnullByDefault;

import org.apache.commons.lang3.tuple.Pair;
import org.jooq.DSLContext;
import org.jooq.util.mysql.MySQLDSL;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.banner.model.ImageSize;
import ru.yandex.direct.core.entity.banner.model.ImageType;
import ru.yandex.direct.core.entity.banner.model.old.OldBannerImage;
import ru.yandex.direct.core.entity.banner.model.old.OldBannerWithBannerImage;
import ru.yandex.direct.core.entity.banner.model.old.OldStatusBannerImageModerate;
import ru.yandex.direct.dbschema.ppc.Tables;
import ru.yandex.direct.dbschema.ppc.enums.BannerImagesStatusshow;
import ru.yandex.direct.dbschema.ppc.tables.records.BannerImagesRecord;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;

import static java.util.Collections.emptyMap;
import static ru.yandex.direct.core.entity.banner.repository.old.mapper.BannerImageMapperProvider.getBannerImageMapper;
import static ru.yandex.direct.core.entity.banner.repository.old.mapper.BannerImageMappings.imageStatusShowToDb;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNER_IMAGES_UPLOADS;
import static ru.yandex.direct.dbschema.ppc.tables.BannerImages.BANNER_IMAGES;
import static ru.yandex.direct.dbschema.ppc.tables.BannerImagesFormats.BANNER_IMAGES_FORMATS;
import static ru.yandex.direct.dbschema.ppc.tables.BannerImagesPool.BANNER_IMAGES_POOL;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Класс для разгрузки {@link OldBannerRepository} –
 * делегат, инкапсулирующий обработку {@code Banner.BANNER_IMAGE}.
 */
@Repository
@ParametersAreNonnullByDefault
@Deprecated
public class OldBannerImageRepository {

    private final ShardHelper shardHelper;
    private final DslContextProvider dslContextProvider;

    private final JooqMapperWithSupplier<OldBannerImage> bannerImageMapper;

    private static final int CHUNK_SIZE = 1000;


    @Autowired
    OldBannerImageRepository(
            ShardHelper shardHelper,
            DslContextProvider dslContextProvider) {
        this.shardHelper = shardHelper;
        this.dslContextProvider = dslContextProvider;

        this.bannerImageMapper = getBannerImageMapper();
    }

    /**
     * Сохраняет непустые banner.bannerImage в таблицу banners_images
     * (включая генерацию imageId для изображения баннера, если banner.bannerImage непустой).
     * <p>
     * Баннерам уже должны быть присвоены id.
     * <p>
     * Если id уже был задан, то запись обновляется, устанавливается статус показа 'Yes'.
     */
    public <B extends OldBannerWithBannerImage> void addOrUpdateBannerImage(Collection<B> banners, DSLContext context) {
        List<B> bannersWithImage = filterList(banners, b -> b.getBannerImage() != null);
        generateAndSetNewIds(bannersWithImage);

        InsertHelper<BannerImagesRecord> insertHelper = new InsertHelper<>(context, BANNER_IMAGES)
                .addAll(bannerImageMapper, mapList(bannersWithImage, B::getBannerImage));

        if (insertHelper.hasAddedRecords()) {
            insertHelper.onDuplicateKeyUpdate()
                    .set(BANNER_IMAGES.STATUS_SHOW, BannerImagesStatusshow.Yes)
                    .set(BANNER_IMAGES.STATUS_MODERATE, MySQLDSL.values(BANNER_IMAGES.STATUS_MODERATE))
                    .set(BANNER_IMAGES.IMAGE_HASH, MySQLDSL.values(BANNER_IMAGES.IMAGE_HASH));
        }

        insertHelper.executeIfRecordsAdded();
    }

    private <B extends OldBannerWithBannerImage> void generateAndSetNewIds(List<B> bannersWithImage) {
        bannersWithImage.forEach(banner -> banner.getBannerImage().setBannerId(banner.getId()));

        List<B> bannersWithImageWithNoId = filterList(bannersWithImage, b -> b.getBannerImage().getId() == null);
        Iterator<Long> bannerImageIds = generateBannerImageIds(bannersWithImageWithNoId).iterator();

        bannersWithImageWithNoId.forEach(banner -> banner.getBannerImage().setId(bannerImageIds.next()));
    }

    /**
     * Сгенерировать новые id для bannerImage.
     * Используется общий генератор с баннером по аналогии с perl - реализацией.
     *
     * @param bannersWithImage баннеры с картинкой
     */
    private <B extends OldBannerWithBannerImage> List<Long> generateBannerImageIds(List<B> bannersWithImage) {
        return shardHelper.generateBannerIdsByBids(mapList(bannersWithImage, B::getId));
    }

    /**
     * "Удалить" изображения объявлений: проставление им {@code StatusShow = No}.
     *
     * @param shard   шард
     * @param banners баннеры, изображения которых нужно удалить
     * @param <B>     тип баннеров
     */
    public <B extends OldBannerWithBannerImage> void deleteBannerImages(int shard, List<B> banners) {
        setBannerImagesStatusShow(shard, mapList(banners, B::getId), false);
    }

    /**
     * Получение сохраненных в БД хешей изображений и размеров принадлежащих указанному клиенту из переданной коллекции
     *
     * @param shard    шард
     * @param clientId id клиента
     * @param hashes   коллекция проверяемых хешей
     * @return коллекция принадлежащих указанному клиенту хешей изображений и размеров из переданной коллекции
     */
    Map<String, Pair<ImageType, ImageSize>> getExistingImageSizes(int shard, ClientId clientId,
                                                                  Collection<String> hashes) {
        if (hashes.isEmpty()) {
            return emptyMap();
        }
        return dslContextProvider.ppc(shard)
                .select(BANNER_IMAGES_POOL.IMAGE_HASH, BANNER_IMAGES_FORMATS.IMAGE_TYPE,
                        BANNER_IMAGES_FORMATS.WIDTH, BANNER_IMAGES_FORMATS.HEIGHT)
                .from(BANNER_IMAGES_POOL)
                .join(BANNER_IMAGES_FORMATS).on(BANNER_IMAGES_FORMATS.IMAGE_HASH.eq(BANNER_IMAGES_POOL.IMAGE_HASH))
                .where(BANNER_IMAGES_POOL.IMAGE_HASH.in(hashes))
                .and(BANNER_IMAGES_POOL.CLIENT_ID.eq(clientId.asLong()))
                .fetchMap(BANNER_IMAGES_POOL.IMAGE_HASH, r -> Pair.of(
                        ImageType.fromSource(r.get(BANNER_IMAGES_FORMATS.IMAGE_TYPE)),
                        new ImageSize()
                                .withWidth(r.get(BANNER_IMAGES_FORMATS.WIDTH).intValue())
                                .withHeight(r.get(BANNER_IMAGES_FORMATS.HEIGHT).intValue())));
    }

    /**
     * установка статуса показа картинки
     *
     * @param shard     шард
     * @param bannerIds список id баннеров
     */
    void setBannerImagesStatusShow(int shard, Collection<Long> bannerIds, boolean statusShow) {
        if (bannerIds.isEmpty()) {
            return;
        }
        dslContextProvider.ppc(shard)
                .update(Tables.BANNER_IMAGES)
                .set(Tables.BANNER_IMAGES.STATUS_SHOW, imageStatusShowToDb(statusShow))
                .where(Tables.BANNER_IMAGES.BID.in(bannerIds))
                .execute();
    }

    /**
     * Установка статуса модерации картинки
     *
     * @param bannerIds список id баннеров
     */
    public void setBannerImagesStatusModerate(int shard, Collection<Long> bannerIds,
                                              OldStatusBannerImageModerate statusModerate) {
        setBannerImagesStatusModerate(dslContextProvider.ppc(shard).dsl(), bannerIds, statusModerate);
    }

    /**
     * Установка статуса модерации картинки
     *
     * @param bannerIds список id баннеров
     */
    public void setBannerImagesStatusModerate(DSLContext context, Collection<Long> bannerIds,
                                              OldStatusBannerImageModerate statusModerate) {
        if (bannerIds.isEmpty()) {
            return;
        }
        context
                .update(Tables.BANNER_IMAGES)
                .set(Tables.BANNER_IMAGES.STATUS_MODERATE, OldStatusBannerImageModerate.toSource(statusModerate))
                .where(Tables.BANNER_IMAGES.BID.in(bannerIds))
                .execute();
    }

    /**
     * Возвращает список image_hash элементов у которых нет связи с таблицами BANNER_IMAGES и BANNER_IMAGES_POOL (по
     * IMAGE_HASH) и c лимитом {@param limit}
     *
     * @param shard шард
     * @param limit количество строк
     */
    public List<String> getBannerImagesFormatsHashesForDelete(int shard, int limit) {
        return dslContextProvider.ppc(shard)
                .select(BANNER_IMAGES_FORMATS.IMAGE_HASH)
                .from(BANNER_IMAGES_FORMATS)
                .leftJoin(BANNER_IMAGES).on(BANNER_IMAGES_FORMATS.IMAGE_HASH.eq(BANNER_IMAGES.IMAGE_HASH))
                .leftJoin(BANNER_IMAGES_POOL).on(BANNER_IMAGES_FORMATS.IMAGE_HASH.eq(BANNER_IMAGES_POOL.IMAGE_HASH))
                .where(BANNER_IMAGES.IMAGE_HASH.isNull())
                .and(Tables.BANNER_IMAGES_POOL.IMAGE_HASH.isNull())
                .limit(limit)
                .fetch(BANNER_IMAGES_FORMATS.IMAGE_HASH);
    }

    /**
     * Удаляет строки из banner_images_formats по IMAGE_HASH {@param hashes} и у которых нет зависимости с таблицами
     * BANNER_IMAGES и BANNER_IMAGES_POOL (по IMAGE_HASH)
     *
     * @param shard  шард
     * @param hashes коллекция image_hash
     * @return количество удаленных строк
     */
    public int deleteBannerImagesFormatsByHashesWithoutRelations(int shard, Collection<String> hashes) {
        return dslContextProvider.ppc(shard)
                .deleteFrom(BANNER_IMAGES_FORMATS)
                .where(BANNER_IMAGES_FORMATS.IMAGE_HASH.in(hashes))
                .andNotExists(dslContextProvider.ppc(shard)
                        .selectOne()
                        .from(BANNER_IMAGES)
                        .where(BANNER_IMAGES.IMAGE_HASH.eq(BANNER_IMAGES_FORMATS.IMAGE_HASH)))
                .andNotExists(dslContextProvider.ppc(shard)
                        .selectOne()
                        .from(Tables.BANNER_IMAGES_POOL)
                        .where(Tables.BANNER_IMAGES_POOL.IMAGE_HASH.eq(BANNER_IMAGES_FORMATS.IMAGE_HASH)))
                .execute();
    }

    /**
     * Возвращает список id у элементов с DATE_ADDED <= {@param lessThanDateAdded} и лимитом {@param limit}
     *
     * @param shard                шард
     * @param lessOrEqualDateAdded граничное время
     * @param limit                количество строк
     */
    public List<Long> getIdsByDateAdded(int shard, LocalDateTime lessOrEqualDateAdded, int limit) {
        return dslContextProvider.ppc(shard)
                .select(BANNER_IMAGES_UPLOADS.ID)
                .from(BANNER_IMAGES_UPLOADS)
                .where(BANNER_IMAGES_UPLOADS.DATE_ADDED.lessOrEqual(lessOrEqualDateAdded))
                .limit(limit)
                .fetch(BANNER_IMAGES_UPLOADS.ID);
    }

    /**
     * Очищает таблицу banner_images_uploads по Id {@param ids}
     *
     * @param shard шард
     * @param ids   коллекция id
     * @return количество удаленных строк
     */
    public int deleteByIds(int shard, Collection<Long> ids) {
        return dslContextProvider.ppc(shard)
                .deleteFrom(BANNER_IMAGES_UPLOADS)
                .where(BANNER_IMAGES_UPLOADS.ID.in(ids))
                .execute();
    }

}
