package ru.yandex.direct.core.entity.banner.repository.old;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.jooq.Condition;
import org.jooq.DSLContext;
import org.jooq.impl.DSL;
import org.jooq.util.mysql.MySQLDSL;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.banner.model.old.OldBannerPixel;
import ru.yandex.direct.core.entity.banner.model.old.OldBannerTns;
import ru.yandex.direct.core.entity.banner.model.old.OldBannerWithPixels;
import ru.yandex.direct.core.entity.banner.model.old.OldBannerWithTns;
import ru.yandex.direct.core.entity.banner.model.pixels.ClientPixelProvider;
import ru.yandex.direct.core.entity.banner.model.pixels.CriterionType;
import ru.yandex.direct.core.entity.banner.model.pixels.PixelCampaignType;
import ru.yandex.direct.core.entity.banner.model.pixels.Provider;
import ru.yandex.direct.dbschema.ppc.tables.records.BannerPixelsRecord;
import ru.yandex.direct.dbschema.ppc.tables.records.BannersTnsRecord;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;

import static java.util.Collections.emptyList;
import static java.util.function.Function.identity;
import static java.util.stream.Collectors.toList;
import static org.apache.commons.collections4.CollectionUtils.isEmpty;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNERS_TNS;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNER_PIXELS;
import static ru.yandex.direct.dbschema.ppc.Tables.CLIENT_PIXEL_PROVIDERS;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.convertibleProperty;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Repository
@Deprecated
public class OldBannerPixelsRepository {

    private final DslContextProvider dslContextProvider;
    private final ShardHelper shardHelper;

    private final JooqMapperWithSupplier<OldBannerPixel> bannerPixelsMapper;
    public final JooqMapperWithSupplier<ClientPixelProvider> clientPixelProvidersMapper;
    private final JooqMapperWithSupplier<OldBannerTns> bannerTnsMapper;

    public OldBannerPixelsRepository(DslContextProvider dslContextProvider,
                                     ShardHelper shardHelper) {
        this.dslContextProvider = dslContextProvider;
        this.shardHelper = shardHelper;

        bannerPixelsMapper = createBannerPixelsMapper();
        clientPixelProvidersMapper = createClientPixelProvidersMapper();
        bannerTnsMapper = createBannerTnsMapper();
    }

    /**
     * Получение списка пикселей баннеров
     *
     * @param shard     шард для запроса
     * @param bannerIds список id баннеров
     * @return мапа bannerId -> список pixel_url
     */
    public Map<Long, List<String>> getPixelsByBannerIds(int shard, Collection<Long> bannerIds) {
        List<OldBannerPixel> pixelsResult = dslContextProvider.ppc(shard)
                .select(BANNER_PIXELS.BID, BANNER_PIXELS.PIXEL_URL)
                .from(BANNER_PIXELS)
                .where(BANNER_PIXELS.BID.in(bannerIds))
                .fetch()
                .map(bannerPixelsMapper::fromDb);

        return StreamEx.of(pixelsResult)
                .mapToEntry(OldBannerPixel::getBannerId, identity())
                .collapseKeys(toList())
                .mapValues(bannerPixels -> mapList(bannerPixels, OldBannerPixel::getPixelUrl))
                .toMap();
    }

    /**
     * Получение из таблицы CLIENT_PIXEL_PROVIDERS по идентификатору клиента выданных ему разрешений
     * на добавляемые пиксели
     *
     * @param shard    Номер шарда
     * @param clientId id клиента
     * @return информация о разрешенных клиенту системах аудита
     */
    public List<ClientPixelProvider> getClientPixelProviders(int shard, Long clientId) {
        return dslContextProvider.ppc(shard)
                .select(CLIENT_PIXEL_PROVIDERS.CLIENT_ID, CLIENT_PIXEL_PROVIDERS.PROVIDER,
                        CLIENT_PIXEL_PROVIDERS.CAMPAIGN_TYPE, CLIENT_PIXEL_PROVIDERS.CRITERION_TYPE)
                .from(CLIENT_PIXEL_PROVIDERS)
                .where(CLIENT_PIXEL_PROVIDERS.CLIENT_ID.eq(clientId))
                .fetch()
                .map(clientPixelProvidersMapper::fromDb);
    }

    /**
     * Добавление записи в таблицу banner_pixels
     *
     * @param bannerPixels список записей
     * @param context      контекст
     */
    public void addBannerPixels(Collection<OldBannerPixel> bannerPixels, DSLContext context) {
        generatePixelIds(bannerPixels);
        InsertHelper<BannerPixelsRecord> insertHelper =
                new InsertHelper<>(context, BANNER_PIXELS).addAll(bannerPixelsMapper, bannerPixels);
        insertHelper.executeIfRecordsAdded();
    }

    /**
     * отвязка пикселей от баннеров
     *
     * @param bannerIdToPixelIds мапа bannerId -> список пикселей
     * @param context            контекст
     */
    public void deleteBannerPixels(Map<Long, List<String>> bannerIdToPixelIds, DSLContext context) {
        if (bannerIdToPixelIds.isEmpty()) {
            return;
        }
        Condition condition = EntryStream.of(bannerIdToPixelIds)
                .filter(entry -> !entry.getValue().isEmpty())
                .map(entry -> BANNER_PIXELS.BID.eq(entry.getKey())
                        .and(BANNER_PIXELS.PIXEL_URL.in(entry.getValue())))
                .reduce(Condition::or)
                .orElse(DSL.falseCondition());
        context.deleteFrom(BANNER_PIXELS)
                .where(condition)
                .execute();
    }

    /**
     * Создает на основе {@link OldBannerWithPixels} список {@link OldBannerPixel}.
     *
     * @param banner медийный баннер
     * @return список {@link OldBannerPixel}
     */
    public static List<OldBannerPixel> extractBannerPixels(OldBannerWithPixels banner) {
        return extractBannerPixels(banner.getId(), banner.getPixels());
    }

    /**
     * создает на основе id баннера и урлов пикселей список {@link OldBannerPixel}.
     *
     * @param bannerId  id баннера
     * @param pixelUrls список урлов пикселей
     * @return список {@link OldBannerPixel}
     */
    public static List<OldBannerPixel> extractBannerPixels(Long bannerId, List<String> pixelUrls) {
        if (isEmpty(pixelUrls)) {
            return emptyList();
        }
        List<OldBannerPixel> bannerPixels = new ArrayList<>();
        for (String pixelUrl : pixelUrls) {
            bannerPixels.add(new OldBannerPixel()
                    .withPixelUrl(pixelUrl)
                    .withBannerId(bannerId));
        }
        return bannerPixels;
    }

    /**
     * Генерирует новые pixel_id и выставляет эти id каждому элементу в переданном списке
     *
     * @param bannerPixels список связок пикселей и баннеров, которым нужно сгенерировать новые pixel_id
     */
    private void generatePixelIds(Collection<OldBannerPixel> bannerPixels) {
        List<Long> ids = shardHelper.generatePixelIds(bannerPixels.size());
        StreamEx.of(bannerPixels).zipWith(ids.stream())
                .forKeyValue(OldBannerPixel::setPixelId);
    }


    /**
     * Создает на основе {@link OldBannerWithTns} объект {@link OldBannerTns}.
     *
     * @param banner медийный баннер
     * @return {@link OldBannerTns}
     */
    public static OldBannerTns extractBannerTns(OldBannerWithTns banner) {
        return new OldBannerTns()
                .withBannerId(banner.getId())
                .withTnsId(banner.getTnsId());
    }

    /**
     * Добавление записи в таблицу banners_tns
     *
     * @param bannerTns список записей
     * @param context   контекст
     */
    public void saveBannerTns(Collection<OldBannerTns> bannerTns, DSLContext context) {
        InsertHelper<BannersTnsRecord> insertHelper =
                new InsertHelper<>(context, BANNERS_TNS).addAll(bannerTnsMapper, bannerTns);
        if (insertHelper.hasAddedRecords()) {
            insertHelper.onDuplicateKeyUpdate()
                    .set(BANNERS_TNS.TNS_ID, MySQLDSL.values(BANNERS_TNS.TNS_ID));
        }
        insertHelper.executeIfRecordsAdded();
    }


    /**
     * Удаление записи из banners_tns
     *
     * @param bannerTns записи на удаление
     * @param context   контекст
     */
    public void deleteBannerTns(Collection<OldBannerTns> bannerTns, DSLContext context) {
        var bannerIds = mapList(bannerTns, OldBannerTns::getBannerId);
        deleteBannerTnsByBannerIds(context, bannerIds);
    }

    /**
     * Удаление tns по id баннеров
     *
     * @param context   контекст
     * @param bannerIds список id баннеров
     */
    // перенесенно в BannerTnsRepository
    public void deleteBannerTnsByBannerIds(DSLContext context, Collection<Long> bannerIds) {
        if (bannerIds.isEmpty()) {
            return;
        }
        context.deleteFrom(BANNERS_TNS)
                .where(BANNERS_TNS.BID.in(bannerIds))
                .execute();
    }

    /**
     * Получение списка tns баннеров
     *
     * @param shard     шард для запроса
     * @param bannerIds список id баннеров
     * @return мапа bannerId -> список tns_id
     */
    public Map<Long, String> getTnsByIds(int shard, Collection<Long> bannerIds) {
        return dslContextProvider.ppc(shard)
                .select(BANNERS_TNS.BID, BANNERS_TNS.TNS_ID)
                .from(BANNERS_TNS)
                .where(BANNERS_TNS.BID.in(bannerIds))
                .fetchMap(BANNERS_TNS.BID, BANNERS_TNS.TNS_ID);
    }


    /**
     * Конвертация бизнес объектов в записи в таблицах и обратно
     */

    private static JooqMapperWithSupplier<OldBannerTns> createBannerTnsMapper() {
        return JooqMapperWithSupplierBuilder.builder(OldBannerTns::new)
                .map(property(OldBannerTns.BANNER_ID, BANNERS_TNS.BID))
                .map(property(OldBannerTns.TNS_ID, BANNERS_TNS.TNS_ID))
                .build();
    }

    private static JooqMapperWithSupplier<OldBannerPixel> createBannerPixelsMapper() {
        return JooqMapperWithSupplierBuilder.builder(OldBannerPixel::new)
                .map(property(OldBannerPixel.BANNER_ID, BANNER_PIXELS.BID))
                .map(property(OldBannerPixel.PIXEL_ID, BANNER_PIXELS.PIXEL_ID))
                .map(property(OldBannerPixel.PIXEL_URL, BANNER_PIXELS.PIXEL_URL))
                .build();
    }

    private static JooqMapperWithSupplier<ClientPixelProvider> createClientPixelProvidersMapper() {
        return JooqMapperWithSupplierBuilder.builder(ClientPixelProvider::new)
                .map(convertibleProperty(ClientPixelProvider.PROVIDER, CLIENT_PIXEL_PROVIDERS.PROVIDER,
                        Provider::fromSource, Provider::toSource))
                .map(convertibleProperty(ClientPixelProvider.PIXEL_CAMPAIGN_TYPE, CLIENT_PIXEL_PROVIDERS.CAMPAIGN_TYPE,
                        PixelCampaignType::fromSource, PixelCampaignType::toSource))
                .map(convertibleProperty(ClientPixelProvider.CRITERION_TYPE,
                        CLIENT_PIXEL_PROVIDERS.CRITERION_TYPE,
                        CriterionType::fromSource, CriterionType::toSource))
                .build();
    }
}
