package ru.yandex.direct.core.entity.banner.repository.old;

import java.util.Collection;
import java.util.List;
import java.util.Map;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.Multimap;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.jooq.Configuration;
import org.jooq.DSLContext;
import org.jooq.Field;
import org.jooq.Record;
import org.jooq.Result;
import org.jooq.SelectFinalStep;
import org.jooq.SelectJoinStep;
import org.jooq.impl.DSL;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.banner.model.old.OldBanner;
import ru.yandex.direct.core.entity.banner.model.old.OldBannerStatusModerate;
import ru.yandex.direct.core.entity.banner.model.old.OldBannerType;
import ru.yandex.direct.core.entity.banner.repository.old.filter.IdFilter;
import ru.yandex.direct.core.entity.banner.repository.old.type.OldBannerRepositoryTypeSupport;
import ru.yandex.direct.core.entity.banner.repository.old.type.OldBannerRepositoryTypeSupportFacade;
import ru.yandex.direct.core.entity.image.repository.BannerImageFormatRepository;
import ru.yandex.direct.dbqueue.LimitOffset;
import ru.yandex.direct.dbschema.ppc.enums.BannerImagesStatusshow;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.model.AppliedChanges;

import static com.google.common.base.Preconditions.checkState;
import static java.util.Collections.emptyList;
import static java.util.function.Function.identity;
import static java.util.stream.Collectors.toList;
import static ru.yandex.direct.core.entity.banner.repository.old.filter.BannersIdFilterFactory.bannerIdsFilter;
import static ru.yandex.direct.core.entity.banner.repository.old.filter.BannersIdFilterFactory.campaignIdsFilter;
import static ru.yandex.direct.core.entity.banner.repository.old.filter.BannersIdFilterFactory.groupIdsFilter;
import static ru.yandex.direct.dbschema.ppc.Tables.AGGREGATOR_DOMAINS;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNERS_CONTENT_PROMOTION;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNERS_CONTENT_PROMOTION_VIDEO;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNERS_INTERNAL;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNERS_MOBILE_CONTENT;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNERS_PERFORMANCE;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNER_BUTTONS;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNER_LOGOS;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNER_PERMALINKS;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNER_PHONES;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNER_PRICES;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNER_TURBOLANDINGS;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNER_TURBOLANDING_PARAMS;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNER_TURBO_GALLERIES;
import static ru.yandex.direct.dbschema.ppc.Tables.IMAGES;
import static ru.yandex.direct.dbschema.ppc.Tables.TURBOLANDINGS;
import static ru.yandex.direct.dbschema.ppc.enums.BannerPermalinksPermalinkAssignType.manual;
import static ru.yandex.direct.dbschema.ppc.tables.BannerDisplayHrefs.BANNER_DISPLAY_HREFS;
import static ru.yandex.direct.dbschema.ppc.tables.BannerImages.BANNER_IMAGES;
import static ru.yandex.direct.dbschema.ppc.tables.BannerImagesFormats.BANNER_IMAGES_FORMATS;
import static ru.yandex.direct.dbschema.ppc.tables.Banners.BANNERS;
import static ru.yandex.direct.dbschema.ppc.tables.FilterDomain.FILTER_DOMAIN;
import static ru.yandex.direct.dbschema.ppc.tables.Phrases.PHRASES;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Repository
@ParametersAreNonnullByDefault
@Deprecated
public class OldBannerRepository {

    private final DslContextProvider ppcDslContextProvider;
    private final ShardHelper shardHelper;
    private final OldBannerImageRepository bannerImageRepository;
    private final BannerImageFormatRepository bannerImageFormatRepository;
    private final OldBannerRepositoryTypeSupportFacade typeSupportFacade;
    private final Collection<Field<?>> allFields;

    @Autowired
    @SuppressWarnings("checkstyle:parameternumber")
    public OldBannerRepository(DslContextProvider ppcDslContextProvider,
                               ShardHelper shardHelper,
                               OldBannerImageRepository bannerImageRepository,
                               BannerImageFormatRepository bannerImageFormatRepository,
                               OldBannerRepositoryTypeSupportFacade typeSupportFacade) {
        this.ppcDslContextProvider = ppcDslContextProvider;
        this.shardHelper = shardHelper;
        this.bannerImageRepository = bannerImageRepository;
        this.bannerImageFormatRepository = bannerImageFormatRepository;
        this.typeSupportFacade = typeSupportFacade;
        this.allFields = typeSupportFacade.getAllBannerFields();
    }

    /**
     * Сохраняет список баннеров
     * <p>
     * В переданных объектах баннеров так же выставляются сгенерированные id
     *
     * @param shard   шард
     * @param banners список баннеров
     * @return список id сохраненных баннеров
     */
    public <B extends OldBanner> List<Long> addBanners(int shard, List<B> banners) {
        generateBannerIds(banners);

        // пока не потребовалось разделять метод на prepare, add, updateRelatedEntities
        typeSupportFacade.groupByTypeSupports(banners).asMap().forEach(
                (typeSupport, typedBanners) -> typeSupport.add(shard, typedBanners));


        return mapList(banners, OldBanner::getId);
    }

    /**
     * Обновить баннеры в БД. Поддерживается передача изменений баннеров разных типов в одной коллекции.
     *
     * @param <B>            наиболее общий тип баннеров в коллекции
     * @param shard          шард
     * @param appliedChanges применяемые изменения над баннерами
     */

    public <B extends OldBanner> void updateBanners(int shard, Collection<AppliedChanges<B>> appliedChanges) {
        typeSupportFacade.groupChangesByTypeSupports(appliedChanges).asMap().forEach(
                ((typeSupport, changes) -> typeSupport.update(shard, changes)));
    }

    /**
     * Сгенерировать id баннеров (bid) на основе id группы (pid),
     * указанного в каждом баннере, и выставить баннеру сгенерированный id
     * для последующей записи в базу.
     * <p>
     * В баннере должен быть указан id группы (pid).
     *
     * @param banners список баннеров, которым необходимо сгенерировать id
     *                для последующей записи в базу
     */
    private <B extends OldBanner> void generateBannerIds(List<B> banners) {
        List<Long> ids = shardHelper.generateBannerIds(mapList(banners, OldBanner::getAdGroupId));
        checkState(banners.size() == ids.size(), "сгенерилось неверное количество bid");
        for(int i = 0; i < ids.size(); ++i) {
            banners.get(i).setId(ids.get(i));
        }
    }

    /**
     * Получает список баннеров по списку их id
     *
     * @param bannerIds список id извлекаемых баннеров
     * @param shard     шард
     * @return список баннеров
     */
    public List<OldBanner> getBanners(int shard, Collection<Long> bannerIds) {
        if (bannerIds.isEmpty()) {
            return emptyList();
        }
        return getBannersByIds(shard, bannerIdsFilter(bannerIds), null);
    }

    /**
     * Получает список баннеров по списку id их групп
     *
     * @param groupIds список id групп извлекаемых баннеров
     * @param shard    шард
     * @return список баннеров
     */
    public List<OldBanner> getBannersByGroupIds(int shard, Collection<Long> groupIds) {
        if (groupIds.isEmpty()) {
            return emptyList();
        }
        return getBannersByIds(shard, groupIdsFilter(groupIds));
    }

    /**
     * Получить баннеры принадлежащие группам.
     * <i>Поддержаны только текстовые баннеры.</i>
     *
     * @param adGroupIds список id групп текстовых объявлений
     * @param shard      шард
     * @return мапа из баннеров, ключ - id группы, значение - список баннеров принадлежащих группе
     */
    public Map<Long, List<OldBanner>> getBannersByAdGroups(int shard, Collection<Long> adGroupIds) {
        return EntryStream.of(
                selectAllBannersFields(shard)
                        .where(BANNERS.PID.in(adGroupIds))
                        .fetchGroups(BANNERS.PID)
                        .entrySet()
                        .stream())
                .mapValues(this::fromDb)
                .toMap();
    }

    private List<OldBanner> fromDb(List<Record> arbitraryBannerTypesRecordList) {
        return StreamEx.of(arbitraryBannerTypesRecordList)
                .mapToEntry(identity(), typeSupportFacade::getTypeSupport)
                .mapKeyValue(this::fromDbWithTypeSupport)
                .collect(toList());
    }

    private OldBanner fromDbWithTypeSupport(Record record, OldBannerRepositoryTypeSupport typeSupport) {
        return typeSupport.createBannerFromRecord(record);
    }

    /**
     * Получает список баннеров по списку id их кампаний
     *
     * @param campaignIds список id кампаний извлекаемых баннеров
     * @param shard       шард
     * @return список баннеров
     */
    public List<OldBanner> getBannersByCampaignIds(int shard, Collection<Long> campaignIds) {
        if (campaignIds.isEmpty()) {
            return emptyList();
        }
        return getBannersByIds(shard, campaignIdsFilter(campaignIds));
    }

    /**
     * Получает список баннеров по списку id их групп или по id самих баннеров.
     * Может собирать баннеры разных типов.
     *
     * @param shard       шард
     * @param filter      список id групп или id баннеров по которым ищем баннеры
     * @param limitOffset ограничения на размер результата
     * @return список баннеров
     */
    private List<OldBanner> getBannersByIds(int shard, IdFilter filter, @Nullable LimitOffset limitOffset) {
        SelectJoinStep<Record> step = selectAllBannersFields(shard);
        return getBannersByIds(shard, filter, step, limitOffset);
    }

    /**
     * Получает список баннеров по списку id их групп или по id самих баннеров.
     * Может собирать баннеры разных типов.
     *
     * @param shard  шард
     * @param filter список id групп или id баннеров по которым ищем баннеры
     * @return список баннеров
     */
    private List<OldBanner> getBannersByIds(int shard, IdFilter filter) {
        SelectJoinStep<Record> step = selectAllBannersFields(shard);
        return getBannersByIds(shard, filter, step, null);
    }

    private List<OldBanner> getBannersByIds(int shard, IdFilter filter, SelectJoinStep<Record> step,
                                            @Nullable LimitOffset limitOffset) {

        filter.apply(step);

        SelectFinalStep<Record> finalStep = step;

        if (limitOffset != null) {
            finalStep = step.offset(limitOffset.offset()).limit(limitOffset.limit());
        }

        Result<Record> records = finalStep.fetch();

        Multimap<OldBannerType, Record> recordTypeMultimap = typeSupportFacade.groupRecordsByTypes(records);


        return EntryStream.of(recordTypeMultimap.asMap())
                .mapKeys(typeSupportFacade::getTypeSupport)
                .flatMapKeyValue((typeSupport, recs) ->
                        typeSupport.createBannersFromRecordsWithAdditionsAttached(shard, recs))
                .toList();
    }

    /**
     * Обновить statusModerate для баннеров с указанным id
     *
     * @param shard     шард
     * @param bannerIds список id баннеров
     * @param status    новый статус
     * @return количество изменённых строк
     */
    public int updateStatusModerate(int shard, Collection<Long> bannerIds, OldBannerStatusModerate status) {
        return updateStatusModerate(ppcDslContextProvider.ppc(shard), bannerIds, status);
    }

    public int updateStatusModerate(Configuration configuration, Collection<Long> bannerIds, OldBannerStatusModerate status) {
        return updateStatusModerate(DSL.using(configuration), bannerIds, status);

    }

    //в новых баннерах см. BannerModerationRepository.updateStatusModerate
    @Deprecated
    public int updateStatusModerate(DSLContext dsl, Collection<Long> bannerIds, OldBannerStatusModerate status) {
        if (bannerIds.isEmpty()) {
            return 0;
        }
        return dsl
                .update(BANNERS)
                .set(BANNERS.STATUS_MODERATE, OldBannerStatusModerate.toSource(status))
                .where(BANNERS.BID.in(bannerIds))
                .execute();
    }

    private SelectJoinStep<Record> selectAllBannersFields(int shard) {
        SelectJoinStep<Record> select = ppcDslContextProvider.ppc(shard)
                .select(allFields)
                .from(BANNERS);
        return joinAllBannersRelatedTables(select);
    }

    private SelectJoinStep<Record> joinAllBannersRelatedTables(SelectJoinStep<Record> step) {
        return step
                .join(PHRASES).on(PHRASES.PID.eq(BANNERS.PID))
                .leftJoin(BANNER_IMAGES).on(BANNER_IMAGES.BID.eq(BANNERS.BID)
                        .and(BANNER_IMAGES.STATUS_SHOW.eq(BannerImagesStatusshow.Yes)))
                .leftJoin(IMAGES).on(IMAGES.BID.eq(BANNERS.BID))
                .leftJoin(BANNER_IMAGES_FORMATS).on(BANNER_IMAGES_FORMATS.IMAGE_HASH.eq(BANNER_IMAGES.IMAGE_HASH))
                .leftJoin(BANNER_LOGOS).on(BANNER_LOGOS.BID.eq(BANNERS.BID))
                .leftJoin(BANNER_BUTTONS).on(BANNER_BUTTONS.BID.eq(BANNERS.BID))
                .leftJoin(BANNER_DISPLAY_HREFS).on(BANNER_DISPLAY_HREFS.BID.eq(BANNERS.BID))
                .leftJoin(BANNERS_PERFORMANCE).on(BANNERS_PERFORMANCE.BID.eq(BANNERS.BID))
                .leftJoin(BANNERS_INTERNAL).on(BANNERS_INTERNAL.BID.eq(BANNERS.BID))
                .leftJoin(FILTER_DOMAIN).on(FILTER_DOMAIN.DOMAIN.eq(BANNERS.DOMAIN))
                .leftJoin(BANNER_TURBOLANDINGS).on(BANNER_TURBOLANDINGS.BID.eq(BANNERS.BID))
                .leftJoin(BANNER_TURBOLANDING_PARAMS).on(BANNER_TURBOLANDING_PARAMS.BID.eq(BANNERS.BID))
                .leftJoin(TURBOLANDINGS).on(TURBOLANDINGS.TL_ID.eq(BANNER_TURBOLANDINGS.TL_ID))
                .leftJoin(BANNER_TURBO_GALLERIES).on(BANNER_TURBO_GALLERIES.BID.eq(BANNERS.BID))
                .leftJoin(BANNERS_MOBILE_CONTENT).on(BANNERS_MOBILE_CONTENT.BID.eq(BANNERS.BID))
                .leftJoin(BANNER_PRICES).on(BANNER_PRICES.BID.eq(BANNERS.BID))
                .leftJoin(BANNERS_CONTENT_PROMOTION_VIDEO).on(BANNERS_CONTENT_PROMOTION_VIDEO.BID.eq(BANNERS.BID))
                .leftJoin(BANNERS_CONTENT_PROMOTION).on(BANNERS_CONTENT_PROMOTION.BID.eq(BANNERS.BID))
                .leftJoin(BANNER_PERMALINKS).on(BANNERS.BID.eq(BANNER_PERMALINKS.BID)
                        .and(BANNER_PERMALINKS.PERMALINK_ASSIGN_TYPE.eq(manual)))
                .leftJoin(BANNER_PHONES).on(BANNERS.BID.eq(BANNER_PHONES.BID))
                .leftJoin(AGGREGATOR_DOMAINS).on(AGGREGATOR_DOMAINS.BID.eq(BANNERS.BID));
    }
}
