package ru.yandex.direct.core.entity.banner.repository.old.container;

import java.util.ArrayList;
import java.util.Collection;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Set;
import java.util.stream.Stream;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.direct.model.Model;

/**
 * DTO, содержащий два множества моделей: которые нужно удалить,
 * и которые нужно добавить или обновить.
 * <p>
 * Полезен, когда логика разбиения на эти множества реализована
 * в одном месте, а вызовы обновления и удаления могут быть
 * в различных классах.
 * <p>
 * Если требуется отделить множества "на добавление" и "на обновление",
 * стоит использовать {@link InsertUpdateDeleteContainer}.
 *
 * @param <M> тип моделей
 */
@ParametersAreNonnullByDefault
public class AddOrUpdateAndDeleteContainer<M extends Model> {

    private final Set<M> modelsToAddOrUpdate = new LinkedHashSet<>();
    private final Set<M> modelsToDelete = new LinkedHashSet<>();

    /**
     * Отметить объект как подлежащий сохранению (добавить или обновить).
     * Добавление не удастся, если этот же объект был ранее передан в {@link #markForDeletion}.
     * После успешного добавления объект будет содержаться в коллекции,
     * возвращаемой методом {@link #getModelsToAddOrUpdate}.
     *
     * @return {@code true}, если добавление объекта успешно, иначе – {@code false}.
     */
    @SuppressWarnings("UnusedReturnValue")
    public boolean markForAddOrUpdate(M model) {
        return put(model, modelsToAddOrUpdate);
    }

    /**
     * Отметить объект как подлежащий удалению. Добавление не удастся, если
     * этот же объект был ранее передан в {@link #markForAddOrUpdate}.
     * После успешного добавления объект будет содержаться в коллекции,
     * возвращаемой методом {@link #getModelsToDelete}.
     *
     * @return {@code true}, если добавление объекта успешно, иначе – {@code false}.
     */
    @SuppressWarnings("UnusedReturnValue")
    public boolean markForDeletion(M model) {
        return put(model, modelsToDelete);
    }

    /**
     * Получить модели, отмеченные подлежащими удалению.
     *
     * @see #markForDeletion
     */
    public List<M> getModelsToDelete() {
        return toList(modelsToDelete);
    }

    /**
     * Получить модели, отмеченные ранее подлежащими сохранению.
     *
     * @see #markForAddOrUpdate
     */
    public List<M> getModelsToAddOrUpdate() {
        return toList(modelsToAddOrUpdate);
    }

    private boolean put(M model, Collection<? super M> destination) {
        boolean alreadyPresent = Stream.of(modelsToAddOrUpdate, modelsToDelete)
                .anyMatch(set -> set.contains(model));
        //noinspection SimplifiableConditionalExpression
        return alreadyPresent ? false : destination.add(model);
    }

    private static <T> List<T> toList(Collection<T> source) {
        return new ArrayList<>(source);
    }
}
