package ru.yandex.direct.core.entity.banner.repository.old.container;

import java.util.ArrayList;
import java.util.Collection;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Set;
import java.util.stream.Stream;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.direct.model.Model;

/**
 * DTO, содержащий три множества моделей: которые нужно удалить,
 * которые нужно добавить, и которые нужно обновить.
 * <p>
 * Полезен, когда логика разбиения на эти множества реализована
 * в одном месте, а вызовы добавления, обновления и удаления могут быть
 * в различных классах.
 * <p>
 * Если разграничение множеств "на добавление" и "на обновление" не требуется,
 * а требуется лишь "сохранить" и "удалить",
 * удобней использовать {@link AddOrUpdateAndDeleteContainer}.
 *
 * @param <M> тип моделей
 */
@ParametersAreNonnullByDefault
public class InsertUpdateDeleteContainer<M extends Model> {

    private final Set<M> modelsToInsert = new LinkedHashSet<>();
    private final Set<M> modelsToUpdate = new LinkedHashSet<>();
    private final Set<M> modelsToDelete = new LinkedHashSet<>();

    /**
     * Отметить объект как подлежащий добавлению.
     * Добавление не удастся, если этот же объект был ранее передан
     * в {@link #markForDeletion} или {@link #markForUpdate}.
     * После успешного добавления объект будет содержаться в коллекции,
     * возвращаемой методом {@link #getModelsToInsert}.
     *
     * @return {@code true}, если добавление объекта успешно, иначе – {@code false}.
     */
    public boolean markForInsertion(M model) {
        return put(model, modelsToInsert);
    }

    /**
     * Отметить объект как подлежащий обновлению.
     * Добавление не удастся, если этот же объект был ранее передан
     * в {@link #markForInsertion} или {@link #markForDeletion}.
     * После успешного добавления объект будет содержаться в коллекции,
     * возвращаемой методом {@link #getModelsToUpdate}.
     *
     * @return {@code true}, если добавление объекта успешно, иначе – {@code false}.
     */
    public boolean markForUpdate(M model) {
        return put(model, modelsToUpdate);
    }

    /**
     * Отметить объект как подлежащий удалению.
     * Добавление не удастся, если этот же объект был ранее передан
     * в {@link #markForInsertion} или {@link #markForUpdate}.
     * После успешного добавления объект будет содержаться в коллекции,
     * возвращаемой методом {@link #getModelsToDelete}.
     *
     * @return {@code true}, если добавление объекта успешно, иначе – {@code false}.
     */
    public boolean markForDeletion(M model) {
        return put(model, modelsToDelete);
    }

    /**
     * Получить модели, отмеченные подлежащими добавлению.
     *
     * @see #markForInsertion
     */
    public List<M> getModelsToInsert() {
        return toList(modelsToInsert);
    }

    /**
     * Получить модели, отмеченные подлежащими обновлению.
     *
     * @see #markForUpdate
     */
    public List<M> getModelsToUpdate() {
        return toList(modelsToUpdate);
    }

    /**
     * Получить модели, отмеченные подлежащими удалению.
     *
     * @see #markForDeletion
     */
    public List<M> getModelsToDelete() {
        return toList(modelsToDelete);
    }

    private boolean put(M model, Collection<? super M> destination) {
        boolean alreadyPresent = Stream.of(modelsToInsert, modelsToUpdate, modelsToDelete)
                .anyMatch(set -> set.contains(model));
        //noinspection SimplifiableConditionalExpression
        return alreadyPresent ? false : destination.add(model);
    }

    private static <T> List<T> toList(Collection<T> source) {
        return new ArrayList<>(source);
    }
}
