package ru.yandex.direct.core.entity.banner.repository.old.type;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.stream.Stream;

import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.DSLContext;
import org.jooq.Field;
import org.jooq.Record;
import org.jooq.impl.DSL;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.banner.model.old.OldBannerCreative;
import ru.yandex.direct.core.entity.banner.model.old.OldBannerType;
import ru.yandex.direct.core.entity.banner.model.old.OldCpcVideoBanner;
import ru.yandex.direct.core.entity.banner.repository.old.container.InsertUpdateDeleteContainer;
import ru.yandex.direct.core.entity.banner.turbolanding.model.OldBannerTurboLanding;
import ru.yandex.direct.core.entity.banner.turbolanding.model.OldBannerTurboLandingParams;
import ru.yandex.direct.core.entity.creative.model.Creative;
import ru.yandex.direct.core.entity.creative.repository.CreativeRepository;
import ru.yandex.direct.core.entity.domain.service.DomainService;
import ru.yandex.direct.core.entity.domain.service.OldBannerAggregatorDomainsService;
import ru.yandex.direct.dbschema.ppc.tables.Banners;
import ru.yandex.direct.dbschema.ppc.tables.records.BannersRecord;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;
import ru.yandex.direct.jooqmapperhelper.JooqUpdateBuilder;
import ru.yandex.direct.model.AppliedChanges;

import static java.util.stream.Collectors.toList;
import static ru.yandex.direct.core.entity.banner.repository.old.mapper.BannerCreativeMapperProvider.getBannerCreativeMapper;
import static ru.yandex.direct.core.entity.banner.repository.old.mapper.CpcVideoBannerMapperProvider.createCpcVideoBannerUpdateBuilder;
import static ru.yandex.direct.core.entity.banner.repository.old.mapper.CpcVideoBannerMapperProvider.getCpcVideoBannerMapper;
import static ru.yandex.direct.core.entity.banner.repository.old.mapper.turbolanding.BannerTurboLandingMapperProvider.getBannerTurboLandingMapper;
import static ru.yandex.direct.core.entity.banner.repository.old.mapper.turbolanding.BannerTurboLandingParamsMapperProvider.getBannerTurboLandingParamsMapper;
import static ru.yandex.direct.core.entity.creative.service.CreativeUtils.extractModerationInfoPropertyFromCreative;
import static ru.yandex.direct.dbschema.ppc.tables.Banners.BANNERS;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Реализация {@link OldBannerRepositoryTypeSupport} для графических объявлений.
 */
@Component
@ParametersAreNonnullByDefault
@Deprecated
public class OldBannerRepositoryCpcVideoTypeSupport extends OldAbstractBannerRepositoryTypeSupport<OldCpcVideoBanner> {

    private static final String TITLE = "title";
    private static final String BODY = "body";

    private final DomainService domainService;
    private final OldBannerWithCreativeSupport bannerWithCreativeSupport;
    private final OldBannerWithTurboLandingSupport bannerWithTurboLandingSupport;
    private final OldBannerAggregatorDomainsService aggregatorDomainsService;

    private final JooqMapperWithSupplier<OldCpcVideoBanner> cpcVideoBannerMapper = getCpcVideoBannerMapper();
    private final JooqMapperWithSupplier<OldBannerCreative> bannerCreativeMapper = getBannerCreativeMapper();
    private final JooqMapperWithSupplier<OldBannerTurboLanding> bannerTurboLandingMapper =
            getBannerTurboLandingMapper();
    private final JooqMapperWithSupplier<OldBannerTurboLandingParams> bannerTurboLandingParamsMapper =
            getBannerTurboLandingParamsMapper();

    private final CreativeRepository creativeRepository;

    private final Collection<Field<?>> allFields;

    @Autowired
    OldBannerRepositoryCpcVideoTypeSupport(DslContextProvider dslContextProvider,
                                           ShardHelper shardHelper,
                                           DomainService domainService,
                                           OldBannerWithCreativeSupport bannerWithCreativeSupport,
                                           OldBannerWithTurboLandingSupport bannerWithTurboLandingSupport,
                                           OldBannerAggregatorDomainsService aggregatorDomainsService,
                                           CreativeRepository creativeRepository) {
        super(dslContextProvider, shardHelper);

        this.domainService = domainService;
        this.bannerWithCreativeSupport = bannerWithCreativeSupport;
        this.bannerWithTurboLandingSupport = bannerWithTurboLandingSupport;
        this.aggregatorDomainsService = aggregatorDomainsService;
        this.creativeRepository = creativeRepository;

        allFields = Stream.of(
                cpcVideoBannerMapper.getFieldsToRead(),
                bannerCreativeMapper.getFieldsToRead(),
                bannerTurboLandingMapper.getFieldsToRead(),
                bannerTurboLandingParamsMapper.getFieldsToRead()
        ).flatMap(Collection::stream)
                .collect(toList());
    }

    @Override
    public OldBannerType getType() {
        return OldBannerType.CPC_VIDEO;
    }

    @Override
    public Collection<Field<?>> getAllFields() {
        return allFields;
    }

    @Override
    public void update(int shard, Collection<AppliedChanges<OldCpcVideoBanner>> appliedChanges) {

        // Берем все объявления с измененными креативами и обновляем title и body у них.
        Map<Long, AppliedChanges<OldCpcVideoBanner>> mapBannerIdToAppliedChanges =
                listToMap(filterList(appliedChanges, ac -> ac.changed(OldCpcVideoBanner.CREATIVE_ID)),
                        ac -> ac.getModel().getId());

        List<OldCpcVideoBanner> banners = mapList(mapBannerIdToAppliedChanges.values(), AppliedChanges::getModel);

        Map<Long, Creative> creatives = listToMap(
                creativeRepository.getCreatives(shard, mapList(banners, OldCpcVideoBanner::getCreativeId)),
                Creative::getId
        );

        banners.forEach(b -> {
            Creative creative = creatives.get(b.getCreativeId());
            AppliedChanges<OldCpcVideoBanner> appliedChangesItem = mapBannerIdToAppliedChanges.get(b.getId());
            appliedChangesItem.modify(OldCpcVideoBanner.TITLE, extractModerationInfoPropertyFromCreative(creative, TITLE));
            appliedChangesItem.modify(OldCpcVideoBanner.BODY, extractModerationInfoPropertyFromCreative(creative, BODY));
        });

        domainService.addNewDomainsAndUpdateDomainIds(shard, appliedChanges);

        InsertUpdateDeleteContainer<OldBannerCreative> bannerCreativesUpdateContainer =
                bannerWithCreativeSupport.prepareCreativesUpdateContainer(appliedChanges);
        Collection<OldBannerCreative> creativesToInsert = bannerCreativesUpdateContainer.getModelsToInsert();
        Collection<OldBannerCreative> creativesToUpdate = bannerCreativesUpdateContainer.getModelsToUpdate();
        Collection<OldBannerCreative> creativesToDelete = bannerCreativesUpdateContainer.getModelsToDelete();


        JooqUpdateBuilder<BannersRecord, OldCpcVideoBanner> bannerUpdateBuilder =
                createCpcVideoBannerUpdateBuilder(appliedChanges);

        // Этот комментарий предупреждает о дублировании логики работы с креативом,
        // сейчас так должно быть, но не обязано дублировать в будущем.
        // (логика внутри транзакции пока дублирует логику обновлеия cpm_banner - но есть подозрения,
        // что работа с perf_creatives может расходиться)
        // при расхождении комментарий удалить.

        dslContext(shard).transaction(ctx -> {
            DSL.using(ctx)
                    .update(BANNERS)
                    .set(bannerUpdateBuilder.getValues())
                    .where(Banners.BANNERS.BID.in(bannerUpdateBuilder.getChangedIds()))
                    .execute();

            bannerWithCreativeSupport.insertCreatives(ctx, creativesToInsert);
            bannerWithCreativeSupport.updateCreatives(ctx, creativesToUpdate);
            bannerWithCreativeSupport.deleteCreatives(ctx, creativesToDelete);

            bannerWithTurboLandingSupport.updateTurbolandings(appliedChanges, ctx);
            bannerWithTurboLandingSupport.updateTurbolandingParams(appliedChanges, ctx);

            aggregatorDomainsService.updateAggregatorDomains(ctx, appliedChanges);
        });

    }

    @Override
    void beforeBannersAdd(int shard, Collection<OldCpcVideoBanner> banners) {
        super.beforeBannersAdd(shard, banners);

        Map<Long, Creative> creatives = listToMap(
                creativeRepository.getCreatives(shard, mapList(banners, OldCpcVideoBanner::getCreativeId)),
                Creative::getId
        );

        banners.forEach(b -> {
            Creative creative = creatives.get(b.getCreativeId());
            b.setTitle(extractModerationInfoPropertyFromCreative(creative, TITLE));
            b.setBody(extractModerationInfoPropertyFromCreative(creative, BODY));
        });
        domainService.addDomainsAndSetDomainIds(shard, banners);
        bannerWithCreativeSupport.addBannerCreatives(banners, shard);
    }

    @Override
    void addBannersToDb(Collection<OldCpcVideoBanner> banners, DSLContext dslContext) {
        super.addBannersToDb(banners, dslContext);

        InsertHelper<BannersRecord> bannersInsertHelper = new InsertHelper<>(dslContext, BANNERS);
        bannersInsertHelper.addAll(cpcVideoBannerMapper, banners);

        bannersInsertHelper.executeIfRecordsAdded();
    }

    @Override
    public void afterBannersAdded(Collection<OldCpcVideoBanner> banners, DSLContext dslContext) {
        bannerWithTurboLandingSupport.updateAfterBannersAdded(banners, dslContext);
        aggregatorDomainsService.updateAggregatorDomains(dslContext, banners);
    }

    @Override
    public OldCpcVideoBanner createBannerFromRecord(Record record) {
        OldCpcVideoBanner banner = cpcVideoBannerMapper.fromDb(record);

        OldBannerTurboLandingParams turboLandingParams = bannerTurboLandingParamsMapper.fromDb(record);
        if (turboLandingParams.getBannerId() != null) {
            banner.withTurboLandingParams(turboLandingParams);
        }
        return banner;
    }

    @Override
    public Stream<OldCpcVideoBanner> createBannersFromRecordsWithAdditionsAttached(int shard, Collection<Record> records) {
        return records.stream().map(this::createBannerFromRecord);
    }
}
