package ru.yandex.direct.core.entity.banner.repository.old.type;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.stream.Stream;

import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.DSLContext;
import org.jooq.Field;
import org.jooq.Record;
import org.jooq.impl.DSL;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.banner.model.old.OldBannerButton;
import ru.yandex.direct.core.entity.banner.model.old.OldBannerCreative;
import ru.yandex.direct.core.entity.banner.model.old.OldBannerPixel;
import ru.yandex.direct.core.entity.banner.model.old.OldBannerTns;
import ru.yandex.direct.core.entity.banner.model.old.OldBannerType;
import ru.yandex.direct.core.entity.banner.model.old.OldBannerWithTns;
import ru.yandex.direct.core.entity.banner.model.old.OldCpmBanner;
import ru.yandex.direct.core.entity.banner.repository.old.OldBannerAdditionalHrefsRepository;
import ru.yandex.direct.core.entity.banner.repository.old.OldBannerButtonsRepository;
import ru.yandex.direct.core.entity.banner.repository.old.OldBannerLogosRepository;
import ru.yandex.direct.core.entity.banner.repository.old.OldBannerMeasurersRepository;
import ru.yandex.direct.core.entity.banner.repository.old.OldBannerPixelsRepository;
import ru.yandex.direct.core.entity.banner.repository.old.container.BannerPixelsUpdateContainer;
import ru.yandex.direct.core.entity.banner.repository.old.container.InsertUpdateDeleteContainer;
import ru.yandex.direct.core.entity.banner.turbolanding.model.OldBannerTurboLanding;
import ru.yandex.direct.core.entity.banner.turbolanding.model.OldBannerTurboLandingParams;
import ru.yandex.direct.core.entity.domain.service.DomainService;
import ru.yandex.direct.core.entity.domain.service.OldBannerAggregatorDomainsService;
import ru.yandex.direct.dbschema.ppc.tables.Banners;
import ru.yandex.direct.dbschema.ppc.tables.records.BannersRecord;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;
import ru.yandex.direct.jooqmapperhelper.JooqUpdateBuilder;
import ru.yandex.direct.model.AppliedChanges;

import static java.util.Collections.emptyList;
import static java.util.stream.Collectors.toList;
import static org.springframework.util.CollectionUtils.isEmpty;
import static ru.yandex.direct.core.entity.banner.repository.old.mapper.BannerButtonsMapperProvider.getBannerButtonsMapper;
import static ru.yandex.direct.core.entity.banner.repository.old.mapper.BannerCreativeMapperProvider.getBannerCreativeMapper;
import static ru.yandex.direct.core.entity.banner.repository.old.mapper.CpmBannerMapperProvider.createCpmBannerUpdateBuilder;
import static ru.yandex.direct.core.entity.banner.repository.old.mapper.CpmBannerMapperProvider.getCpmBannerMapper;
import static ru.yandex.direct.core.entity.banner.repository.old.mapper.turbolanding.BannerTurboLandingMapperProvider.getBannerTurboLandingMapper;
import static ru.yandex.direct.core.entity.banner.repository.old.mapper.turbolanding.BannerTurboLandingParamsMapperProvider.getBannerTurboLandingParamsMapper;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNERS;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNER_BUTTONS;
import static ru.yandex.direct.utils.FunctionalUtils.filterAndMapList;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;

/**
 * Реализация {@link OldBannerRepositoryTypeSupport} для медийных баннеров.
 */
@Component
@ParametersAreNonnullByDefault
@Deprecated
public class OldBannerRepositoryCpmBannerTypeSupport extends OldAbstractBannerRepositoryTypeSupport<OldCpmBanner> {

    private final DomainService domainService;
    private final OldBannerPixelsRepository bannerPixelsRepository;
    private final OldBannerLogosRepository<OldCpmBanner> bannerLogosRepository;
    private final OldBannerButtonsRepository bannerButtonsRepository;
    private final OldBannerWithCreativeSupport bannerWithCreativeSupport;
    private final OldBannerWithPixelsSupport bannerWithPixelsSupport;
    private final OldBannerWithTurboLandingSupport bannerWithTurboLandingSupport;
    private final OldBannerWithMeasurersSupport bannerWithMeasurersSupport;
    private final OldBannerMeasurersRepository bannerMeasurersRepository;
    private final OldBannerAdditionalHrefsRepository bannerAdditionalHrefsRepository;
    private final OldBannerAggregatorDomainsService aggregatorDomainsService;

    private final JooqMapperWithSupplier<OldCpmBanner> cpmBannerMapper = getCpmBannerMapper();
    private final JooqMapperWithSupplier<OldBannerCreative> bannerCreativeMapper = getBannerCreativeMapper();
    private final JooqMapperWithSupplier<OldBannerTurboLanding> bannerTurboLandingMapper =
            getBannerTurboLandingMapper();
    private final JooqMapperWithSupplier<OldBannerTurboLandingParams> bannerTurboLandingParamsMapper =
            getBannerTurboLandingParamsMapper();
    private final JooqMapperWithSupplier<OldBannerButton> bannerButtonsMapper = getBannerButtonsMapper();

    private final Collection<Field<?>> allFields;

    @Autowired
    OldBannerRepositoryCpmBannerTypeSupport(
            DslContextProvider dslContextProvider,
            ShardHelper shardHelper,
            DomainService domainService,
            OldBannerPixelsRepository bannerPixelsRepository,
            OldBannerButtonsRepository bannerButtonsRepository,
            OldBannerWithCreativeSupport bannerWithCreativeSupport,
            OldBannerWithPixelsSupport bannerWithPixelsSupport,
            OldBannerWithTurboLandingSupport bannerWithTurboLandingSupport,
            OldBannerWithMeasurersSupport bannerWithMeasurersSupport,
            OldBannerMeasurersRepository bannerMeasurersRepository,
            OldBannerAdditionalHrefsRepository bannerAdditionalHrefsRepository,
            OldBannerAggregatorDomainsService aggregatorDomainsService) {
        super(dslContextProvider, shardHelper);

        this.domainService = domainService;
        this.bannerPixelsRepository = bannerPixelsRepository;
        this.bannerLogosRepository = new OldBannerLogosRepository<>(dslContextProvider, OldCpmBanner::new);
        this.bannerButtonsRepository = bannerButtonsRepository;
        this.bannerWithCreativeSupport = bannerWithCreativeSupport;
        this.bannerWithPixelsSupport = bannerWithPixelsSupport;
        this.bannerWithTurboLandingSupport = bannerWithTurboLandingSupport;
        this.bannerWithMeasurersSupport = bannerWithMeasurersSupport;
        this.bannerMeasurersRepository = bannerMeasurersRepository;
        this.bannerAdditionalHrefsRepository = bannerAdditionalHrefsRepository;
        this.aggregatorDomainsService = aggregatorDomainsService;

        allFields = Stream.of(cpmBannerMapper.getFieldsToRead(),
                bannerCreativeMapper.getFieldsToRead(),
                bannerTurboLandingMapper.getFieldsToRead(),
                bannerTurboLandingParamsMapper.getFieldsToRead(),
                bannerButtonsMapper.getFieldsToRead())
                .flatMap(Collection::stream)
                .collect(toList());
    }

    @Override
    public OldBannerType getType() {
        return OldBannerType.CPM_BANNER;
    }

    @Override
    public Collection<Field<?>> getAllFields() {
        return allFields;
    }

    @Override
    public void update(int shard, Collection<AppliedChanges<OldCpmBanner>> appliedChanges) {
        domainService.addNewDomainsAndUpdateDomainIds(shard, appliedChanges);

        InsertUpdateDeleteContainer<OldBannerCreative> bannerCreativesUpdateContainer =
                bannerWithCreativeSupport.prepareCreativesUpdateContainer(appliedChanges);
        Collection<OldBannerCreative> creativesToInsert = bannerCreativesUpdateContainer.getModelsToInsert();
        Collection<OldBannerCreative> creativesToUpdate = bannerCreativesUpdateContainer.getModelsToUpdate();
        Collection<OldBannerCreative> creativesToDelete = bannerCreativesUpdateContainer.getModelsToDelete();

        // подготовка пикселей к изменению
        BannerPixelsUpdateContainer bannerPixelsUpdateContainer =
                bannerWithPixelsSupport.preparePixelsUpdateContainer(appliedChanges);
        List<OldBannerPixel> bannersPixelsToAdd = bannerPixelsUpdateContainer.getBannerPixelsToAdd();
        Map<Long, List<String>> bannersPixelsToDelete = bannerPixelsUpdateContainer.getBannerPixelsToDelete();

        JooqUpdateBuilder<BannersRecord, OldCpmBanner> bannerUpdateBuilder = createCpmBannerUpdateBuilder(appliedChanges);

        dslContext(shard).transaction(ctx -> {
            DSL.using(ctx)
                    .update(BANNERS)
                    .set(bannerUpdateBuilder.getValues())
                    .where(Banners.BANNERS.BID.in(bannerUpdateBuilder.getChangedIds()))
                    .execute();

            bannerWithCreativeSupport.insertCreatives(ctx, creativesToInsert);
            bannerWithCreativeSupport.updateCreatives(ctx, creativesToUpdate);
            bannerWithCreativeSupport.deleteCreatives(ctx, creativesToDelete);

            bannerPixelsRepository.addBannerPixels(bannersPixelsToAdd, ctx.dsl());
            bannerPixelsRepository.deleteBannerPixels(bannersPixelsToDelete, ctx.dsl());

            updateBannerLogosTable(appliedChanges, ctx.dsl());
            updateBannerButtonsTable(appliedChanges, ctx.dsl());
            updateBannerTnsTable(appliedChanges, ctx.dsl());

            bannerWithTurboLandingSupport.updateTurbolandings(appliedChanges, ctx);
            bannerWithTurboLandingSupport.updateTurbolandingParams(appliedChanges, ctx);

            bannerWithMeasurersSupport.updateMeasurers(appliedChanges, ctx);

            bannerAdditionalHrefsRepository.updateBannerAdditionalHrefs(ctx.dsl(), appliedChanges);

            aggregatorDomainsService.updateAggregatorDomains(ctx, appliedChanges);
        });
    }

    private void updateBannerTnsTable(Collection<AppliedChanges<OldCpmBanner>> appliedChanges, DSLContext context) {
        List<OldBannerTns> newValues = new ArrayList<>();
        List<OldBannerTns> oldValues = new ArrayList<>();

        for (AppliedChanges<OldCpmBanner> ac : appliedChanges) {
            if (!ac.changed(OldBannerWithTns.TNS_ID)) {
                continue;
            }
            OldBannerTns newValue = OldBannerPixelsRepository.extractBannerTns(ac.getModel());
            if (newValue.getTnsId() == null) {
                oldValues.add(newValue.withTnsId(ac.getOldValue(OldBannerWithTns.TNS_ID)));
            } else {
                newValues.add(newValue);
            }
        }

        bannerPixelsRepository.saveBannerTns(newValues, context);
        bannerPixelsRepository.deleteBannerTns(oldValues, context);
    }

    private void updateBannerLogosTable(Collection<AppliedChanges<OldCpmBanner>> appliedChanges, DSLContext dsl) {
        List<OldCpmBanner> bannersToUpdate = filterAndMapList(appliedChanges,
                ac -> ac.changedAndNotDeleted(OldCpmBanner.LOGO_IMAGE_HASH), AppliedChanges::getModel);
        bannerLogosRepository.update(dsl, bannersToUpdate);

        List<Long> bannersToDelete = filterAndMapList(appliedChanges,
                ac -> ac.deleted(OldCpmBanner.LOGO_IMAGE_HASH), ac -> ac.getModel().getId());
        bannerLogosRepository.delete(dsl, bannersToDelete);
    }

    private void updateBannerButtonsTable(Collection<AppliedChanges<OldCpmBanner>> appliedChanges, DSLContext dsl) {
        List<OldBannerButton> buttonsToUpdate = filterAndMapList(appliedChanges,
                ac -> ac.changedAndNotDeleted(OldCpmBanner.BUTTON_ACTION)
                        || ac.changedAndNotDeleted(OldCpmBanner.BUTTON_CAPTION)
                        || ac.changedAndNotDeleted(OldCpmBanner.BUTTON_HREF),
                ac -> toBannerButton(ac.getModel()));

        bannerButtonsRepository.update(dsl, buttonsToUpdate);

        List<Long> buttonsToDelete = filterAndMapList(appliedChanges,
                ac -> ac.deleted(OldCpmBanner.BUTTON_ACTION), ac -> ac.getModel().getId());
        bannerButtonsRepository.delete(dsl, buttonsToDelete);
    }

    private OldBannerButton toBannerButton(OldCpmBanner cpmBanner) {
        return new OldBannerButton()
                .withBannerId(cpmBanner.getId())
                .withAction(cpmBanner.getButtonAction())
                .withCaption(cpmBanner.getButtonCaption())
                .withHref(cpmBanner.getButtonHref())
                .withStatusModerate(cpmBanner.getButtonStatusModerate());
    }

    @Override
    public void add(int shard, Collection<OldCpmBanner> banners) {
        // prepare
        domainService.addDomainsAndSetDomainIds(shard, banners);
        bannerWithCreativeSupport.addBannerCreatives(banners, shard);
        bannerWithMeasurersSupport.addMeasurers(banners, shard);
        DSLContext dslContext = dslContext(shard);

        // write
        InsertHelper<BannersRecord> bannersInsertHelper = new InsertHelper<>(dslContext, BANNERS);
        bannersInsertHelper.addAll(cpmBannerMapper, banners);

        bannersInsertHelper.executeIfRecordsAdded();

        aggregatorDomainsService.updateAggregatorDomains(dslContext, banners);
        addBannerToBannerPixelsTable(banners, dslContext);
        addBannerToBannersTnsTable(banners, dslContext);
        addBannerToBannerLogosTable(banners, dslContext);
        addBannerToBannerButtonsTable(banners, dslContext);
        bannerWithTurboLandingSupport.updateAfterBannersAdded(banners, dslContext);
        bannerAdditionalHrefsRepository.addBannerAdditionalHrefs(dslContext, banners);
    }

    private void addBannerToBannerPixelsTable(Collection<OldCpmBanner> banners, DSLContext context) {
        List<OldBannerPixel> bannersAllPixelsList = banners
                .stream()
                .filter(b -> !isEmpty(b.getPixels()))
                .map(OldBannerPixelsRepository::extractBannerPixels)
                .flatMap(List::stream)
                .collect(toList());
        bannerPixelsRepository.addBannerPixels(bannersAllPixelsList, context);
    }

    private void addBannerToBannersTnsTable(Collection<OldCpmBanner> banners, DSLContext context) {
        List<OldBannerTns> bannersAllTnsList = banners
                .stream()
                .map(OldBannerPixelsRepository::extractBannerTns)
                .filter(banner -> banner.getTnsId() != null)
                .collect(toList());
        bannerPixelsRepository.saveBannerTns(bannersAllTnsList, context);
    }

    private void addBannerToBannerLogosTable(Collection<OldCpmBanner> banners, DSLContext context) {
        bannerLogosRepository.update(context, filterList(banners, b -> b.getLogoImageHash() != null));
    }

    private void addBannerToBannerButtonsTable(Collection<OldCpmBanner> banners, DSLContext context) {
        List<OldBannerButton> buttons = filterAndMapList(banners, b -> b.getButtonAction() != null, this::toBannerButton);
        bannerButtonsRepository.update(context, buttons);
    }

    @Override
    public OldCpmBanner createBannerFromRecord(Record record) {
        var banner = cpmBannerMapper.fromDb(record);

        var turboLandingParams = bannerTurboLandingParamsMapper.fromDb(record);
        if (turboLandingParams.getBannerId() != null) {
            banner.withTurboLandingParams(turboLandingParams);
        }

        if (record.get(BANNER_BUTTONS.KEY) != null) {
            var bannerButton = bannerButtonsMapper.fromDb(record);
            if (bannerButton.getBannerId() != null) {
                banner.withButtonAction(bannerButton.getAction())
                        .withButtonCaption(bannerButton.getCaption())
                        .withButtonHref(bannerButton.getHref())
                        .withButtonStatusModerate(bannerButton.getStatusModerate());
            }
        }

        return banner;
    }

    @Override
    public Stream<OldCpmBanner> createBannersFromRecordsWithAdditionsAttached(int shard, Collection<Record> records) {
        var banners = records.stream().map(this::createBannerFromRecord).collect(toList());
        var bannerIds = listToSet(banners, OldCpmBanner::getId);

        var bannerIdToPixelUrls = bannerPixelsRepository.getPixelsByBannerIds(shard, bannerIds);
        var bannerIdToBannerMeasurers = bannerMeasurersRepository.getMeasurersByBannerIds(shard, bannerIds);
        var bannerIdToBannerTns = bannerPixelsRepository.getTnsByIds(shard, bannerIds);
        var bannerIdToAdditionalHrefs = bannerAdditionalHrefsRepository.getAdditionalHrefs(shard, bannerIds);

        return banners.stream().peek(banner ->
                banner.withPixels(bannerIdToPixelUrls.getOrDefault(banner.getId(), emptyList()))
                        .withMeasurers(bannerIdToBannerMeasurers.getOrDefault(banner.getId(), emptyList()))
                        .withTnsId(bannerIdToBannerTns.get(banner.getId()))
                        .withAdditionalHrefs(bannerIdToAdditionalHrefs.getOrDefault(banner.getId(), emptyList()))
        );
    }
}
