package ru.yandex.direct.core.entity.banner.repository.old.type;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.DSLContext;
import org.jooq.Field;
import org.jooq.Record;
import org.jooq.impl.DSL;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.banner.model.BannerAddition;
import ru.yandex.direct.core.entity.banner.model.old.OldBanner;
import ru.yandex.direct.core.entity.banner.model.old.OldBannerImage;
import ru.yandex.direct.core.entity.banner.model.old.OldBannerType;
import ru.yandex.direct.core.entity.banner.model.old.OldDynamicBanner;
import ru.yandex.direct.core.entity.banner.repository.old.OldBannerAdditionsRepository;
import ru.yandex.direct.core.entity.banner.repository.old.OldBannerHrefRepository;
import ru.yandex.direct.core.entity.banner.repository.old.OldBannerImageRepository;
import ru.yandex.direct.core.entity.banner.repository.old.container.AddOrUpdateAndDeleteContainer;
import ru.yandex.direct.core.entity.banner.repository.old.container.CalloutsUpdateContainer;
import ru.yandex.direct.core.entity.banner.repository.old.mapper.DynamicBannerMapperProvider;
import ru.yandex.direct.core.entity.domain.service.DomainService;
import ru.yandex.direct.core.entity.domain.service.OldBannerAggregatorDomainsService;
import ru.yandex.direct.dbschema.ppc.tables.records.BannersRecord;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapper;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;
import ru.yandex.direct.jooqmapperhelper.JooqUpdateBuilder;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelProperty;
import ru.yandex.direct.model.ModelWithId;

import static java.util.Collections.emptyList;
import static java.util.function.Function.identity;
import static java.util.stream.Collectors.toList;
import static org.apache.commons.lang3.ObjectUtils.defaultIfNull;
import static org.springframework.util.CollectionUtils.isEmpty;
import static ru.yandex.direct.core.entity.banner.repository.old.mapper.BannerImageMapperProvider.getBannerImageMapper;
import static ru.yandex.direct.core.entity.banner.repository.old.type.OldBannerWithBannerImageSupport.prepareBannerImagesUpdateContainer;
import static ru.yandex.direct.dbschema.ppc.tables.BannerDisplayHrefs.BANNER_DISPLAY_HREFS;
import static ru.yandex.direct.dbschema.ppc.tables.Banners.BANNERS;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;

/**
 * Реализация {@link OldBannerRepositoryTypeSupport} для динамических баннеров.
 */
@Component
@ParametersAreNonnullByDefault
@Deprecated
class OldBannerRepositoryDynamicTypeSupport extends OldAbstractBannerRepositoryTypeSupport<OldDynamicBanner> {

    private final OldBannerAdditionsRepository bannerAdditionsRepository;
    private final DomainService domainService;
    private final OldBannerHrefRepository bannerHrefRepository;
    private final OldBannerImageRepository bannerImageRepository;
    private final OldBannerAggregatorDomainsService aggregatorDomainsService;
    private final OldBannerWithCalloutsSupport bannerWithCalloutsSupport;
    private final DynamicBannerMapperProvider mapperProvider;

    private final JooqMapperWithSupplier<OldDynamicBanner> dynamicBannerMapper;
    private final JooqMapperWithSupplier<OldBannerImage> bannerImageMapper;
    private final Collection<Field<?>> allFields;

    @Autowired
    OldBannerRepositoryDynamicTypeSupport(DslContextProvider dslContextProvider,
                                          ShardHelper shardHelper,
                                          OldBannerAdditionsRepository bannerAdditionsRepository,
                                          DomainService domainService,
                                          OldBannerHrefRepository bannerHrefRepository,
                                          OldBannerImageRepository bannerImageRepository,
                                          OldBannerAggregatorDomainsService aggregatorDomainsService,
                                          OldBannerWithCalloutsSupport bannerWithCalloutsSupport,
                                          DynamicBannerMapperProvider dynamicBannerMapperProvider) {
        super(dslContextProvider, shardHelper);

        this.bannerAdditionsRepository = bannerAdditionsRepository;
        this.domainService = domainService;
        this.bannerHrefRepository = bannerHrefRepository;
        this.bannerImageRepository = bannerImageRepository;
        this.aggregatorDomainsService = aggregatorDomainsService;
        this.bannerWithCalloutsSupport = bannerWithCalloutsSupport;
        this.mapperProvider = dynamicBannerMapperProvider;

        this.dynamicBannerMapper = dynamicBannerMapperProvider.getDynamicBannerMapper();
        this.bannerImageMapper = getBannerImageMapper();

        this.allFields = Stream.of(dynamicBannerMapper, bannerImageMapper)
                .map(JooqMapper::getFieldsToRead)
                .flatMap(Collection::stream)
                .collect(toList());
    }

    @Override
    public OldBannerType getType() {
        return OldBannerType.DYNAMIC;
    }

    @Override
    public Collection<Field<?>> getAllFields() {
        return allFields;
    }

    /**
     * @param shard          шард
     * @param appliedChanges изменения моделей
     * @inheritDoc Обновление некоторых полей баннера и таблицы bannerPerformance (если креатив изменялся).
     */
    @Override
    public void update(int shard, Collection<AppliedChanges<OldDynamicBanner>> appliedChanges) {

        domainService.addNewDomainsAndUpdateDomainIds(shard, appliedChanges);

        // подготовка уточнений к изменению
        CalloutsUpdateContainer calloutsUpdateContainer =
                bannerWithCalloutsSupport.prepareCalloutsUpdateContainer(appliedChanges);
        List<BannerAddition> bannersCalloutsToAddOrUpdate = calloutsUpdateContainer.getBannersCalloutsToAddOrUpdate();
        Map<Long, List<Long>> bannersCalloutsToDelete = calloutsUpdateContainer.getBannersCalloutsToDelete();

        // banner images
        AddOrUpdateAndDeleteContainer<OldDynamicBanner> bannerImagesUpdateContainer =
                prepareBannerImagesUpdateContainer(appliedChanges);
        List<OldDynamicBanner> bannersToAddOrUpdateImages = bannerImagesUpdateContainer.getModelsToAddOrUpdate();
        List<OldDynamicBanner> bannersToDeleteImages = bannerImagesUpdateContainer.getModelsToDelete();

        // prepare banner displayHref to update
        AddOrUpdateAndDeleteContainer<OldDynamicBanner> displayHrefsUpdateContainer =
                bannerHrefRepository.prepareDisplayHrefsUpdateContainer(appliedChanges);
        Set<Long> displayHrefBidsToDelete = listToSet(displayHrefsUpdateContainer.getModelsToDelete(), OldDynamicBanner::getId);
        List<OldDynamicBanner> bannersToAddOrUpdateDisplayHrefs = displayHrefsUpdateContainer.getModelsToAddOrUpdate();

        JooqUpdateBuilder<BannersRecord, OldDynamicBanner> bannerUpdateBuilder =
                mapperProvider.createDynamicBannerUpdateBuilder(appliedChanges);

        final DSLContext context = dslContext(shard);
        context.transaction(ctx -> {
            DSL.using(ctx)
                    .update(BANNERS)
                    .set(bannerUpdateBuilder.getValues())
                    .where(BANNERS.BID.in(bannerUpdateBuilder.getChangedIds()))
                    .execute();

            // add or update banner displayHrefs
            bannerHrefRepository.addOrUpdateBannerDisplayHrefs(DSL.using(ctx), bannersToAddOrUpdateDisplayHrefs);

            // add or update banner images
            bannerImageRepository.addOrUpdateBannerImage(bannersToAddOrUpdateImages, DSL.using(ctx));

            // "delete" banner images - change status show to 'No'
            if (!bannersToDeleteImages.isEmpty()) {
                bannerImageRepository.deleteBannerImages(shard, bannersToDeleteImages);
            }

            // delete display hrefs
            if (!displayHrefBidsToDelete.isEmpty()) {
                bannerHrefRepository.deleteBannerDisplayHrefs(DSL.using(ctx), displayHrefBidsToDelete);
            }

            // привязка уточнений баннеру
            bannerAdditionsRepository.addOrUpdateBannerAdditions(DSL.using(ctx), bannersCalloutsToAddOrUpdate);

            // отвязка уточнений от баннера
            bannerAdditionsRepository.deleteBannersAdditions(bannersCalloutsToDelete, DSL.using(ctx));

            aggregatorDomainsService.updateAggregatorDomains(ctx, appliedChanges);
        });
    }

    /**
     * @param shard   шард
     * @param banners баннеры
     * @inheritDoc Создаются домены.
     */
    @Override
    public void beforeBannersAdd(int shard, Collection<OldDynamicBanner> banners) {
        domainService.addDomainsAndSetDomainIds(shard, banners);
    }

    @Override
    public void addBannersToDb(Collection<OldDynamicBanner> banners, DSLContext dslContext) {
        new InsertHelper<>(dslContext, BANNERS)
                .addAll(dynamicBannerMapper, banners)
                .executeIfRecordsAdded();
    }

    @Override
    public void afterBannersAdded(Collection<OldDynamicBanner> banners, DSLContext dslContext) {
        bannerImageRepository.addOrUpdateBannerImage(banners, dslContext);
        addBannerToBannerDisplayHrefsTable(banners, dslContext);
        addBannerToBannerAdditionTable(banners, dslContext);
        aggregatorDomainsService.updateAggregatorDomains(dslContext, banners);
    }

    private void addBannerToBannerDisplayHrefsTable(Collection<OldDynamicBanner> banners, DSLContext context) {
        Predicate<OldDynamicBanner> ifDisplayHrefNotNull = banner -> banner.getDisplayHref() != null;

        new InsertHelper<>(context, BANNER_DISPLAY_HREFS)
                .addAll(dynamicBannerMapper, banners, identity(), ifDisplayHrefNotNull)
                .executeIfRecordsAdded();
    }

    private void addBannerToBannerAdditionTable(Collection<OldDynamicBanner> banners, DSLContext context) {
        List<BannerAddition> bannersAllAdditionsList = banners
                .stream()
                .filter(b -> !isEmpty(b.getCalloutIds()))
                .map(OldBannerAdditionsRepository::extractCalloutAdditions)
                .flatMap(List::stream)
                .collect(Collectors.toList());
        bannerAdditionsRepository.addOrUpdateBannerAdditions(context, bannersAllAdditionsList);
    }

    @Override
    List<ModelProperty<? super OldDynamicBanner, ?>> getPropertiesForOpts() {
        List<ModelProperty<? super OldDynamicBanner, ?>> list = new ArrayList<>();
        list.add(OldBanner.GEO_FLAG);
        list.add(OldDynamicBanner.DISPLAY_HREF);
        return list;
    }

    @Override
    String optsToDb(OldDynamicBanner banner) {
        return DynamicBannerMapperProvider.bannerOptsToDb(banner);
    }

    @Override
    public OldDynamicBanner createBannerFromRecord(Record record) {
        OldDynamicBanner banner = dynamicBannerMapper.fromDb(record);
        if (bannerImageMapper.canReadAtLeastOneProperty(record.fields())) {
            OldBannerImage bannerImage = bannerImageMapper.fromDb(record);
            if (bannerImage.getId() != null) {
                banner.setBannerImage(bannerImage);
            }
        }
        return banner;
    }

    @Override
    public Stream<OldDynamicBanner> createBannersFromRecordsWithAdditionsAttached(int shard, Collection<Record> records) {
        List<OldDynamicBanner> banners = records.stream().map(this::createBannerFromRecord).collect(toList());

        //достаем уточнения
        Set<Long> bannerIds = listToSet(banners, ModelWithId::getId);
        Map<Long, List<Long>> bannerIdToCalloutIds = bannerAdditionsRepository.getCalloutIdsByBannerIds(shard, bannerIds);
        return banners.stream().peek(banner ->
                banner.withCalloutIds(defaultIfNull(bannerIdToCalloutIds.get(banner.getId()), emptyList())));
    }

}
