package ru.yandex.direct.core.entity.banner.repository.old.type;

import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.stream.Stream;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.DSLContext;
import org.jooq.Field;
import org.jooq.Record;
import org.jooq.impl.DSL;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.banner.model.old.Image;
import ru.yandex.direct.core.entity.banner.model.old.OldBannerButton;
import ru.yandex.direct.core.entity.banner.model.old.OldBannerCreative;
import ru.yandex.direct.core.entity.banner.model.old.OldBannerSubtypeEnum;
import ru.yandex.direct.core.entity.banner.model.old.OldBannerType;
import ru.yandex.direct.core.entity.banner.model.old.OldBannerWithButton;
import ru.yandex.direct.core.entity.banner.model.old.OldBannerWithImage;
import ru.yandex.direct.core.entity.banner.model.old.OldImageBanner;
import ru.yandex.direct.core.entity.banner.model.old.OldImageCreativeBanner;
import ru.yandex.direct.core.entity.banner.model.old.OldImageHashBanner;
import ru.yandex.direct.core.entity.banner.model.old.StatusImageModerate;
import ru.yandex.direct.core.entity.banner.old.BannerBeanUtil;
import ru.yandex.direct.core.entity.banner.repository.old.OldBannerButtonsRepository;
import ru.yandex.direct.core.entity.banner.repository.old.OldBannerLogosRepository;
import ru.yandex.direct.core.entity.banner.repository.old.OldImageRepository;
import ru.yandex.direct.core.entity.banner.repository.old.container.InsertUpdateDeleteContainer;
import ru.yandex.direct.core.entity.banner.turbolanding.model.OldBannerTurboLanding;
import ru.yandex.direct.core.entity.banner.turbolanding.model.OldBannerTurboLandingParams;
import ru.yandex.direct.core.entity.domain.service.DomainService;
import ru.yandex.direct.core.entity.domain.service.OldBannerAggregatorDomainsService;
import ru.yandex.direct.dbschema.ppc.enums.PhrasesAdgroupType;
import ru.yandex.direct.dbschema.ppc.tables.Banners;
import ru.yandex.direct.dbschema.ppc.tables.Phrases;
import ru.yandex.direct.dbschema.ppc.tables.records.BannersRecord;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;
import ru.yandex.direct.jooqmapperhelper.JooqUpdateBuilder;
import ru.yandex.direct.model.AppliedChanges;

import static com.google.common.base.Preconditions.checkNotNull;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toSet;
import static ru.yandex.direct.core.entity.banner.repository.old.mapper.BannerCreativeMapperProvider.getBannerCreativeMapper;
import static ru.yandex.direct.core.entity.banner.repository.old.mapper.ImageBannerMapperProvider.createImageBannerUpdateBuilder;
import static ru.yandex.direct.core.entity.banner.repository.old.mapper.ImageBannerMapperProvider.getImageCreativeBannerMapper;
import static ru.yandex.direct.core.entity.banner.repository.old.mapper.ImageBannerMapperProvider.getImageHashBannerMapper;
import static ru.yandex.direct.core.entity.banner.repository.old.mapper.ImageMapperProvider.getImageMapper;
import static ru.yandex.direct.core.entity.banner.repository.old.mapper.turbolanding.BannerTurboLandingMapperProvider.getBannerTurboLandingMapper;
import static ru.yandex.direct.core.entity.banner.repository.old.mapper.turbolanding.BannerTurboLandingParamsMapperProvider.getBannerTurboLandingParamsMapper;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNERS;
import static ru.yandex.direct.dbschema.ppc.Tables.IMAGES;
import static ru.yandex.direct.dbschema.ppc.Tables.PHRASES;
import static ru.yandex.direct.multitype.typesupport.TypeFilteringUtils.filterAppliedChangesForModelClass;
import static ru.yandex.direct.multitype.typesupport.TypeFilteringUtils.filterModelsOfClass;
import static ru.yandex.direct.utils.FunctionalUtils.filterAndMapList;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;

/**
 * Реализация {@link OldBannerRepositoryTypeSupport} для графических объявлений.
 */
@Component
@ParametersAreNonnullByDefault
@Deprecated
class OldBannerRepositoryImageTypeSupport extends OldAbstractBannerRepositoryTypeSupport<OldImageBanner> {

    private final OldImageRepository imageRepository;
    private final DomainService domainService;
    private final OldBannerWithCreativeSupport bannerWithCreativeSupport;
    private final OldBannerWithTurboLandingSupport bannerWithTurboLandingSupport;
    private final OldBannerAggregatorDomainsService aggregatorDomainsService;
    private final OldBannerWithOrganizationSupport bannerWithOrganizationSupport;
    private final OldBannerLogosRepository<OldImageBanner> bannerLogosRepository;
    private final OldBannerButtonsRepository bannerButtonsRepository;

    private final JooqMapperWithSupplier<Image> imageMapper = getImageMapper();
    private final JooqMapperWithSupplier<OldImageHashBanner> imageHashBannerMapper = getImageHashBannerMapper();
    private final JooqMapperWithSupplier<OldImageCreativeBanner> imageCreativeBannerMapper =
            getImageCreativeBannerMapper();
    private final JooqMapperWithSupplier<OldBannerCreative> bannerCreativeMapper = getBannerCreativeMapper();
    private final JooqMapperWithSupplier<OldBannerTurboLanding> bannerTurboLandingMapper =
            getBannerTurboLandingMapper();
    private final JooqMapperWithSupplier<OldBannerTurboLandingParams> bannerTurboLandingParamsMapper =
            getBannerTurboLandingParamsMapper();


    private final Collection<Field<?>> allFields;

    @Autowired
    OldBannerRepositoryImageTypeSupport(
            DslContextProvider dslContextProvider,
            ShardHelper shardHelper,
            OldImageRepository imageRepository,
            DomainService domainService,
            OldBannerWithCreativeSupport bannerWithCreativeSupport,
            OldBannerWithTurboLandingSupport bannerWithTurboLandingSupport,
            OldBannerAggregatorDomainsService aggregatorDomainsService,
            OldBannerWithOrganizationSupport bannerWithOrganizationSupport,
            OldBannerButtonsRepository bannerButtonsRepository) {
        super(dslContextProvider, shardHelper);

        this.imageRepository = imageRepository;
        this.domainService = domainService;
        this.bannerWithCreativeSupport = bannerWithCreativeSupport;
        this.bannerWithTurboLandingSupport = bannerWithTurboLandingSupport;
        this.aggregatorDomainsService = aggregatorDomainsService;
        this.bannerWithOrganizationSupport = bannerWithOrganizationSupport;
        this.bannerLogosRepository = new OldBannerLogosRepository<>(dslContextProvider, OldImageHashBanner::new);
        this.bannerButtonsRepository = bannerButtonsRepository;

        allFields = Stream.of(imageCreativeBannerMapper.getFieldsToRead(),
                imageHashBannerMapper.getFieldsToRead(),
                imageMapper.getFieldsToRead(),
                bannerCreativeMapper.getFieldsToRead(),
                bannerTurboLandingMapper.getFieldsToRead(),
                bannerTurboLandingParamsMapper.getFieldsToRead(),
                Collections.singletonList(Phrases.PHRASES.ADGROUP_TYPE))
                .flatMap(Collection::stream)
                .collect(toSet());
    }

    @Override
    public OldBannerType getType() {
        return OldBannerType.IMAGE_AD;
    }

    @Override
    public Collection<Field<?>> getAllFields() {
        return allFields;
    }

    @Override
    public void update(int shard, Collection<AppliedChanges<OldImageBanner>> appliedChanges) {
        domainService.addNewDomainsAndUpdateDomainIds(shard, appliedChanges);

        // images
        Collection<AppliedChanges<OldImageHashBanner>> imageHashBannersChanges =
                filterAppliedChangesForModelClass(appliedChanges, OldImageHashBanner.class);

        List<OldImageHashBanner> bannersToAddOrUpdateImages = imageHashBannersChanges.stream()
                .filter(ac -> !ac.deleted(OldImageHashBanner.IMAGE) && isImageChanged(ac))
                .map(AppliedChanges::getModel)
                .collect(toList());

        // creatives
        Collection<AppliedChanges<OldImageCreativeBanner>> imageCreativeBannersChanges =
                filterAppliedChangesForModelClass(appliedChanges, OldImageCreativeBanner.class);

        InsertUpdateDeleteContainer<OldBannerCreative> bannerCreativesUpdateContainer =
                bannerWithCreativeSupport.prepareCreativesUpdateContainer(imageCreativeBannersChanges);
        Collection<OldBannerCreative> creativesToInsert = bannerCreativesUpdateContainer.getModelsToInsert();
        Collection<OldBannerCreative> creativesToUpdate = bannerCreativesUpdateContainer.getModelsToUpdate();
        Collection<OldBannerCreative> creativesToDelete = bannerCreativesUpdateContainer.getModelsToDelete();

        JooqUpdateBuilder<BannersRecord, OldImageBanner> bannerUpdateBuilder =
                createImageBannerUpdateBuilder(appliedChanges);

        dslContext(shard).transaction(ctx -> {
            DSL.using(ctx)
                    .update(BANNERS)
                    .set(bannerUpdateBuilder.getValues())
                    .where(Banners.BANNERS.BID.in(bannerUpdateBuilder.getChangedIds()))
                    .execute();

            // images
            imageRepository.addOrUpdateImage(bannersToAddOrUpdateImages, DSL.using(ctx));

            // creatives
            bannerWithCreativeSupport.insertCreatives(ctx, creativesToInsert);
            bannerWithCreativeSupport.updateCreatives(ctx, creativesToUpdate);
            bannerWithCreativeSupport.deleteCreatives(ctx, creativesToDelete);

            bannerWithTurboLandingSupport.updateTurbolandings(appliedChanges, ctx);
            bannerWithTurboLandingSupport.updateTurbolandingParams(appliedChanges, ctx);

            aggregatorDomainsService.updateAggregatorDomains(ctx, appliedChanges);

            bannerWithOrganizationSupport.updateBannerPermalinks(DSL.using(ctx), appliedChanges);
        });
    }

    @Override
    public void add(int shard, Collection<OldImageBanner> banners) {
        // prepare
        domainService.addDomainsAndSetDomainIds(shard, banners);

        Collection<OldImageHashBanner> imageHashBanners = filterImageHashBanners(banners);
        Collection<OldImageCreativeBanner> imageCreativeBanners = filterImageCreativeBanners(banners);
        bannerWithCreativeSupport.addBannerCreatives(imageCreativeBanners, shard);
        DSLContext dslContext = dslContext(shard);

        // write
        InsertHelper<BannersRecord> bannersInsertHelper = new InsertHelper<>(dslContext, BANNERS);
        bannersInsertHelper.addAll(imageHashBannerMapper, imageHashBanners);
        bannersInsertHelper.addAll(imageCreativeBannerMapper, imageCreativeBanners);

        bannersInsertHelper.executeIfRecordsAdded();

        aggregatorDomainsService.updateAggregatorDomains(dslContext, banners);

        // update related entities
        imageRepository.addOrUpdateImage(imageHashBanners, dslContext);
        bannerWithTurboLandingSupport.updateAfterBannersAdded(banners, dslContext);
        bannerWithOrganizationSupport.linkOrganizationsToBanners(dslContext, banners);
        bannerWithOrganizationSupport.addOrganizationCountersToBannersCampaigns(dslContext, banners);

        addBannerToBannerLogosTable(banners, dslContext);
        addBannerToBannerButtonsTable(banners, dslContext);
    }

    private Collection<OldImageCreativeBanner> filterImageCreativeBanners(Collection<OldImageBanner> banners) {
        return filterModelsOfClass(banners, OldImageCreativeBanner.class);
    }

    private Collection<OldImageHashBanner> filterImageHashBanners(Collection<OldImageBanner> banners) {
        return filterModelsOfClass(banners, OldImageHashBanner.class);
    }

    private void addBannerToBannerLogosTable(Collection<OldImageBanner> banners, DSLContext context) {
        bannerLogosRepository.update(context, filterList(banners, b -> b.getLogoImageHash() != null));
    }

    private void addBannerToBannerButtonsTable(Collection<OldImageBanner> banners, DSLContext context) {
        List<OldBannerButton> buttons = filterAndMapList(banners, b -> b.getButtonAction() != null, this::toBannerButton);
        bannerButtonsRepository.update(context, buttons);
    }

    private OldBannerButton toBannerButton(OldBannerWithButton banner) {
        return new OldBannerButton()
                .withBannerId(banner.getId())
                .withAction(banner.getButtonAction())
                .withCaption(banner.getButtonCaption())
                .withHref(banner.getButtonHref())
                .withStatusModerate(banner.getButtonStatusModerate());
    }

    /**
     * @param record запись в БД, содержащая общие поля присущие любому объявлению,
     *               а также {@link ru.yandex.direct.dbschema.ppc.tables.Images#IMAGE_ID}.
     * @return Инстанс {@link OldImageHashBanner} или {@link OldImageCreativeBanner}.
     * @inheritDoc
     */
    @Override
    public OldImageBanner createBannerFromRecord(Record record) {
        boolean hasImageId = record.get(IMAGES.IMAGE_ID) != null;
        OldImageBanner banner = hasImageId ? imageHashBannerFromDb(record) : imageCreativeBannerFromDb(record);

        OldBannerTurboLandingParams turboLandingParams = bannerTurboLandingParamsMapper.fromDb(record);
        if (turboLandingParams.getBannerId() != null) {
            banner.withTurboLandingParams(turboLandingParams);
        }
        return banner;
    }

    private OldImageCreativeBanner imageCreativeBannerFromDb(Record record) {
        PhrasesAdgroupType adgroupType = record.get(PHRASES.ADGROUP_TYPE);
        checkNotNull(adgroupType);
        OldBannerSubtypeEnum subtype = PhrasesAdgroupType.base.equals(adgroupType) ? OldBannerSubtypeEnum.TEXT_AD_BUILDER_AD
                : OldBannerSubtypeEnum.MOBILE_APP_AD_BUILDER_AD;
        return imageCreativeBannerMapper.fromDb(record).withSubType(subtype);
    }

    private OldImageHashBanner imageHashBannerFromDb(Record record) {
        PhrasesAdgroupType adgroupType = record.get(PHRASES.ADGROUP_TYPE);
        checkNotNull(adgroupType);
        OldBannerSubtypeEnum subtype = PhrasesAdgroupType.base.equals(adgroupType) ? OldBannerSubtypeEnum.TEXT_IMAGE_AD
                : OldBannerSubtypeEnum.MOBILE_APP_IMAGE_AD;

        OldImageHashBanner banner = imageHashBannerMapper.fromDb(record).withSubType(subtype);
        Image image = imageMapper.fromDb(record);
        if (image.getId() != null) {
            banner.setImage(image);
        }
        return banner;
    }

    @Override
    public Stream<OldImageBanner> createBannersFromRecordsWithAdditionsAttached(int shard, Collection<Record> records) {
        return records.stream().map(this::createBannerFromRecord);
    }

    public static boolean isImageChanged(@Nonnull AppliedChanges<? extends OldBannerWithImage> ac) {
        return isImageChanged(ac.getOldValue(OldBannerWithImage.IMAGE), ac.getNewValue(OldBannerWithImage.IMAGE));
    }


    public static boolean isImageChanged(@Nullable Image oldImage, @Nullable Image newImage) {
        //в новых баннерах см. BannerWithImageUtils
        boolean imageChanged;
        if (oldImage == null && newImage == null) {
            imageChanged = false;
        } else if (oldImage != null && newImage != null) {
            imageChanged = !newImage.getImageHash().equals(oldImage.getImageHash());
            imageChanged = imageChanged || !Objects.equals(newImage.getImageText(), oldImage.getImageText());
            imageChanged = imageChanged || !Objects.equals(newImage.getDisclaimerText(),
                    oldImage.getDisclaimerText());
            imageChanged = imageChanged ||
                    (newImage.getStatusModerate() != null
                            && newImage.getStatusModerate() != oldImage.getStatusModerate());
        } else {
            imageChanged = true;
        }
        return imageChanged;
    }

    public static Image changeImageStatusModerate(Image image, StatusImageModerate newStatus) {
        return BannerBeanUtil.cloneImage(image).withStatusModerate(newStatus);
    }
}
