package ru.yandex.direct.core.entity.banner.repository.old.type;

import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.stream.Stream;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.DSLContext;
import org.jooq.Field;
import org.jooq.Record;
import org.jooq.impl.DSL;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.banner.model.old.Image;
import ru.yandex.direct.core.entity.banner.model.old.OldBannerType;
import ru.yandex.direct.core.entity.banner.model.old.OldBannerWithImage;
import ru.yandex.direct.core.entity.banner.model.old.OldMcBanner;
import ru.yandex.direct.core.entity.banner.model.old.StatusImageModerate;
import ru.yandex.direct.core.entity.banner.old.BannerBeanUtil;
import ru.yandex.direct.core.entity.banner.repository.old.OldImageRepository;
import ru.yandex.direct.core.entity.domain.service.DomainService;
import ru.yandex.direct.core.entity.domain.service.OldBannerAggregatorDomainsService;
import ru.yandex.direct.dbschema.ppc.enums.PhrasesAdgroupType;
import ru.yandex.direct.dbschema.ppc.tables.Banners;
import ru.yandex.direct.dbschema.ppc.tables.Phrases;
import ru.yandex.direct.dbschema.ppc.tables.records.BannersRecord;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;
import ru.yandex.direct.jooqmapperhelper.JooqUpdateBuilder;
import ru.yandex.direct.model.AppliedChanges;

import static com.google.common.base.Preconditions.checkNotNull;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toSet;
import static ru.yandex.direct.core.entity.banner.repository.old.mapper.ImageMapperProvider.getImageMapper;
import static ru.yandex.direct.core.entity.banner.repository.old.mapper.McBannerMapperProvider.createMcBannerUpdateBuilder;
import static ru.yandex.direct.core.entity.banner.repository.old.mapper.McBannerMapperProvider.getMcBannerMapper;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNERS;
import static ru.yandex.direct.dbschema.ppc.Tables.PHRASES;
import static ru.yandex.direct.multitype.typesupport.TypeFilteringUtils.filterAppliedChangesForModelClass;

/**
 * Реализация {@link OldBannerRepositoryTypeSupport} для графических объявлений на поиске.
 */
@Component
@ParametersAreNonnullByDefault
@Deprecated
public class OldBannerRepositoryMcBannerTypeSupport extends OldAbstractBannerRepositoryTypeSupport<OldMcBanner> {
    private final OldImageRepository imageRepository;
    private final DomainService domainService;
    private final OldBannerAggregatorDomainsService aggregatorDomainsService;
    private final JooqMapperWithSupplier<Image> imageMapper = getImageMapper();
    private final JooqMapperWithSupplier<OldMcBanner> mcBannerMapper = getMcBannerMapper();
    private final Collection<Field<?>> allFields;

    @Autowired
    OldBannerRepositoryMcBannerTypeSupport(
            DslContextProvider dslContextProvider,
            ShardHelper shardHelper,
            OldImageRepository imageRepository,
            DomainService domainService,
            OldBannerAggregatorDomainsService aggregatorDomainsService) {
        super(dslContextProvider, shardHelper);
        this.imageRepository = imageRepository;
        this.domainService = domainService;
        this.aggregatorDomainsService = aggregatorDomainsService;
        allFields = Stream.of(mcBannerMapper.getFieldsToRead(),
                imageMapper.getFieldsToRead(),
                Collections.singletonList(Phrases.PHRASES.ADGROUP_TYPE))
                .flatMap(Collection::stream)
                .collect(toSet());
    }

    @Override
    public OldBannerType getType() {
        return OldBannerType.MCBANNER;
    }

    @Override
    public Collection<Field<?>> getAllFields() {
        return allFields;
    }

    @Override
    public void update(int shard, Collection<AppliedChanges<OldMcBanner>> appliedChanges) {
        domainService.addNewDomainsAndUpdateDomainIds(shard, appliedChanges);
        // images
        Collection<AppliedChanges<OldMcBanner>> mcBannersChanges =
                filterAppliedChangesForModelClass(appliedChanges, OldMcBanner.class);
        List<OldMcBanner> bannersToAddOrUpdateImages = mcBannersChanges.stream()
                .filter(ac -> !ac.deleted(OldMcBanner.IMAGE) && isImageChanged(ac))
                .map(AppliedChanges::getModel)
                .collect(toList());
        JooqUpdateBuilder<BannersRecord, OldMcBanner> bannerUpdateBuilder = createMcBannerUpdateBuilder(appliedChanges);
        dslContext(shard).transaction(ctx -> {
            DSL.using(ctx)
                    .update(BANNERS)
                    .set(bannerUpdateBuilder.getValues())
                    .where(Banners.BANNERS.BID.in(bannerUpdateBuilder.getChangedIds()))
                    .execute();
            // images
            imageRepository.addOrUpdateImage(bannersToAddOrUpdateImages, DSL.using(ctx));

            aggregatorDomainsService.updateAggregatorDomains(ctx, appliedChanges);
        });
    }

    @Override
    public void add(int shard, Collection<OldMcBanner> banners) {
        // prepare
        domainService.addDomainsAndSetDomainIds(shard, banners);
        DSLContext dslContext = dslContext(shard);
        // write
        InsertHelper<BannersRecord> bannersInsertHelper = new InsertHelper<>(dslContext, BANNERS);
        bannersInsertHelper.addAll(mcBannerMapper, banners);
        bannersInsertHelper.executeIfRecordsAdded();
        // update related entities
        aggregatorDomainsService.updateAggregatorDomains(dslContext, banners);
        imageRepository.addOrUpdateImage(banners, dslContext);
    }

    /**
     * @param record запись в БД, содержащая общие поля присущие любому объявлению,
     *               а также {@link ru.yandex.direct.dbschema.ppc.tables.Images#IMAGE_ID}.
     * @return Инстанс {@link OldMcBanner}.
     * @inheritDoc
     */
    @Override
    public OldMcBanner createBannerFromRecord(Record record) {
        PhrasesAdgroupType adgroupType = record.get(PHRASES.ADGROUP_TYPE);
        checkNotNull(adgroupType);
        OldMcBanner banner = mcBannerMapper.fromDb(record);
        Image image = imageMapper.fromDb(record);
        if (image.getId() != null) {
            banner.setImage(image);
        }
        return banner;
    }

    @Override
    public Stream<OldMcBanner> createBannersFromRecordsWithAdditionsAttached(int shard, Collection<Record> records) {
        return records.stream().map(this::createBannerFromRecord);
    }

    public static boolean isImageChanged(@Nonnull AppliedChanges<? extends OldBannerWithImage> ac) {
        return isImageChanged(ac.getOldValue(OldBannerWithImage.IMAGE), ac.getNewValue(OldBannerWithImage.IMAGE));
    }


    public static boolean isImageChanged(@Nullable Image oldImage, @Nullable Image newImage) {
        //в новых баннерах см. BannerWithImageUtils
        boolean imageChanged;
        if (oldImage == null && newImage == null) {
            imageChanged = false;
        } else if (oldImage != null && newImage != null) {
            imageChanged = !newImage.getImageHash().equals(oldImage.getImageHash());
            imageChanged = imageChanged || !Objects.equals(newImage.getImageText(), oldImage.getImageText());
            imageChanged = imageChanged || !Objects.equals(newImage.getDisclaimerText(),
                    oldImage.getDisclaimerText());
            imageChanged = imageChanged ||
                    (newImage.getStatusModerate() != null
                            && newImage.getStatusModerate() != oldImage.getStatusModerate());
        } else {
            imageChanged = true;
        }
        return imageChanged;
    }

    public static Image changeImageStatusModerate(Image image, StatusImageModerate newStatus) {
        return BannerBeanUtil.cloneImage(image).withStatusModerate(newStatus);
    }
}
