package ru.yandex.direct.core.entity.banner.repository.old.type;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Predicate;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.jooq.Configuration;
import org.jooq.Field;
import org.jooq.impl.DSL;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.banner.model.old.OldBannerCreative;
import ru.yandex.direct.core.entity.banner.model.old.OldBannerCreativeStatusModerate;
import ru.yandex.direct.core.entity.banner.model.old.OldBannerWithCreative;
import ru.yandex.direct.core.entity.banner.repository.old.OldBannerCreativeRepository;
import ru.yandex.direct.core.entity.banner.repository.old.container.InsertUpdateDeleteContainer;
import ru.yandex.direct.dbschema.ppc.enums.BannersPerformanceStatusmoderate;
import ru.yandex.direct.dbschema.ppc.tables.BannersPerformance;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.Model;
import ru.yandex.direct.model.ModelProperty;

import static java.util.stream.Collectors.toList;
import static ru.yandex.direct.core.entity.banner.repository.old.mapper.BannerCreativeMapperProvider.getBannerCreativeMapper;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNERS_PERFORMANCE;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Общая логика обработки баннеров с креативами в операциях репозитория.
 * Делегат для наследников {@link OldBannerRepositoryTypeSupport},
 * работающих с объявлениями, которые могут содержать креатив.
 *
 * @see OldBannerRepositoryTextTypeSupport
 * @see OldBannerRepositoryImageTypeSupport
 * @see OldBannerRepositoryCpmBannerTypeSupport
 * @see OldBannerRepositoryCpmOutdoorTypeSupport
 * @see OldBannerRepositoryCpmIndoorTypeSupport
 * @see OldBannerRepositoryTypeSupport
 */
@Component
@ParametersAreNonnullByDefault
@Deprecated
class OldBannerWithCreativeSupport {

    private final ShardHelper shardHelper;
    private final OldBannerCreativeRepository bannerCreativeRepository;

    private final JooqMapperWithSupplier<OldBannerCreative> bannerCreativeMapper = getBannerCreativeMapper();

    @Autowired
    public OldBannerWithCreativeSupport(
            ShardHelper shardHelper,
            OldBannerCreativeRepository bannerCreativeRepository) {
        this.shardHelper = shardHelper;
        this.bannerCreativeRepository = bannerCreativeRepository;
    }


    /**
     * Сохраняет креативы в таблицу banners_performance
     *
     * @param banners список баннеров
     * @param shard   шард
     */
    <B extends OldBannerWithCreative> void addBannerCreatives(Collection<B> banners, int shard) {
        List<B> bannersWithCreatives = StreamEx.of(banners)
                .filter(b -> b.getCreativeId() != null)
                .toList();
        List<Long> bannerCreativeIds = shardHelper.generateBannerCreativeIds(bannersWithCreatives.size());
        List<OldBannerCreative> creatives = StreamEx.of(bannersWithCreatives)
                .zipWith(bannerCreativeIds.stream())
                .mapKeyValue((b, bcId) -> new OldBannerCreative()
                        .withId(bcId)
                        .withCreativeId(b.getCreativeId())
                        .withBannerId(b.getId())
                        .withAdGroupId(b.getAdGroupId())
                        .withCampaignId(b.getCampaignId())
                        .withStatusModerate(b.getCreativeStatusModerate()))
                .collect(toList());
        bannerCreativeRepository.addBannerCreatives(shard, creatives);
    }

    /**
     * Разделить баннеры на те, у которых требуется удалить креатив,
     * те, которым нужно добавить, и те, которым – обновить.
     *
     * @param appliedChangesCollection коллекция изменений, применяемых к баннерам.
     * @return DTO, содержащий три множества баннеров, {@link InsertUpdateDeleteContainer}.
     */
    <B extends OldBannerWithCreative> InsertUpdateDeleteContainer<OldBannerCreative> prepareCreativesUpdateContainer(
            Collection<AppliedChanges<B>> appliedChangesCollection) {
        final InsertUpdateDeleteContainer<OldBannerCreative>
                bannerCreativesUpdateContainer = new InsertUpdateDeleteContainer<>();

        mapToBannerCreative(appliedChangesCollection, ac -> isInsertingValue(ac, B.CREATIVE_ID))
                .forEach(bannerCreativesUpdateContainer::markForInsertion);
        mapToBannerCreative(appliedChangesCollection, ac -> isDeletingValue(ac, B.CREATIVE_ID))
                .forEach(bannerCreativesUpdateContainer::markForDeletion);

        mapToBannerCreative(appliedChangesCollection, this::isCreativeUpdated)
                .forEach(bannerCreativesUpdateContainer::markForUpdate);

        return bannerCreativesUpdateContainer;
    }

    private <B extends OldBannerWithCreative> boolean isCreativeUpdated(AppliedChanges<B> ac) {
        return (ac.changed(B.CREATIVE_ID) || ac.changed(B.CREATIVE_STATUS_MODERATE))
                && ac.getOldValue(B.CREATIVE_ID) != null
                && ac.getNewValue(B.CREATIVE_ID) != null;
    }

    /**
     * Построить мапу для обновления jooq-ом поля
     * {@link BannersPerformance#CREATIVE_ID}.
     *
     * @param creativesToUpdate коллекция баннер-креативов, содержащих {@link OldBannerCreative#getBannerId}
     * @return Соответствие ppc.banners.id -> ppc.banners_performance.id для
     * использования в jooq-овом update выражении.
     */
    Field<Long> constructCreativeIdFieldUpdate(Collection<OldBannerCreative> creativesToUpdate) {
        return DSL.choose(BANNERS_PERFORMANCE.BID)
                .mapValues(StreamEx.of(creativesToUpdate)
                        .toMap(OldBannerCreative::getBannerId, OldBannerCreative::getCreativeId));
    }


    /**
     * Построить мапу для обновления jooq-ом поля
     * {@link BannersPerformance#STATUS_MODERATE}.
     *
     * @param creativesToUpdate коллекция баннер-креативов, содержащих {@link OldBannerCreative#getBannerId}
     * @return Соответствие ppc.banners.id -> ppc.banners_performance.status_moderate для
     * использования в jooq-овом update выражении.
     */
    Field<BannersPerformanceStatusmoderate> constructCreativeStatusModerateFieldUpdate(
            Collection<OldBannerCreative> creativesToUpdate) {
        Map<Long, BannersPerformanceStatusmoderate> creativeStatusModerateToUpdate = StreamEx.of(creativesToUpdate)
                .filter(t -> t.getStatusModerate() != null)
                .toMap(OldBannerCreative::getBannerId,
                        bc -> OldBannerCreativeStatusModerate.toSource(bc.getStatusModerate()));
        if (creativeStatusModerateToUpdate.isEmpty()) {
            return BANNERS_PERFORMANCE.STATUS_MODERATE;
        }
        return DSL.choose(BANNERS_PERFORMANCE.BID)
                .mapValues(creativeStatusModerateToUpdate)
                .otherwise(BANNERS_PERFORMANCE.STATUS_MODERATE);
    }

    private <B extends OldBannerWithCreative> List<OldBannerCreative> mapToBannerCreative(
            Collection<AppliedChanges<B>> appliedChanges, Predicate<AppliedChanges<B>> filter) {
        return appliedChanges.stream()
                .filter(filter)
                .map(AppliedChanges::getModel)
                .map(b -> new OldBannerCreative()
                        .withBannerId(b.getId())
                        .withAdGroupId(b.getAdGroupId())
                        .withCampaignId(b.getCampaignId())
                        .withCreativeId(b.getCreativeId())
                        .withStatusModerate(b.getCreativeStatusModerate()))
                .collect(toList());
    }

    private static <B extends Model> boolean isDeletingValue(AppliedChanges<B> appliedChanges,
                                                             ModelProperty<? super B, ?> property) {
        return appliedChanges.getOldValue(property) != null && appliedChanges.getNewValue(property) == null;
    }

    private static <B extends Model> boolean isInsertingValue(AppliedChanges<B> appliedChanges,
                                                              ModelProperty<? super B, ?> property) {
        return appliedChanges.getOldValue(property) == null && appliedChanges.getNewValue(property) != null;
    }


    /**
     * Добавить креативы в таблицу banner_performance в транзакции
     *
     * @param ctx               конфигурация подключения для выполнения в транзакции
     * @param creativesToInsert креативы для вставки
     */
    void insertCreatives(Configuration ctx, Collection<OldBannerCreative> creativesToInsert) {
        List<Long> bannerCreativeIds = shardHelper.generateBannerCreativeIds(creativesToInsert.size());
        StreamEx.of(creativesToInsert).zipWith(bannerCreativeIds.stream())
                .forKeyValue(OldBannerCreative::setId);

        new InsertHelper<>(DSL.using(ctx), BannersPerformance.BANNERS_PERFORMANCE)
                .addAll(bannerCreativeMapper, creativesToInsert)
                .executeIfRecordsAdded();
    }

    /**
     * Обновить креативы в таблице banner_performance в транзакции
     *
     * @param ctx               конфигурация подключения для выполнения в транзакции
     * @param creativesToUpdate креативы для обновления
     */
    void updateCreatives(Configuration ctx, Collection<OldBannerCreative> creativesToUpdate) {
        Set<Long> bannerIdsOfUpdatedRecords = listToSet(creativesToUpdate, OldBannerCreative::getBannerId);

        Field<Long> creativeIdFieldUpdate = constructCreativeIdFieldUpdate(creativesToUpdate);
        Field<BannersPerformanceStatusmoderate> creativeStatusModerateByBannerIdField =
                constructCreativeStatusModerateFieldUpdate(creativesToUpdate);

        if (!bannerIdsOfUpdatedRecords.isEmpty()) {
            DSL.using(ctx).update(BANNERS_PERFORMANCE)
                    .set(BANNERS_PERFORMANCE.CREATIVE_ID, creativeIdFieldUpdate)
                    .set(BANNERS_PERFORMANCE.STATUS_MODERATE, creativeStatusModerateByBannerIdField)
                    .where(BANNERS_PERFORMANCE.BID.in(bannerIdsOfUpdatedRecords))
                    .execute();
        }
    }

    /**
     * Удалить креативы из таблицы banner_performance в транзакции
     *
     * @param ctx               конфигурация подключения для выполнения в транзакции
     * @param creativesToDelete креативы для удаления
     */
    void deleteCreatives(Configuration ctx, Collection<OldBannerCreative> creativesToDelete) {
        Set<Long> bannerIdsOfDeletedRecords = listToSet(creativesToDelete, OldBannerCreative::getBannerId);

        if (!bannerIdsOfDeletedRecords.isEmpty()) {
            DSL.using(ctx).delete(BANNERS_PERFORMANCE)
                    .where(BANNERS_PERFORMANCE.BID.in(mapList(creativesToDelete, OldBannerCreative::getBannerId)))
                    .execute();
        }
    }
}
