package ru.yandex.direct.core.entity.banner.repository.selection;

import java.util.ArrayList;
import java.util.Collection;
import java.util.EnumSet;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableMap;
import org.jooq.Condition;
import org.jooq.GroupField;
import org.jooq.Record;
import org.jooq.SelectGroupByStep;
import org.jooq.SelectHavingStep;
import org.jooq.SelectJoinStep;
import org.jooq.TableLike;

import ru.yandex.direct.core.entity.YesNo;
import ru.yandex.direct.core.entity.adgroup.model.ContentPromotionAdgroupType;
import ru.yandex.direct.core.entity.banner.container.AdsSelectionCriteria;
import ru.yandex.direct.core.entity.banner.model.ExtensionStatus;
import ru.yandex.direct.core.entity.banner.model.State;
import ru.yandex.direct.core.entity.banner.model.Status;
import ru.yandex.direct.dbschema.ppc.enums.BannerImagesStatusmoderate;
import ru.yandex.direct.dbschema.ppc.enums.BannerImagesStatusshow;
import ru.yandex.direct.dbschema.ppc.enums.BannersAdditionsAdditionsType;
import ru.yandex.direct.dbschema.ppc.enums.BannersBannerType;
import ru.yandex.direct.dbschema.ppc.enums.BannersPerformanceStatusmoderate;
import ru.yandex.direct.dbschema.ppc.enums.BannersPhoneflag;
import ru.yandex.direct.dbschema.ppc.enums.BannersStatusactive;
import ru.yandex.direct.dbschema.ppc.enums.BannersStatusarch;
import ru.yandex.direct.dbschema.ppc.enums.BannersStatusmoderate;
import ru.yandex.direct.dbschema.ppc.enums.BannersStatuspostmoderate;
import ru.yandex.direct.dbschema.ppc.enums.BannersStatusshow;
import ru.yandex.direct.dbschema.ppc.enums.BannersStatussitelinksmoderate;
import ru.yandex.direct.dbschema.ppc.enums.BannersType;
import ru.yandex.direct.dbschema.ppc.enums.CampOptionsStatusmetricacontrol;
import ru.yandex.direct.dbschema.ppc.enums.CampaignsArchived;
import ru.yandex.direct.dbschema.ppc.enums.CampaignsStatusactive;
import ru.yandex.direct.dbschema.ppc.enums.CampaignsStatusshow;
import ru.yandex.direct.dbschema.ppc.enums.ImagesStatusmoderate;

import static com.google.common.base.Preconditions.checkArgument;
import static java.util.Arrays.asList;
import static java.util.Collections.singletonList;
import static ru.yandex.direct.dbschema.ppc.Tables.ADDITIONS_ITEM_CALLOUTS;
import static ru.yandex.direct.dbschema.ppc.Tables.ADGROUPS_CONTENT_PROMOTION;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNERS_ADDITIONS;
import static ru.yandex.direct.dbschema.ppc.Tables.BANNERS_PERFORMANCE;
import static ru.yandex.direct.dbschema.ppc.Tables.BS_DEAD_DOMAINS;
import static ru.yandex.direct.dbschema.ppc.Tables.CAMP_OPTIONS;
import static ru.yandex.direct.dbschema.ppc.Tables.DOMAINS;
import static ru.yandex.direct.dbschema.ppc.Tables.IMAGES;
import static ru.yandex.direct.dbschema.ppc.tables.BannerImages.BANNER_IMAGES;
import static ru.yandex.direct.dbschema.ppc.tables.Banners.BANNERS;
import static ru.yandex.direct.dbschema.ppc.tables.Campaigns.CAMPAIGNS;
import static ru.yandex.direct.utils.FunctionalUtils.mapSet;

@ParametersAreNonnullByDefault
public class Converter {

    private static final Map<TableLike<?>, Condition> TABLES_LEFT_JOIN_CONDITIONS =
            ImmutableMap.<TableLike<?>, Condition>builder()
                    .put(IMAGES, IMAGES.BID.eq(BANNERS.BID))
                    .put(BANNER_IMAGES, BANNER_IMAGES.BID.eq(BANNERS.BID).and(BANNER_IMAGES.STATUS_SHOW.eq(
                            BannerImagesStatusshow.Yes))) // statusShow = No значит что картинка отвязана
                    .put(BANNERS_PERFORMANCE, BANNERS_PERFORMANCE.BID.eq(BANNERS.BID))
                    .put(DOMAINS, DOMAINS.DOMAIN.eq(BANNERS.DOMAIN))
                    .put(BS_DEAD_DOMAINS, BS_DEAD_DOMAINS.DOMAIN_ID.eq(DOMAINS.DOMAIN_ID))
                    //.put(BS_DEAD_DOMAINS, BS_DEAD_DOMAINS.DOMAIN_ID.eq(BANNERS.DOMAIN_ID))
                    .put(ADGROUPS_CONTENT_PROMOTION, ADGROUPS_CONTENT_PROMOTION.PID.eq(BANNERS.PID))
                    .build();

    private static final Map<TableLike<?>, Condition> TABLES_JOIN_CONDITIONS =
            ImmutableMap.<TableLike<?>, Condition>builder()
                    .put(CAMPAIGNS, CAMPAIGNS.CID.eq(BANNERS.CID))
                    .put(CAMP_OPTIONS, CAMP_OPTIONS.CID.eq(CAMPAIGNS.CID))
                    .put(BANNERS_ADDITIONS, BANNERS_ADDITIONS.BID.eq(BANNERS.BID)
                            .and(BANNERS_ADDITIONS.ADDITIONS_TYPE.eq(BannersAdditionsAdditionsType.callout)))
                    .put(ADDITIONS_ITEM_CALLOUTS,
                            ADDITIONS_ITEM_CALLOUTS.ADDITIONS_ITEM_ID.eq(BANNERS_ADDITIONS.ADDITIONS_ITEM_ID))
                    .build();

    private static final Map<ExtensionStatus, List<BannersPhoneflag>> VCARD_STATUSES =
            ImmutableMap.<ExtensionStatus, List<BannersPhoneflag>>builder()
                    .put(ExtensionStatus.ACCEPTED, singletonList(BannersPhoneflag.Yes))
                    .put(ExtensionStatus.DRAFT, singletonList(BannersPhoneflag.New))
                    .put(ExtensionStatus.MODERATION,
                            asList(BannersPhoneflag.Ready, BannersPhoneflag.Sending, BannersPhoneflag.Sent))
                    .put(ExtensionStatus.REJECTED, singletonList(BannersPhoneflag.No))
                    .build();

    private static final Map<ExtensionStatus, List<BannersStatussitelinksmoderate>> SITELINKS_STATUSES =
            ImmutableMap.<ExtensionStatus, List<BannersStatussitelinksmoderate>>builder()
                    .put(ExtensionStatus.ACCEPTED, singletonList(BannersStatussitelinksmoderate.Yes))
                    .put(ExtensionStatus.DRAFT, singletonList(BannersStatussitelinksmoderate.New))
                    .put(ExtensionStatus.MODERATION,
                            asList(BannersStatussitelinksmoderate.Ready, BannersStatussitelinksmoderate.Sending,
                                    BannersStatussitelinksmoderate.Sent))
                    .put(ExtensionStatus.REJECTED, singletonList(BannersStatussitelinksmoderate.No))
                    .build();

    private static final Map<ExtensionStatus, List<BannerImagesStatusmoderate>> IMAGES_STATUSES =
            ImmutableMap.<ExtensionStatus, List<BannerImagesStatusmoderate>>builder()
                    .put(ExtensionStatus.ACCEPTED, singletonList(BannerImagesStatusmoderate.Yes))
                    .put(ExtensionStatus.DRAFT, singletonList(BannerImagesStatusmoderate.New))
                    .put(ExtensionStatus.MODERATION,
                            asList(BannerImagesStatusmoderate.Ready, BannerImagesStatusmoderate.Sending,
                                    BannerImagesStatusmoderate.Sent))
                    .put(ExtensionStatus.REJECTED, singletonList(BannerImagesStatusmoderate.No))
                    .build();

    public static <R extends Record> SelectHavingStep<R> buildSelectConditionStep(
            SelectJoinStep<R> selectJoinStep, AdsSelectionCriteria selectionCriteria) {
        Set<Long> adIds = selectionCriteria.getAdIds();
        Set<Long> adGroupIds = selectionCriteria.getAdGroupIds();
        Set<Long> campaignIds = selectionCriteria.getCampaignIds();

        checkArgument(!adIds.isEmpty() || !adGroupIds.isEmpty() || !campaignIds.isEmpty(),
                "AdIds, AdGroupIds or CampaignIds must be specified!");

        Set<TableLike<?>> tables = new LinkedHashSet<>(); // the order of tables is important
        List<Condition> conditions = new ArrayList<>();
        List<GroupField> groupBy = new ArrayList<>();

        conditions.add(BANNERS.PID.notEqual(0L));

        if (!adIds.isEmpty()) {
            conditions.add(BANNERS.BID.in(adIds));
        }

        if (!adGroupIds.isEmpty()) {
            conditions.add(BANNERS.PID.in(adGroupIds));
        }

        if (!campaignIds.isEmpty()) {
            conditions.add(BANNERS.CID.in(campaignIds));
        }

        Set<BannersBannerType> types = selectionCriteria.getTypes();
        if (!types.isEmpty()) {
            conditions.add(BANNERS.BANNER_TYPE.in(types));
        }

        Set<State> states = selectionCriteria.getStates();
        if (!states.isEmpty() && states.size() < State.values().length) {
            // если нужны баннеры во всех возможных состояниях, то не усложняем запрос дополнительными условиями отбора
            TablesAndConditionContainer container = convertStates(states);
            tables.addAll(container.tables);
            conditions.add(container.condition);
        }

        Set<Status> statuses = selectionCriteria.getStatuses();
        if (!statuses.isEmpty() && statuses.size() < Status.values().length) {
            // если нужны баннеры со всеми возможными статусами, то не усложняем запрос дополнительными условиями отбора
            TablesAndConditionContainer container = convertStatuses(statuses);
            tables.addAll(container.tables);
            conditions.add(container.condition);
        }

        YesNo mobile = selectionCriteria.getMobile();
        if (mobile != null) {
            conditions.add(BANNERS.TYPE.eq(YesNo.YES.equals(mobile) ? BannersType.mobile : BannersType.desktop));
        }

        Set<Long> vcardIds = selectionCriteria.getVCardIds();
        if (!vcardIds.isEmpty()) {
            conditions.add(BANNERS.VCARD_ID.in(vcardIds));
        }

        Set<Long> sitelinkSetIds = selectionCriteria.getSitelinkSetIds();
        if (!sitelinkSetIds.isEmpty()) {
            conditions.add(BANNERS.SITELINKS_SET_ID.in(sitelinkSetIds));
        }

        Set<String> adImageHashes = selectionCriteria.getAdImageHashes();
        if (!adImageHashes.isEmpty()) {
            tables.add(IMAGES);
            tables.add(BANNER_IMAGES);
            conditions.add(IMAGES.IMAGE_HASH.in(adImageHashes).or(BANNER_IMAGES.IMAGE_HASH.in(adImageHashes)));
        }

        Set<ExtensionStatus> vcardStatuses = selectionCriteria.getVCardStatuses();
        if (!vcardStatuses.isEmpty() && vcardStatuses.size() < ExtensionStatus.values().length) {
            // если нужны визитки со всеми возможными статусами, то не усложняем запрос дополнительными условиями отбора
            conditions.add(BANNERS.PHONEFLAG
                    .in(convertExtensionStatuses(vcardStatuses, BannersPhoneflag.class, VCARD_STATUSES)));
        }

        Set<ExtensionStatus> sitelinksStatuses = selectionCriteria.getSitelinksStatuses();
        if (!sitelinksStatuses.isEmpty() && sitelinksStatuses.size() < ExtensionStatus.values().length) {
            // если нужны визитки со всеми возможными статусами, то не усложняем запрос дополнительными условиями отбора
            conditions.add(BANNERS.STATUS_SITELINKS_MODERATE
                    .in(convertExtensionStatuses(sitelinksStatuses, BannersStatussitelinksmoderate.class,
                            SITELINKS_STATUSES)));
        }

        Set<ExtensionStatus> adImageStatuses = selectionCriteria.getAdImageStatuses();
        if (!adImageStatuses.isEmpty() && adImageStatuses.size() < ExtensionStatus.values().length) {
            // если нужны визитки со всеми возможными статусами, то не усложняем запрос дополнительными условиями отбора
            tables.add(BANNER_IMAGES);
            conditions.add(BANNER_IMAGES.STATUS_MODERATE
                    .in(convertExtensionStatuses(adImageStatuses, BannerImagesStatusmoderate.class, IMAGES_STATUSES)));
        }

        Set<ContentPromotionAdgroupType> contentPromotionAdgroupTypes =
                selectionCriteria.getContentPromotionAdgroupTypes();
        if (!contentPromotionAdgroupTypes.isEmpty()) {
            tables.add(ADGROUPS_CONTENT_PROMOTION);
            conditions.add(ADGROUPS_CONTENT_PROMOTION.PID.isNull().or(
                    ADGROUPS_CONTENT_PROMOTION.CONTENT_PROMOTION_TYPE.in(
                            mapSet(contentPromotionAdgroupTypes, ContentPromotionAdgroupType::toSource))));
        }

        Set<Long> adExtensionIds = selectionCriteria.getAdExtensionIds();
        if (!adExtensionIds.isEmpty()) {
            tables.add(BANNERS_ADDITIONS);
            tables.add(ADDITIONS_ITEM_CALLOUTS);
            conditions.add(BANNERS_ADDITIONS.ADDITIONS_ITEM_ID.in(adExtensionIds));
            groupBy.add(BANNERS.BID);
        }

        for (TableLike<?> table : tables) {
            if (TABLES_LEFT_JOIN_CONDITIONS.containsKey(table)) {
                selectJoinStep = selectJoinStep.leftJoin(table).on(TABLES_LEFT_JOIN_CONDITIONS.get(table));
            } else {
                selectJoinStep = selectJoinStep.join(table).on(TABLES_JOIN_CONDITIONS.get(table));
            }
        }

        SelectGroupByStep<R> groupByStep = selectJoinStep.where(conditions);

        return !groupBy.isEmpty() ? groupByStep.groupBy(groupBy) : groupByStep;
    }

    private static TablesAndConditionContainer convertStates(Set<State> states) {
        checkArgument(!states.isEmpty(), "Set of ad states cannot be empty");

        Set<TableLike<?>> tables = new LinkedHashSet<>();
        List<Condition> conditions = new ArrayList<>();

        for (State state : states) {
            switch (state) {
                // NB - order of cases is important
                case ARCHIVED:
                    tables.add(CAMPAIGNS);

                    conditions.add(BANNERS.STATUS_ARCH.eq(BannersStatusarch.Yes)
                            .or(CAMPAIGNS.ARCHIVED.eq(CampaignsArchived.Yes)));

                    break;
                case SUSPENDED:
                    tables.add(CAMPAIGNS);

                    conditions.add(BANNERS.STATUS_ARCH.eq(BannersStatusarch.No)
                            .and(BANNERS.STATUS_SHOW.eq(BannersStatusshow.No))
                            .and(CAMPAIGNS.ARCHIVED.eq(CampaignsArchived.No)));

                    break;
                case OFF_BY_MONITORING:
                    tables.add(CAMPAIGNS);
                    tables.add(CAMP_OPTIONS);
                    tables.add(DOMAINS);
                    tables.add(BS_DEAD_DOMAINS);

                    conditions.add(BANNERS.STATUS_ARCH.eq(BannersStatusarch.No)
                            .and(BANNERS.STATUS_SHOW.eq(BannersStatusshow.Yes))
                            .and(CAMPAIGNS.ARCHIVED.eq(CampaignsArchived.No))
                            .and(CAMP_OPTIONS.STATUS_METRICA_CONTROL.eq(CampOptionsStatusmetricacontrol.Yes))
                            .and(BS_DEAD_DOMAINS.DOMAIN_ID.isNotNull()));

                    break;
                case ON:
                    tables.add(CAMPAIGNS);

                    conditions.add(BANNERS.STATUS_ACTIVE.eq(BannersStatusactive.Yes)
                            .and(BANNERS.STATUS_ARCH.eq(BannersStatusarch.No))
                            .and(BANNERS.STATUS_SHOW.eq(BannersStatusshow.Yes))
                            .and(CAMPAIGNS.STATUS_ACTIVE.eq(CampaignsStatusactive.Yes))
                            .and(CAMPAIGNS.ARCHIVED.eq(CampaignsArchived.No))
                            .and(CAMPAIGNS.STATUS_SHOW.eq(CampaignsStatusshow.Yes)));

                    break;
                case OFF:
                    tables.add(CAMPAIGNS);
                    tables.add(CAMP_OPTIONS);
                    tables.add(DOMAINS);
                    tables.add(BS_DEAD_DOMAINS);

                    Condition offNotByMonitoring =
                            CAMP_OPTIONS.STATUS_METRICA_CONTROL.eq(CampOptionsStatusmetricacontrol.No)
                                    .or(BS_DEAD_DOMAINS.DOMAIN_ID.isNull());

                    conditions.add(BANNERS.STATUS_ARCH.eq(BannersStatusarch.No)
                            .and(BANNERS.STATUS_SHOW.eq(BannersStatusshow.Yes))
                            .and(CAMPAIGNS.ARCHIVED.eq(CampaignsArchived.No))
                            .and(CAMPAIGNS.STATUS_SHOW.eq(CampaignsStatusshow.No))
                            .and(offNotByMonitoring));

                    conditions.add(BANNERS.STATUS_ACTIVE.eq(BannersStatusactive.No)
                            .and(BANNERS.STATUS_ARCH.eq(BannersStatusarch.No))
                            .and(BANNERS.STATUS_SHOW.eq(BannersStatusshow.Yes))
                            .and(CAMPAIGNS.ARCHIVED.eq(CampaignsArchived.No))
                            .and(offNotByMonitoring));

                    conditions.add(BANNERS.STATUS_ARCH.eq(BannersStatusarch.No)
                            .and(BANNERS.STATUS_SHOW.eq(BannersStatusshow.Yes))
                            .and(CAMPAIGNS.ARCHIVED.eq(CampaignsArchived.No))
                            .and(CAMPAIGNS.STATUS_ACTIVE.eq(CampaignsStatusactive.No))
                            .and(offNotByMonitoring));
                    break;
                default:
                    throw new IllegalArgumentException("Not supported ad state: " + state);
            }
        }

        Condition condition = conditions.stream().reduce(Condition::or).orElseThrow(
                () -> new IllegalArgumentException("Can not build conditions for selection ads by states!"));

        return new TablesAndConditionContainer(tables, condition);
    }

    /**
     * NB:
     * - считается что status_moderate = 'New' может быть только при создании банера,
     * в дальнейшем при изменении баннера либо подчиненного объекта устанавливается
     * status_moderate = 'Ready' у измененного объекта
     * - для ГО считается что для отправки в БК картинка должна быть промодерирована,
     * хотя сам баннер может быть на постмодерации
     * <p>
     * Так как статусы вычисляются последовательно, то при вычислении более
     * "поздних" статусов допустимо не рассматривать некоторые значения полей,
     * при которых статус был бы определен ранее (например, при проверке не является
     * ли PREACCEPTED конечным можно не рассматривать imagead_status_moderate = 'No',
     * т.к. в этом случае ранее был бы вычислен статус REJECTED).
     */
    private static TablesAndConditionContainer convertStatuses(Set<Status> statuses) {
        checkArgument(!statuses.isEmpty(), "Set of ad statuses cannot be empty");

        Set<TableLike<?>> tables = new LinkedHashSet<>();
        List<Condition> conditions = new ArrayList<>();

        tables.add(IMAGES);
        tables.add(BANNERS_PERFORMANCE);

        for (Status status : statuses) {
            switch (status) {
                // NB - order of cases is important
                case DRAFT:
                    conditions.add(BANNERS.STATUS_MODERATE.eq(BannersStatusmoderate.New));

                    conditions.add(BANNERS.BANNER_TYPE.eq(BannersBannerType.image_ad)
                            .and(IMAGES.STATUS_MODERATE.eq(ImagesStatusmoderate.New)));

                    conditions.add(BANNERS.BANNER_TYPE.eq(BannersBannerType.image_ad)
                            .and(BANNERS_PERFORMANCE.STATUS_MODERATE.eq(BannersPerformanceStatusmoderate.New)));

                    conditions.add(BANNERS.BANNER_TYPE.eq(BannersBannerType.cpm_banner)
                            .and(BANNERS_PERFORMANCE.STATUS_MODERATE.eq(BannersPerformanceStatusmoderate.New)));

                    break;
                case ACCEPTED:
                    conditions.add(BANNERS.BANNER_TYPE.ne(BannersBannerType.image_ad)
                            .and(BANNERS.BANNER_TYPE.ne(BannersBannerType.cpm_banner))
                            .and(BANNERS.STATUS_MODERATE.eq(BannersStatusmoderate.Yes))
                            .and(BANNERS.STATUS_POST_MODERATE.eq(BannersStatuspostmoderate.Yes)));

                    conditions.add(BANNERS.BANNER_TYPE.eq(BannersBannerType.image_ad)
                            .and(BANNERS.STATUS_MODERATE.eq(BannersStatusmoderate.Yes))
                            .and(BANNERS.STATUS_POST_MODERATE.eq(BannersStatuspostmoderate.Yes))
                            .and(IMAGES.STATUS_MODERATE.eq(ImagesStatusmoderate.Yes)));

                    conditions.add(BANNERS.BANNER_TYPE.eq(BannersBannerType.image_ad)
                            .and(BANNERS.STATUS_MODERATE.eq(BannersStatusmoderate.Yes))
                            // для объявления с канвасными креативами banners.statusPostModerate не имеет значения
                            .and(BANNERS_PERFORMANCE.STATUS_MODERATE.eq(BannersPerformanceStatusmoderate.Yes)));

                    conditions.add(BANNERS.BANNER_TYPE.eq(BannersBannerType.cpm_banner)
                            .and(BANNERS.STATUS_MODERATE.eq(BannersStatusmoderate.Yes))
                            .and(BANNERS_PERFORMANCE.STATUS_MODERATE.eq(BannersPerformanceStatusmoderate.Yes)));

                    break;
                case REJECTED:
                    conditions.add(BANNERS.STATUS_MODERATE.eq(BannersStatusmoderate.No));

                    conditions.add(BANNERS.BANNER_TYPE.eq(BannersBannerType.image_ad)
                            .and(IMAGES.STATUS_MODERATE.eq(ImagesStatusmoderate.No)));

                    conditions.add(BANNERS.BANNER_TYPE.eq(BannersBannerType.image_ad)
                            .and(BANNERS_PERFORMANCE.STATUS_MODERATE.eq(BannersPerformanceStatusmoderate.No)));

                    conditions.add(BANNERS.BANNER_TYPE.eq(BannersBannerType.cpm_banner)
                            .and(BANNERS_PERFORMANCE.STATUS_MODERATE.eq(BannersPerformanceStatusmoderate.No)));

                    break;
                case PREACCEPTED:
                    conditions.add(BANNERS.BANNER_TYPE.ne(BannersBannerType.image_ad)
                            .and(BANNERS.BANNER_TYPE.ne(BannersBannerType.cpm_banner))
                            .and(BANNERS.STATUS_MODERATE.in(BannersStatusmoderate.Sending, BannersStatusmoderate.Sent))
                            .and(BANNERS.STATUS_POST_MODERATE.eq(BannersStatuspostmoderate.Yes)));

                    conditions.add(BANNERS.BANNER_TYPE.eq(BannersBannerType.image_ad)
                            .and(BANNERS.STATUS_MODERATE.in(BannersStatusmoderate.Sending, BannersStatusmoderate.Sent))
                            .and(BANNERS.STATUS_POST_MODERATE.eq(BannersStatuspostmoderate.Yes))
                            .and(IMAGES.STATUS_MODERATE.eq(ImagesStatusmoderate.Yes)));

                    // для объявления с канвасными креативами такого статуса нет

                    break;
                case MODERATION:
                    conditions.add(BANNERS.BANNER_TYPE.ne(BannersBannerType.image_ad)
                            .and(BANNERS.BANNER_TYPE.ne(BannersBannerType.cpm_banner))
                            .and(BANNERS.STATUS_MODERATE.in(BannersStatusmoderate.Sending, BannersStatusmoderate.Sent,
                                    BannersStatusmoderate.Ready))
                            .and(BANNERS.STATUS_POST_MODERATE
                                    .in(BannersStatuspostmoderate.No, BannersStatuspostmoderate.Rejected)));

                    conditions.add(BANNERS.BANNER_TYPE.eq(BannersBannerType.image_ad)
                            .and(IMAGES.STATUS_MODERATE.in(ImagesStatusmoderate.Sending, ImagesStatusmoderate.Sent,
                                    ImagesStatusmoderate.Ready)));

                    conditions.add(BANNERS.BANNER_TYPE.eq(BannersBannerType.image_ad)
                            .and(BANNERS.STATUS_MODERATE.in(BannersStatusmoderate.Sending, BannersStatusmoderate.Sent,
                                    BannersStatusmoderate.Ready))
                            .and(BANNERS.STATUS_POST_MODERATE
                                    .in(BannersStatuspostmoderate.No, BannersStatuspostmoderate.Rejected))
                            .and(IMAGES.STATUS_MODERATE.ne(ImagesStatusmoderate.No)));

                    conditions.add(BANNERS.BANNER_TYPE.eq(BannersBannerType.image_ad)
                            .and(BANNERS.STATUS_MODERATE
                                    .ne(BannersStatusmoderate.No)) // New намерено не включено, т.к. в этом случае
                            // ранее будет вычислено значение DRAFT
                            .and(BANNERS_PERFORMANCE.STATUS_MODERATE
                                    .in(BannersPerformanceStatusmoderate.Sending, BannersPerformanceStatusmoderate.Sent,
                                            BannersPerformanceStatusmoderate.Ready)));

                    conditions.add(BANNERS.BANNER_TYPE.eq(BannersBannerType.image_ad)
                            .and(BANNERS.STATUS_MODERATE.in(BannersStatusmoderate.Sending, BannersStatusmoderate.Sent,
                                    BannersStatusmoderate.Ready))
                            .and(BANNERS_PERFORMANCE.STATUS_MODERATE.ne(BannersPerformanceStatusmoderate.No)));

                    conditions.add(BANNERS.BANNER_TYPE.eq(BannersBannerType.cpm_banner)
                            .and(BANNERS.STATUS_MODERATE
                                    .ne(BannersStatusmoderate.No)) // New намерено не включено, т.к. в этом случае
                            // ранее будет вычислено значение DRAFT
                            .and(BANNERS_PERFORMANCE.STATUS_MODERATE
                                    .in(BannersPerformanceStatusmoderate.Sending, BannersPerformanceStatusmoderate.Sent,
                                            BannersPerformanceStatusmoderate.Ready)));

                    conditions.add(BANNERS.BANNER_TYPE.eq(BannersBannerType.cpm_banner)
                            .and(BANNERS.STATUS_MODERATE.in(BannersStatusmoderate.Sending, BannersStatusmoderate.Sent,
                                    BannersStatusmoderate.Ready))
                            .and(BANNERS_PERFORMANCE.STATUS_MODERATE.ne(BannersPerformanceStatusmoderate.No)));

                    break;
            }
        }

        Condition condition = conditions.stream().reduce(Condition::or).orElseThrow(
                () -> new IllegalArgumentException("Can not build conditions for selection ads by statuses!"));

        return new TablesAndConditionContainer(tables, condition);
    }

    private static <P, R extends Enum<R>> Set<R> convertExtensionStatuses(Collection<P> values, Class<R> clazz,
                                                                          Map<P, List<R>> map) {
        checkArgument(!values.isEmpty(), "No extension statuses given!");

        Set<R> result = EnumSet.noneOf(clazz);
        for (P fromValue : values) {
            List<R> toValue = map.get(fromValue);

            if (toValue == null) {
                throw new IllegalArgumentException("Not supported value of extension status: " + fromValue);
            }

            result.addAll(toValue);
        }

        return result;
    }

    private static class TablesAndConditionContainer {
        private final Set<TableLike<?>> tables;
        private final Condition condition;

        TablesAndConditionContainer(Set<TableLike<?>> tables, Condition condition) {
            this.tables = tables;
            this.condition = condition;
        }
    }
}
