package ru.yandex.direct.core.entity.banner.repository.type;

import java.util.Collection;
import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.DSLContext;
import org.jooq.Field;
import org.jooq.JoinType;
import org.jooq.Record;
import org.jooq.TableField;

import ru.yandex.direct.core.entity.banner.container.BannerRepositoryContainer;
import ru.yandex.direct.core.entity.banner.model.Banner;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapper;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;
import ru.yandex.direct.jooqmapperhelper.UpdateHelper;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.multitype.entity.JoinQuery;

import static ru.yandex.direct.dbschema.ppc.Tables.BANNERS;
import static ru.yandex.direct.utils.FunctionalUtils.filterAndMapList;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;

/**
 * Абстрактный класс для реализации на базе него репозиторных тайп-саппортов конкретных ассетов.
 * <p>
 * Подходит для отношения один-к-одному к основной таблице banners.
 * <p>
 * Под капотом для вставки используется insert, для обновления - update.
 * Подходит для случая, когда поля базы хранятся непосредственно в баннере, без промежуточного контейнера.
 * <p>
 * Разбивает строки на 3 множества - добавить, обновить, удалить.
 * <p>
 * Альтернатива: {@link AbstractFlatRelatedEntityUpsertRepositoryTypeSupport}.
 * Преимущество: нет необходимости самостоятельно писать update на jooq.
 * Недостаток: более сложное условие обновления в методе isUpdateEntity,
 * легко ошибиться при первоначальном написании и добавлении новых полей.
 *
 * @param <B> - интерфейс
 * @param <R> - тип записи Jooq в базе
 */
@ParametersAreNonnullByDefault
public abstract class AbstractFlatRelatedEntityRepositoryTypeSupport
        <B extends Banner, R extends Record>
        extends AbstractBannerRepositoryTypeSupport<B> {

    private final TableField<R, Long> bannerIdField;
    private final JooqMapper<B> jooqMapper;

    protected AbstractFlatRelatedEntityRepositoryTypeSupport(DslContextProvider dslContextProvider,
                                                             TableField<R, Long> bannerIdField,
                                                             JooqMapper<B> jooqMapper) {
        super(dslContextProvider);
        this.bannerIdField = bannerIdField;
        this.jooqMapper = jooqMapper;
    }

    @Override
    public void updateAdditionTables(DSLContext context,
                                     BannerRepositoryContainer updateParameters,
                                     Collection<AppliedChanges<B>> appliedChanges) {
        List<B> entitiesForAdd = filterAndMapList(appliedChanges, this::isAddEntity, AppliedChanges::getModel);
        List<AppliedChanges<B>> entitiesForUpdate = filterList(appliedChanges, this::isUpdateEntity);
        List<AppliedChanges<B>> entitiesForDelete = filterList(appliedChanges, this::isDeleteEntity);

        // Стоит знать, что баннер может одновременно попасть в entitiesForDelete и entitiesForAdd, как правило
        // это происходит когда обновилось поле входящее в primary key.
        // Важно сначала удалить, потом добавлять, т.к. обычно мы удаляем по primary key не смотря на другие поля.
        // порядок выполнения важен
        if (entitiesForDelete.size() > 0) {
            deleteEntities(context, updateParameters, entitiesForDelete);
        }
        if (entitiesForUpdate.size() > 0) {
            updateEntities(context, updateParameters, entitiesForUpdate);
        }
        if (entitiesForAdd.size() > 0) {
            addEntities(context, updateParameters, entitiesForAdd);
        }
    }

    @Override
    public void insertToAdditionTables(DSLContext context,
                                       BannerRepositoryContainer parametersContainer,
                                       Collection<B> models) {
        Collection<B> entities = filterList(models, this::isAddEntity);
        if (entities.size() > 0) {
            addEntities(context, parametersContainer, entities);
        }
    }

    protected abstract boolean isAddEntity(B model);

    protected abstract boolean isAddEntity(AppliedChanges<B> appliedChange);

    protected abstract boolean isUpdateEntity(AppliedChanges<B> appliedChange);

    protected abstract boolean isDeleteEntity(AppliedChanges<B> appliedChange);

    /**
     * Обработчик баннеров, которые нужно добавить. Будет вызван только если коллекция баннеров не пустая.
     */
    protected void addEntities(DSLContext context, BannerRepositoryContainer updateParameters, Collection<B> banners) {
        new InsertHelper<>(context, bannerIdField.getTable())
                .addAll(getJooqMapper(), banners)
                .executeIfRecordsAdded();
    }

    /**
     * Обработчик баннеров, которые нужно обновить. Будет вызван только если коллекция баннеров не пустая.
     */
    protected void updateEntities(DSLContext context, BannerRepositoryContainer updateParameters,
                                  Collection<AppliedChanges<B>> appliedChanges) {
        new UpdateHelper<>(context, bannerIdField)
                .processUpdateAll(getJooqMapper(), appliedChanges)
                .execute();
    }

    /**
     * Обработчик баннеров, которые нужно удалить. Будет вызван только если коллекция баннеров не пустая.
     */
    protected void deleteEntities(DSLContext context, BannerRepositoryContainer updateParameters,
                                  Collection<AppliedChanges<B>> appliedChanges) {
        var bannerIds = listToSet(appliedChanges, ac -> ac.getModel().getId());
        context.delete(bannerIdField.getTable())
                .where(bannerIdField.in(bannerIds))
                .execute();
    }

    @Override
    public List<JoinQuery> joinQuery() {
        return List.of(new JoinQuery(bannerIdField.getTable(), JoinType.LEFT_OUTER_JOIN,
                bannerIdField.eq(BANNERS.BID)));
    }

    @Override
    public Collection<Field<?>> getFields() {
        return getJooqMapper().getFieldsToRead();
    }

    @Override
    public final <M extends B> void fillFromRecord(M model, Record record) {
        getJooqMapper().fromDb(record, model);
    }

    protected JooqMapper<B> getJooqMapper() {
        return jooqMapper;
    }

}
