package ru.yandex.direct.core.entity.banner.repository.type;

import java.util.Collection;
import java.util.function.Predicate;

import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.Field;
import org.jooq.Record;
import org.jooq.TableField;

import ru.yandex.direct.core.entity.banner.model.Banner;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.Model;
import ru.yandex.direct.model.ModelProperty;

/**
 * Абстрактный класс для реализации на базе него репозиторных тайп-саппортов конкретных ассетов.
 * <p>
 * Подходит для отношения один-к-одному к основной таблице banners.
 * <p>
 * Под капотом для вставки и обновления используется insert ... on duplicate key update.
 * Подходит для случая, когда поля из базы хранятся в баннере не на верхнем уровне, а в промежуточном контейнере.
 * <p>
 * При чтении баннера проверяет необходимость заполнения поля модели с помощью предиката isSelectedPropertyNotEmpty.
 * При добавлении и обновлении баннера проверяет необходимость создания или удаления записи в базе с помощью
 * того же предиката - isSelectedPropertyNotEmpty.
 * <p>
 * ВАЖНО: используйте структуру модели с вложенным контейнером только в том случае,
 * если он всегда будет обновляться целиком! Если же поля нужно обновлять по-отдельности, если объект будет содержать
 * служебные поля, то такой вариант организации приведет к проблемам и костылям!!!
 * В этом случае храните поля непосредственно в баннере, без промежуточных объектов, и используйте соответствующие
 * абстрактные репозиторные тайп-саппорты - AbstractFlatRelatedEntityUpsertRepositoryTypeSupport и
 * AbstractFlatRelatedEntityRepositoryTypeSupport.
 *
 * @param <B> - интерфейс
 * @param <R> - тип записи Jooq в базе
 * @param <P> - тип контейнера, который хранится непосредственно в баннере и содержит в себе полезные поля с данными
 */
@ParametersAreNonnullByDefault
public abstract class AbstractNestedRelatedEntityUpsertRepositoryTypeSupport
        <B extends Banner, R extends Record, P extends Model>
        extends AbstractRelatedEntityUpsertRepositoryTypeSupport<B, R> {

    private final ModelProperty<? super B, P> modelProperty;
    private final Predicate<P> isSelectedPropertyNotEmpty;
    private final JooqMapperWithSupplier<P> jooqMapper;

    public AbstractNestedRelatedEntityUpsertRepositoryTypeSupport(
            DslContextProvider dslContextProvider,
            TableField<R, Long> bannerIdField,
            JooqMapperWithSupplier<P> jooqMapper,
            ModelProperty<? super B, P> modelProperty,
            Predicate<P> isSelectedPropertyNotEmpty) {
        super(dslContextProvider, bannerIdField);
        this.jooqMapper = jooqMapper;
        this.modelProperty = modelProperty;
        this.isSelectedPropertyNotEmpty = isSelectedPropertyNotEmpty;
    }

    @Override
    protected boolean isAddEntity(B model) {
        return modelProperty.get(model) != null;
    }

    @Override
    protected boolean isDeleteEntity(AppliedChanges<B> ac) {
        return ac.deleted(modelProperty);
    }

    @Override
    protected boolean isUpsertEntity(AppliedChanges<B> ac) {
        return ac.changedAndNotDeleted(modelProperty);
    }

    @Override
    public <M extends B> void fillFromRecord(M model, Record record) {
        P nestedProperty = getJooqMapper().fromDb(record);
        if (isSelectedPropertyNotEmpty.test(nestedProperty)) {
            modelProperty.set(model, nestedProperty);
        }
    }

    @Override
    public Collection<Field<?>> getFields() {
        return getJooqMapper().getFieldsToRead();
    }

    protected JooqMapperWithSupplier<P> getJooqMapper() {
        return jooqMapper;
    }

}
