package ru.yandex.direct.core.entity.banner.repository.type;

import java.util.Collection;
import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.DSLContext;
import org.jooq.JoinType;
import org.jooq.Record;
import org.jooq.TableField;

import ru.yandex.direct.core.entity.banner.container.BannerRepositoryContainer;
import ru.yandex.direct.core.entity.banner.model.Banner;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.multitype.entity.JoinQuery;

import static ru.yandex.direct.dbschema.ppc.Tables.BANNERS;
import static ru.yandex.direct.utils.FunctionalUtils.filterAndMapList;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;

/**
 * Саппорт для отдельной таблицы с отношением один-к-одному к таблице banners по столбцу bannerId.
 * Разбивает строки на 2 множества - добавить или обновить, удалить.
 *
 * @param <B> - баннер
 * @param <R> - jooq класс таблицы в базе данных
 */
@ParametersAreNonnullByDefault
public abstract class AbstractRelatedEntityUpsertRepositoryTypeSupport
        <B extends Banner, R extends Record>
        extends AbstractBannerRepositoryTypeSupport<B> {

    private final TableField<R, Long> bannerIdField;

    protected AbstractRelatedEntityUpsertRepositoryTypeSupport(DslContextProvider dslContextProvider,
                                                               TableField<R, Long> bannerIdField) {
        super(dslContextProvider);
        this.bannerIdField = bannerIdField;
    }

    protected abstract void upsertEntity(DSLContext context, Collection<B> banners);

    protected abstract boolean isAddEntity(B model);

    protected abstract boolean isUpsertEntity(AppliedChanges<B> appliedChange);

    protected abstract boolean isDeleteEntity(AppliedChanges<B> appliedChange);

    @Override
    public void updateAdditionTables(DSLContext context, BannerRepositoryContainer updateParameters,
                                     Collection<AppliedChanges<B>> appliedChanges) {

        List<AppliedChanges<B>> entitiesForDelete = filterList(appliedChanges, this::isDeleteEntity);

        List<B> entitiesForAddOrUpdate =
                filterAndMapList(appliedChanges, this::isUpsertEntity, AppliedChanges::getModel);

        // Стоит знать, что баннер может одновременно попасть в entitiesForDelete и entitiesForAddOrUpdate, как правило
        // это происходит когда обновилось поле входящее в primary key.
        // Важно сначала удалить, потом добавлять, т.к. обычно мы удаляем по primary key не смотря на другие поля.
        if (!entitiesForDelete.isEmpty()) {
            deleteEntities(context, updateParameters, entitiesForDelete);
        }

        if (!entitiesForAddOrUpdate.isEmpty()) {
            upsertEntity(context, entitiesForAddOrUpdate);
        }
    }

    @Override
    public final void insertToAdditionTables(DSLContext context,
                                             BannerRepositoryContainer parametersContainer,
                                             Collection<B> models) {
        Collection<B> entities = filterList(models, this::isAddEntity);
        if (!entities.isEmpty()) {
            upsertEntity(context, entities);
        }
    }

    @Override
    public List<JoinQuery> joinQuery() {
        return List.of(new JoinQuery(bannerIdField.getTable(),
                JoinType.LEFT_OUTER_JOIN,
                bannerIdField.eq(BANNERS.BID)));
    }

    protected void deleteEntities(DSLContext context, BannerRepositoryContainer updateParameters,
                                  Collection<AppliedChanges<B>> appliedChanges) {
        var bannerIds = listToSet(appliedChanges, ac -> ac.getModel().getId());
        context.delete(bannerIdField.getTable())
                .where(bannerIdField.in(bannerIds))
                .execute();
    }

}
