package ru.yandex.direct.core.entity.banner.repository.type;

import java.util.Collection;
import java.util.Set;
import java.util.function.Function;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableSet;
import org.jooq.Field;
import org.jooq.Record;
import org.jooq.TableField;

import ru.yandex.direct.core.entity.banner.model.Banner;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperBuilder;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelProperty;

import static com.google.common.base.Preconditions.checkState;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.convertibleProperty;
import static ru.yandex.direct.jooqmapper.write.WriterBuilders.fromProperty;

/**
 * Абстрактный класс для реализации на базе него репозиторных тайп-саппортов конкретных ассетов.
 * <p>
 * Подходит для отношения один-к-одному к основной таблице banners.
 * <p>
 * Для таблицы, в которой ровно два поля:
 * 1. первичный ключ bannerId
 * 2. поле с полезными данными
 * <p>
 * ВАЖНО: используйте, если есть высокая степень уверенности, что новые поля в таблицу и интерфейс добавляться не будут;
 * иначе добавление даже одного поля приведет к необходимости перехода на другой абстрактный класс.
 *
 * @param <B> - интерфейс
 * @param <V> - тип поля с данными в интерфейсе
 * @param <R> - jooq класс
 * @param <F> - тип поля с данными в jooq классе
 */
@ParametersAreNonnullByDefault
public abstract class AbstractRelatedSingleFieldRepositoryTypeSupport<B extends Banner, V, R extends Record, F>
        extends AbstractFlatRelatedEntityRepositoryTypeSupport<B, R> {

    private final ModelProperty<? super B, V> modelProperty;
    private final Set<Field<?>> singleFieldAndIdFieldSet;

    public AbstractRelatedSingleFieldRepositoryTypeSupport(DslContextProvider dslContextProvider,
                                                           ModelProperty<? super B, V> modelProperty,
                                                           TableField<R, Long> bannerIdField,
                                                           TableField<R, F> field,
                                                           Function<F, V> readConverter,
                                                           Function<V, F> writeConverter) {
        super(dslContextProvider,
                bannerIdField,
                JooqMapperBuilder.<B>builder()
                        .writeField(bannerIdField, fromProperty(B.ID))
                        .map(convertibleProperty(modelProperty, field, readConverter::apply, writeConverter::apply))
                        .build()
        );
        checkState(bannerIdField.getTable().equals(field.getTable()));

        this.modelProperty = modelProperty;
        this.singleFieldAndIdFieldSet = ImmutableSet.of(field, bannerIdField);
    }

    @Override
    public final Collection<Field<?>> getFields() {
        // Поскольку mapper использует writeField, мы переопределили метод, чтобы возвращать и такие поля
        // тоже - это даст возможность доставать ограничивающие интерфейсы из базы не читая модели полностью.
        // Ещё лучше научится бы из jooqMapper делать getAllFields(), а не getFieldsToRead().
        return singleFieldAndIdFieldSet;
    }

    @Override
    protected boolean isAddEntity(B model) {
        return modelProperty.get(model) != null;
    }

    @Override
    protected boolean isAddEntity(AppliedChanges<B> appliedChange) {
        return appliedChange.assigned(modelProperty);
    }

    @Override
    protected boolean isUpdateEntity(AppliedChanges<B> appliedChange) {
        return appliedChange.replaced(modelProperty);
    }

    @Override
    protected boolean isDeleteEntity(AppliedChanges<B> appliedChange) {
        return appliedChange.deleted(modelProperty);
    }

}
