package ru.yandex.direct.core.entity.banner.service;

import java.util.List;
import java.util.Set;

import javax.annotation.Nullable;

import one.util.streamex.StreamEx;
import org.jooq.DSLContext;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.StatusBsSynced;
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.core.entity.banner.container.BannerAdditionalActionsContainer;
import ru.yandex.direct.core.entity.banner.model.Banner;
import ru.yandex.direct.core.entity.banner.type.creative.BannerCreativeRepository;
import ru.yandex.direct.core.entity.banner.type.flags.BannerUserFlagsUpdateInfo;
import ru.yandex.direct.core.entity.banner.type.flags.BannerUserFlagsUpdatesRepository;
import ru.yandex.direct.core.entity.campaign.model.CampaignStatusModerate;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.client.service.ClientGeoService;
import ru.yandex.direct.core.entity.creative.repository.CreativeRepository;
import ru.yandex.direct.core.entity.moderation.service.ModerationService;
import ru.yandex.direct.dbutil.model.ClientId;

import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Service
public class BannerAdditionalActionsService {

    private final CampaignRepository campaignRepository;
    private final AdGroupRepository adGroupRepository;
    private final CreativeRepository creativeRepository;
    private final BannerCreativeRepository bannerCreativeRepository;
    private final ClientGeoService clientGeoService;
    private final ModerationService moderationService;
    private final BannerUserFlagsUpdatesRepository bannerUserFlagsUpdatesRepository;

    public BannerAdditionalActionsService(CampaignRepository campaignRepository,
                                          AdGroupRepository adGroupRepository,
                                          CreativeRepository creativeRepository,
                                          BannerCreativeRepository bannerCreativeRepository,
                                          ClientGeoService clientGeoService,
                                          ModerationService moderationService,
                                          BannerUserFlagsUpdatesRepository bannerUserFlagsUpdatesRepository) {
        this.campaignRepository = campaignRepository;
        this.adGroupRepository = adGroupRepository;
        this.creativeRepository = creativeRepository;
        this.bannerCreativeRepository = bannerCreativeRepository;
        this.clientGeoService = clientGeoService;
        this.moderationService = moderationService;
        this.bannerUserFlagsUpdatesRepository = bannerUserFlagsUpdatesRepository;
    }

    public void processAdditionalActionsContainer(DSLContext dslContext,
                                                  BannerAdditionalActionsContainer container) {
        markDraftPerformanceAdGroupsAsModerated(dslContext, container.getClientId(),
                container.getDraftPerformanceAdGroupAsModerated());
        setStatusBlGeneratedProcessing(dslContext, container.getAdGroupsIdsForSettingBlProcessing());
        remoderateAdGroups(dslContext, container.getAdGroupIdsForRemoderation());
        remoderateCampaigns(dslContext, container.getCampaignIdsForRemoderation());
        remoderateCampaignsForContentPromotion(dslContext,
                container.getCampaignIdsForRemoderationForContentPromotion());

        resyncAdGroupsForBS(dslContext, container.getAdGroupsIdsForBSResync());

        updateCreativesGeo(dslContext,
                StreamEx.of(container.getCreativeIdsToSetGeo()).toList(),
                container.getClientRegionId());
        sendCreativesToModeration(dslContext, container.getCreativeIdsToModerate());

        deleteBannerModerationFlags(dslContext, container.getBannersToClearModerationFlags());
        saveBannerUserFlagsUpdates(dslContext, container.getBannerUserFlagsUpdates());
    }

    private void remoderateAdGroups(DSLContext dslContext, Set<Long> adGroupIdsForRemoderation) {
        adGroupRepository.sendDraftAdGroupToModerate(dslContext, adGroupIdsForRemoderation);
    }

    private void remoderateCampaigns(DSLContext dslContext, Set<Long> campaignIdsForRemoderation) {
        campaignRepository.sendRejectedCampaignsToModerate(dslContext, campaignIdsForRemoderation);
    }

    private void remoderateCampaignsForContentPromotion(DSLContext dslContext, Set<Long> campaignIdsForRemoderation) {
        campaignRepository.setStatusModerateIfNewOrNo(dslContext,
                campaignIdsForRemoderation, CampaignStatusModerate.READY);
    }

    private void resyncAdGroupsForBS(DSLContext dslContext, Set<Long> adGroupIdsForBSResync) {
        adGroupRepository.updateStatusBsSynced(dslContext.configuration(), adGroupIdsForBSResync, StatusBsSynced.NO);
    }

    public void sendCreativesToModeration(DSLContext dslContext, Set<Long> creativeIds) {
        creativeRepository.sendCreativesToModeration(dslContext, creativeIds);
    }

    private void setStatusBlGeneratedProcessing(DSLContext dslContext, Set<Long> adGroupIdsForForSettingBlProcessing) {
        adGroupRepository.setStatusBlGeneratedProcessing(dslContext, adGroupIdsForForSettingBlProcessing);
    }

    private void markDraftPerformanceAdGroupsAsModerated(DSLContext dslContext,
                                                         ClientId clientId,
                                                         Set<Long> draftPerformanceAdGroupAsModerated) {
        adGroupRepository.markDraftPerformanceAdGroupsAsModerated(dslContext, clientId,
                draftPerformanceAdGroupAsModerated);
    }

    public void updateCreativesGeo(DSLContext context, List<Long> creativeIds, @Nullable Long clientRegionId) {
        if (creativeIds.size() == 0) {
            return;
        }
        var geoTree = clientGeoService.getClientTranslocalGeoTree(clientRegionId);
        var geoByCreativeId = bannerCreativeRepository.getJoinedGeo(context.configuration(), geoTree, creativeIds);
        creativeRepository.updateCreativesGeo(context, geoByCreativeId);
    }

    private void deleteBannerModerationFlags(DSLContext dslContext, List<Banner> banners) {
        moderationService.deleteBannerModerationFlags(dslContext, mapList(banners, Banner::getId));
    }

    private void saveBannerUserFlagsUpdates(DSLContext dslContext, List<BannerUserFlagsUpdateInfo> bannerUserFlagsUpdates) {
        bannerUserFlagsUpdatesRepository.saveBannerUserFlagsUpdateTimestamp(dslContext, bannerUserFlagsUpdates);
    }
}
