package ru.yandex.direct.core.entity.banner.service;

import java.time.LocalDateTime;
import java.util.Comparator;
import java.util.List;
import java.util.function.Consumer;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.aggregatedstatuses.repository.AggregatedStatusesRepository;
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.core.entity.banner.model.BannerWithSystemFields;
import ru.yandex.direct.core.entity.banner.repository.BannerModerationRepository;
import ru.yandex.direct.core.entity.banner.repository.BannerModifyRepository;
import ru.yandex.direct.core.entity.banner.repository.BannerTypedRepository;
import ru.yandex.direct.core.entity.banner.service.moderation.BannerModerationHelper;
import ru.yandex.direct.core.entity.banner.service.validation.ArchiveUnarchiveBannerValidationService;
import ru.yandex.direct.core.entity.banner.type.image.BannerImageRepository;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.operation.Operation;
import ru.yandex.direct.operation.creator.IgnoreDuplicatesOperationCreator;
import ru.yandex.direct.operation.creator.OperationCreator;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.result.Result;

import static ru.yandex.direct.core.entity.banner.service.validation.defects.BannerDefects.adMoreThanOnceInRequest;

/**
 * Сервис-обертка для архивации и разархивации баннеров.
 */
@Service
@ParametersAreNonnullByDefault
public class BannerArchiveUnarchiveService {
    private final ShardHelper shardHelper;
    private final AdGroupRepository adGroupRepository;
    private final CampaignRepository campaignRepository;
    private final BannerImageRepository imageRepository;
    private final BannerTypedRepository typedRepository;
    private final BannerModifyRepository modifyRepository;
    private final BannerModerationRepository bannerModerationRepository;
    private final AggregatedStatusesRepository aggregatedStatusesRepository;
    private final DslContextProvider ppcDslContextProvider;
    private final ArchiveUnarchiveBannerValidationService validationService;
    private final BannerModerationHelper moderationHelper;
    private final FeatureService featureService;

    @Autowired
    public BannerArchiveUnarchiveService(ShardHelper shardHelper,
                                         AdGroupRepository adGroupRepository,
                                         CampaignRepository campaignRepository,
                                         AggregatedStatusesRepository aggregatedStatusesRepository,
                                         DslContextProvider ppcDslContextProvider,
                                         ArchiveUnarchiveBannerValidationService validationService,
                                         BannerModerationHelper moderationHelper,
                                         FeatureService featureService,
                                         BannerImageRepository imageRepository,
                                         BannerTypedRepository typedRepository,
                                         BannerModifyRepository modifyRepository,
                                         BannerModerationRepository bannerModerationRepository) {
        this.shardHelper = shardHelper;
        this.adGroupRepository = adGroupRepository;
        this.campaignRepository = campaignRepository;
        this.imageRepository = imageRepository;
        this.aggregatedStatusesRepository = aggregatedStatusesRepository;
        this.ppcDslContextProvider = ppcDslContextProvider;
        this.validationService = validationService;
        this.moderationHelper = moderationHelper;
        this.featureService = featureService;
        this.typedRepository = typedRepository;
        this.modifyRepository = modifyRepository;
        this.bannerModerationRepository = bannerModerationRepository;
    }

    /**
     * Заархивировать/разархивировать баннеры. Частный случай update.
     * В списке изменений принимается только изменение флага архивации
     *
     * @param clientId      ID клиента
     * @param operatorUid   UID оператора
     * @param bannerChanges Список изменений, принимается только флаг {@link BannerWithSystemFields#STATUS_ARCHIVED}
     * @param archive       true если пришел запрос на архивацию, иначе false
     */
    public MassResult<Long> archiveUnarchiveBanners(ClientId clientId, Long operatorUid,
                                                    List<ModelChanges<BannerWithSystemFields>> bannerChanges,
                                                    boolean archive) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);

        Comparator<ModelChanges<BannerWithSystemFields>> comparator = Comparator.comparing(ModelChanges::getId);
        LocalDateTime updateBefore = LocalDateTime.now();
        OperationCreator<ModelChanges<BannerWithSystemFields>, Operation<Long>> operationCreator =
                changes -> new BannersArchiveUnarchiveOperation(
                        shard, operatorUid, changes,
                        validationService, adGroupRepository, typedRepository,
                        modifyRepository, bannerModerationRepository,
                        campaignRepository, aggregatedStatusesRepository,
                        clientId, archive, updateBefore,
                        ppcDslContextProvider, moderationHelper,
                        featureService.getEnabledForClientId(clientId), imageRepository);
        Consumer<Result<Long>> resultVisitor = result -> result.getValidationResult()
                .addWarning(adMoreThanOnceInRequest());

        return new IgnoreDuplicatesOperationCreator<>(operationCreator, comparator, resultVisitor)
                .create(bannerChanges)
                .prepareAndApply();
    }
}
