package ru.yandex.direct.core.entity.banner.service;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.Nullable;

import one.util.streamex.StreamEx;
import org.jooq.DSLContext;
import org.jooq.Query;

import ru.yandex.direct.core.entity.banner.model.BannerWithCreative;
import ru.yandex.direct.core.entity.banner.repository.BannerRepository;
import ru.yandex.direct.core.entity.banner.service.validation.DeleteBannerValidationService;
import ru.yandex.direct.core.entity.client.service.ClientGeoService;
import ru.yandex.direct.core.entity.creative.repository.CreativeRepository;
import ru.yandex.direct.core.entity.metrika.repository.MetrikaCampaignRepository;
import ru.yandex.direct.core.entity.moderation.repository.ModerationRepository;
import ru.yandex.direct.core.entity.redirectcheckqueue.repository.RedirectCheckQueueRepository;
import ru.yandex.direct.core.entity.turbolanding.model.TurboLanding;
import ru.yandex.direct.core.entity.turbolanding.model.TurboLandingMetrikaCountersAndGoals;
import ru.yandex.direct.core.entity.turbolanding.repository.TurboLandingRepository;
import ru.yandex.direct.dbschema.ppc.enums.ModObjectVersionObjType;
import ru.yandex.direct.dbschema.ppc.enums.ModReasonsType;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardKey;
import ru.yandex.direct.dbutil.sharding.ShardSupport;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.operation.Applicability;
import ru.yandex.direct.operation.operationwithid.AbstractOperationWithId;
import ru.yandex.direct.regions.GeoTree;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static java.util.function.Function.identity;
import static java.util.stream.Collectors.counting;
import static ru.yandex.direct.utils.FunctionalUtils.filterAndMapToSet;

/**
 * Операция удаления баннера
 */
public class BannerDeleteOperation extends AbstractOperationWithId {
    private final DslContextProvider ppcDslContextProvider;
    private final ShardSupport shardSupport;
    private final int shard;
    private final DeleteBannerValidationService deleteBannerValidationService;
    private final long operatorUid;
    private final ClientId clientId;
    private final ClientGeoService clientGeoService;
    private final BannerRepository bannerRepository;
    private final CreativeRepository creativeRepository;
    private final TurboLandingRepository turboLandingRepository;
    private final MetrikaCampaignRepository metrikaCampaignRepository;
    private final ModerationRepository moderationRepository;
    private final RedirectCheckQueueRepository redirectCheckQueueRepository;

    public BannerDeleteOperation(
            int shard,
            ClientId clientId,
            long operatorUid,
            List<Long> modelIds,
            Applicability applicability, DslContextProvider ppcDslContextProvider,
            ShardSupport shardSupport,
            DeleteBannerValidationService deleteBannerValidationService,
            ClientGeoService clientGeoService,
            BannerRepository bannerRepository,
            CreativeRepository creativeRepository,
            TurboLandingRepository turboLandingRepository,
            MetrikaCampaignRepository metrikaCampaignRepository,
            ModerationRepository moderationRepository,
            RedirectCheckQueueRepository redirectCheckQueueRepository
    ) {
        super(applicability, modelIds);
        this.shard = shard;
        this.ppcDslContextProvider = ppcDslContextProvider;
        this.shardSupport = shardSupport;
        this.deleteBannerValidationService = deleteBannerValidationService;
        this.operatorUid = operatorUid;
        this.clientId = clientId;
        this.clientGeoService = clientGeoService;
        this.bannerRepository = bannerRepository;
        this.creativeRepository = creativeRepository;
        this.turboLandingRepository = turboLandingRepository;
        this.metrikaCampaignRepository = metrikaCampaignRepository;
        this.moderationRepository = moderationRepository;
        this.redirectCheckQueueRepository = redirectCheckQueueRepository;
    }

    @Override
    protected ValidationResult<List<Long>, Defect> validate(List<Long> ids) {
        return deleteBannerValidationService.validateDelete(shard, operatorUid, clientId, ids);
    }

    protected GeoTree getGeoTreeByClientId(@Nullable ClientId clientId) {
        return clientGeoService.getClientTranslocalGeoTree(clientId);
    }

    @Override
    protected void execute(List<Long> bannerIds) {
        if (bannerIds.isEmpty()) {
            return;
        }
        GeoTree geoTree = getGeoTreeByClientId(clientId);
        // Все удаление должно идти в одной транзакции
        ppcDslContextProvider.ppcTransaction(shard, config -> {
            DSLContext context = config.dsl();

            // Получить ID связанных изображений и креативов
            Map<Long, Long> campaignIdsByBannerIds = bannerRepository.relations
                    .getCampaignIdsByBannerIds(context, bannerIds);
            Map<Long, TurboLanding> turbolandingsByBannerId = turboLandingRepository
                    .getTurbolandingsLinkedWithBanners(context, bannerIds);
            List<BannerWithCreative> bannersWithCreative = bannerRepository.type
                    .getSafely(shard, bannerIds, BannerWithCreative.class);
            List<Long> canvasOrVideoIds = bannerRepository.creative.getBannerPerformanceCreativeIds(context,
                    bannerIds);
            Set<Long> imageIds = StreamEx
                    .of(bannerRepository.image.getImageIdsByBannerIds(context, bannerIds).values())
                    .toFlatCollection(identity(), HashSet::new);

            // Перед удалением записей из banners и banner_turbolandings нужно уменьшить счетчики в camp_metrika_goals
            decreaseCampaignMetrikaGoalCounters(context, turbolandingsByBannerId, campaignIdsByBannerIds);

            bannerRepository.common.deleteBanners(context, bannerIds);

            bannerRepository.additionalHrefs.deleteAdditionalHrefsForBanners(context, bannerIds);

            bannerRepository.tns.deleteBannerTnsByBannerIds(context, bannerIds);
            bannerRepository.pixels.deleteBannerPixelsByBannerIds(context, bannerIds);

            Set<Long> creativeIds = filterAndMapToSet(bannersWithCreative,
                    banner -> banner.getCreativeId() != null,
                    BannerWithCreative::getCreativeId);
            Map<Long, String> geoByCreativeId = bannerRepository.creative.getJoinedGeo(config, geoTree, creativeIds);
            creativeRepository.updateCreativesGeo(context, geoByCreativeId);

            redirectCheckQueueRepository.deleteBannersFromQueue(context, bannerIds);
            metrikaCampaignRepository.deleteBannerTurbolandingCounters(context, bannerIds);
            deleteModerationData(context, bannerIds, canvasOrVideoIds, imageIds, campaignIdsByBannerIds);
            bannerRepository.measures.deleteBannersMeasurers(context, bannerIds);
            bannerRepository.names.deleteNamesForBanners(context, bannerIds);
            bannerRepository.logos.delete(context, bannerIds);
            bannerRepository.buttons.delete(context, bannerIds);
            bannerRepository.bigKingImage.delete(context, bannerIds);
            bannerRepository.performanceMain.delete(context, bannerIds);
        });
        // в ru.yandex.direct.oneshot.oneshots.cleanup.DeleteBanners тоже перечислены таблицы
        // при добавлении новых желательно актуализировать список и там
        shardSupport.deleteValues(ShardKey.BID, bannerIds);
    }

    /**
     * Удаляет данные модерации по баннерам и связанным с ними изображениям и креативам
     *
     * @param context                Контекст
     * @param bannerIds              Список ID баннеров
     * @param imageIds               Список ID картинок графических объявлений
     * @param campaignIdsByBannerIds Мап id баннера -> id кампании
     */
    private void deleteModerationData(DSLContext context, Collection<Long> bannerIds,
                                      Collection<Long> canvasOrVideoIds, Collection<Long> imageIds,
                                      Map<Long, Long> campaignIdsByBannerIds) {
        // удалить причины отклонения
        List<Query> toDelete = new ArrayList<>();
        toDelete.add(moderationRepository.deleteModReasonBatch(
                context,
                Arrays.asList(ModReasonsType.banner, ModReasonsType.contactinfo,
                        ModReasonsType.sitelinks_set, ModReasonsType.image, ModReasonsType.display_href),
                bannerIds));
        toDelete.add(moderationRepository.deleteModReasonBatch(context,
                Collections.singletonList(ModReasonsType.image_ad), imageIds));
        toDelete.add(moderationRepository.deleteModReasonBatch(context,
                Arrays.asList(ModReasonsType.canvas, ModReasonsType.video_addition), canvasOrVideoIds));

        //удалить версии объектов для модерации
        toDelete.add(moderationRepository.deleteModObjectVersionBatch(context,
                Arrays.asList(ModObjectVersionObjType.banner,
                        ModObjectVersionObjType.contactinfo, ModObjectVersionObjType.sitelinks_set,
                        ModObjectVersionObjType.image, ModObjectVersionObjType.display_href,
                        ModObjectVersionObjType.image_ad, ModObjectVersionObjType.canvas,
                        ModObjectVersionObjType.video_addition),
                bannerIds));

        context.batch(toDelete).execute();
    }


    /**
     * Уменьшает счетчики турболендингов для целей в camp_metrika_goals
     */
    private void decreaseCampaignMetrikaGoalCounters(
            DSLContext context,
            Map<Long, TurboLanding> turbolandingsByBannerId,
            Map<Long, Long> campaignIdByBannerId
    ) {
        if (turbolandingsByBannerId.isEmpty()) {
            return;
        }

        ArrayList<Query> toUpdate = new ArrayList<>();
        for (Long bannerId : turbolandingsByBannerId.keySet()) {
            TurboLandingMetrikaCountersAndGoals countersAndGoals = new TurboLandingMetrikaCountersAndGoals(
                    turbolandingsByBannerId.get(bannerId).getMetrikaCounters());
            List<Long> goalIds = countersAndGoals.getAllMetrikaGoals();
            Long campaignId = campaignIdByBannerId.get(bannerId);
            Map<Long, Long> linksByGoal = StreamEx.of(goalIds).collect(Collectors.groupingBy(identity(), counting()));
            if (linksByGoal.isEmpty()) {
                continue;
            }
            toUpdate.add(metrikaCampaignRepository.decreaseMetrikaGoalsLinksCountBatch(context, linksByGoal,
                    campaignId));
        }
        context.batch(toUpdate).execute();
    }
}
