package ru.yandex.direct.core.entity.banner.service;

import java.util.Collection;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Predicate;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.jetbrains.annotations.NotNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.copyentity.CopyOperationContainer;
import ru.yandex.direct.core.copyentity.EntityService;
import ru.yandex.direct.core.entity.banner.container.AdsCountCriteria;
import ru.yandex.direct.core.entity.banner.container.AdsSelectionCriteria;
import ru.yandex.direct.core.entity.banner.model.Banner;
import ru.yandex.direct.core.entity.banner.model.BannerPrice;
import ru.yandex.direct.core.entity.banner.model.BannerWithAdGroupId;
import ru.yandex.direct.core.entity.banner.model.BannerWithPixels;
import ru.yandex.direct.core.entity.banner.model.BannerWithSystemFields;
import ru.yandex.direct.core.entity.banner.model.BannerWithVcard;
import ru.yandex.direct.core.entity.banner.model.TextBanner;
import ru.yandex.direct.core.entity.banner.repository.BannerCommonRepository;
import ru.yandex.direct.core.entity.banner.repository.BannerRelationsRepository;
import ru.yandex.direct.core.entity.banner.repository.BannerTypedRepository;
import ru.yandex.direct.core.entity.banner.service.validation.DeleteBannerValidationService;
import ru.yandex.direct.core.entity.banner.type.href.BannerDomainRepository;
import ru.yandex.direct.core.entity.banner.type.price.BannerPriceRepository;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.campaign.service.accesschecker.CampaignSubObjectAccessCheckerFactory;
import ru.yandex.direct.core.entity.campaign.service.accesschecker.CampaignSubObjectAccessValidator;
import ru.yandex.direct.core.entity.campaign.service.validation.CampaignAccessType;
import ru.yandex.direct.core.entity.domain.model.Domain;
import ru.yandex.direct.core.entity.domain.repository.DomainRepository;
import ru.yandex.direct.core.entity.domain.service.DomainService;
import ru.yandex.direct.core.entity.trustedredirects.service.TrustedRedirectsService;
import ru.yandex.direct.dbschema.ppc.enums.BannersBannerType;
import ru.yandex.direct.dbutil.QueryWithForbiddenShardMapping;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.sharding.ShardKey;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.multitype.entity.LimitOffset;
import ru.yandex.direct.operation.Applicability;
import ru.yandex.direct.rbac.RbacRole;
import ru.yandex.direct.rbac.RbacService;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static java.util.Collections.emptyList;
import static java.util.Collections.emptyMap;
import static java.util.Collections.singletonList;
import static java.util.function.Function.identity;
import static java.util.stream.Collectors.groupingBy;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toSet;
import static ru.yandex.direct.core.entity.banner.repository.filter.BannerFilterFactory.bannerCampaignIdFilter;
import static ru.yandex.direct.core.entity.banner.type.href.BannerWithHrefUtils.toUnicodeDomain;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Service
@ParametersAreNonnullByDefault
public class BannerService implements EntityService<BannerWithAdGroupId, Long> {

    private static final Logger logger = LoggerFactory.getLogger(BannerService.class);

    private final ShardHelper shardHelper;
    private final BannerTypedRepository bannerTypedRepository;
    private final BannersAddOperationFactory bannersAddOperationFactory;
    private final CampaignSubObjectAccessCheckerFactory campaignSubObjectAccessCheckerFactory;
    private final BannersUpdateOperationFactory bannersUpdateOperationFactory;
    private final BannerPriceRepository bannerPriceRepository;
    private final BannerRelationsRepository bannerRelationsRepository;
    private final CampaignRepository campaignRepository;
    private final DeleteBannerValidationService deleteBannerValidationService;
    private final RbacService rbacService;
    private final BannerDomainRepository bannerDomainRepository;
    private final TrustedRedirectsService trustedRedirectsService;
    private final BannerCommonRepository bannerCommonRepository;
    private final DomainService domainService;
    private final DomainRepository domainRepository;
    private final BannerDeleteOperationFactory bannerDeleteOperationFactory;

    @Autowired
    public BannerService(
            ShardHelper shardHelper,
            BannerTypedRepository bannerTypedRepository,
            CampaignSubObjectAccessCheckerFactory campaignSubObjectAccessCheckerFactory,
            BannersAddOperationFactory bannersAddOperationFactory,
            BannersUpdateOperationFactory bannersUpdateOperationFactory,
            BannerPriceRepository bannerPriceRepository,
            BannerRelationsRepository bannerRelationsRepository,
            CampaignRepository campaignRepository,
            DeleteBannerValidationService deleteBannerValidationService,
            RbacService rbacService,
            BannerDomainRepository bannerDomainRepository,
            TrustedRedirectsService trustedRedirectsService,
            BannerCommonRepository bannerCommonRepository,
            DomainService domainService,
            DomainRepository domainRepository,
            BannerDeleteOperationFactory bannerDeleteOperationFactory) {
        this.shardHelper = shardHelper;
        this.bannerTypedRepository = bannerTypedRepository;
        this.campaignSubObjectAccessCheckerFactory = campaignSubObjectAccessCheckerFactory;
        this.bannersAddOperationFactory = bannersAddOperationFactory;
        this.bannersUpdateOperationFactory = bannersUpdateOperationFactory;
        this.bannerPriceRepository = bannerPriceRepository;
        this.bannerRelationsRepository = bannerRelationsRepository;
        this.campaignRepository = campaignRepository;
        this.deleteBannerValidationService = deleteBannerValidationService;
        this.rbacService = rbacService;
        this.bannerDomainRepository = bannerDomainRepository;
        this.trustedRedirectsService = trustedRedirectsService;
        this.bannerCommonRepository = bannerCommonRepository;
        this.domainService = domainService;
        this.domainRepository = domainRepository;
        this.bannerDeleteOperationFactory = bannerDeleteOperationFactory;
    }

    @Override
    public MassResult<Long> add(
            ClientId clientId, Long operatorUid, List<BannerWithAdGroupId> entities, Applicability applicability) {
        var operation = bannersAddOperationFactory.createAddOperation(
                applicability, false, entities, clientId, operatorUid,
                true, false);
        return operation.prepareAndApply();
    }
    @Override
    public MassResult<Long> copy(CopyOperationContainer copyContainer,
                                 List<BannerWithAdGroupId> entities, Applicability applicability) {
        var operation = bannersAddOperationFactory.createAddOperation(
                applicability, false, entities,
                copyContainer.getClientIdTo(), copyContainer.getOperatorUid(),
                true, true);
        return operation.prepareAndApply();
    }

    @Override
    public List<BannerWithAdGroupId> get(ClientId clientId, Long operatorUid, Collection<Long> ids) {
        return mapList(getBannersByIds(ids), identity());
    }

    /**
     * Возвращает {@link List}, где элементами являются найденные баннеры
     *
     * @param operatorUid       uid оператора
     * @param clientId          id клиента
     * @param selectionCriteria условия отбора групп
     * @param limitOffset       ограничения на полученный список
     * @return список найденных баннеров
     */
    public List<BannerWithSystemFields> getBannersBySelectionCriteria(
            long operatorUid, ClientId clientId,
            AdsSelectionCriteria selectionCriteria, LimitOffset limitOffset) {
        if (selectionCriteria.getAdGroupIds().isEmpty()
                && selectionCriteria.getCampaignIds().isEmpty()
                && selectionCriteria.getAdIds().isEmpty()) {
            return emptyList();
        }

        int shard = shardHelper.getShardByClientIdStrictly(clientId);

        Map<Long, List<Long>> bannerIdsByCampaignId =
                bannerRelationsRepository.getCampaignIdToBannerIdsBySelectionCriteria(shard, selectionCriteria);

        if (bannerIdsByCampaignId.isEmpty()) {
            return emptyList();
        }

        Set<Long> campaignIds = bannerIdsByCampaignId.keySet();

        Set<Long> filteredCampaignIds = filterCampaignIds(operatorUid, clientId, campaignIds);

        Set<Long> bannerIds =
                bannerIdsByCampaignId.entrySet().stream().filter(e -> filteredCampaignIds.contains(e.getKey()))
                        .flatMap(e -> e.getValue().stream()).sorted().collect(
                        Collectors.toCollection(LinkedHashSet::new));

        var banners = bannerTypedRepository.getBanners(shard, bannerIds, limitOffset);
        return mapList(banners, b -> (BannerWithSystemFields) b);
    }

    /**
     * Получить список простых описаний баннеров по списку их идентификаторов
     */
    @QueryWithForbiddenShardMapping("bid")
    public List<BannerWithSystemFields> getBannersByIds(Collection<Long> bannerIds) {
        return shardHelper.groupByShard(bannerIds, ShardKey.BID)
                .stream()
                .map(e -> bannerTypedRepository.getStrictlyFullyFilled(e.getKey(), e.getValue(),
                        BannerWithSystemFields.class))
                .flatMap(List::stream)
                .collect(toList());
    }

    public List<Banner> getNonArchivedBannersByCampaignIds(int shard,
                                                Long operatorUid,
                                                ClientId clientId,
                                                Collection<Long> campaignIds,
                                                @Nullable LimitOffset limitOffset) {
        Set<Long> filteredCampaignIds = filterCampaignIds(operatorUid, clientId, campaignIds);
        return bannerTypedRepository.getNonArchivedBannersByCampaignIds(shard, filteredCampaignIds, limitOffset);
    }

    public List<Banner> getBannersByCampaignIds(int shard,
                                                Long operatorUid,
                                                ClientId clientId,
                                                Collection<Long> campaignIds,
                                                @Nullable LimitOffset limitOffset) {
        Set<Long> filteredCampaignIds = filterCampaignIds(operatorUid, clientId, campaignIds);
        return bannerTypedRepository.getBannersByCampaignIds(shard, filteredCampaignIds, limitOffset);
    }

    @NotNull
    private Set<Long> filterCampaignIds(Long operatorUid, ClientId clientId, Collection<Long> campaignIds) {
        CampaignSubObjectAccessValidator checker = campaignSubObjectAccessCheckerFactory
                .newCampaignChecker(operatorUid, clientId, campaignIds)
                .createValidator(CampaignAccessType.READ);

        return campaignIds.stream()
                .map(checker)
                .filter(vr -> !vr.hasAnyErrors())
                .map(ValidationResult::getValue)
                .collect(toSet());
    }

    public <T extends BannerWithSystemFields> List<BannerWithSystemFields> getBannersByCampaignIds(
            Collection<Long> campaignIds,
            Class<T> bannersClass
    ) {
        return shardHelper.groupByShard(campaignIds, ShardKey.CID)
                .stream()
                .map(e -> bannerTypedRepository.getBannersByCampaignIdsAndClass(e.getKey(), e.getValue(),
                        bannersClass))
                .flatMap(List::stream)
                .nonNull()
                .collect(toList());
    }

    public List<Banner> getBannersByCampaignIds(
            Collection<Long> campaignIds, Collection<Class<? extends Banner>> bannersClasses
    ) {
        return shardHelper.groupByShard(campaignIds, ShardKey.CID)
                .stream()
                .map(e -> bannerTypedRepository.getSafely(e.getKey(), bannerCampaignIdFilter(campaignIds),
                        bannersClasses))
                .flatMap(List::stream)
                .nonNull()
                .collect(toList());
    }

    /**
     * Получить список простых описаний баннеров по списку идентификаторов их кампаний
     */
    public List<BannerWithSystemFields> getBannersByCampaignIds(Collection<Long> campaignIds) {
        return getBannersByCampaignIds(campaignIds, BannerWithSystemFields.class);
    }

    public void updateBannerVcardIds(int shard, Long operatorUid, ClientId clientId, Collection<Long> bannerIds,
                                     Map<Long, Long> vcardIdByBannerId) {
        List<BannerWithVcard> banners = bannerTypedRepository.getSafely(shard, bannerIds, BannerWithVcard.class);
        List<ModelChanges<BannerWithSystemFields>> bannerChanges = StreamEx.of(banners)
                .map(banner -> ModelChanges
                        .build(banner, BannerWithVcard.VCARD_ID, vcardIdByBannerId.get(banner.getId()))
                        .castModelUp(BannerWithSystemFields.class))
                .toList();

        MassResult<Long> result = bannersUpdateOperationFactory.createPartialUpdateOperation(bannerChanges,
                operatorUid, clientId)
                .prepareAndApply();

        ValidationResult<?, Defect> validationResult = result.getValidationResult();
        if (validationResult.hasAnyErrors()) {
            logger.warn("errors on banners vcard updating: {}", validationResult.flattenErrors());
        }
    }

    @QueryWithForbiddenShardMapping("bid")
    public Map<Long, BannerPrice> getNewBannerPricesByBannerId(Collection<Long> bannerIds) {
        return shardHelper.groupByShard(bannerIds, ShardKey.BID)
                .stream()
                .map(e -> bannerPriceRepository.getBannerPricesByBannerIds(e.getKey(), e.getValue()))
                .flatMapToEntry(identity())
                .distinctKeys()
                .toMap();
    }

    @QueryWithForbiddenShardMapping("pid")
    public Map<Long, List<BannerWithSystemFields>> getBannersByAdGroupIds(Collection<Long> adGroupIds) {
        return shardHelper.groupByShard(adGroupIds, ShardKey.PID).stream()
                .mapKeyValue(bannerTypedRepository::getBannersByGroupIds)
                .flatMap(Collection::stream)
                .map(x -> (BannerWithSystemFields) x)
                .collect(groupingBy(BannerWithSystemFields::getAdGroupId));
    }

    public Map<Long, Long> getAdGroupIdsByBannerIds(ClientId clientId, Collection<Long> bannerIds) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        return bannerRelationsRepository.getAdGroupIdsByBannerIds(shard, bannerIds);
    }

    public Map<Long, Long> getLastChangedBannerIds(int shard, ClientId clientId,
                                                   Map<Long, List<BannersBannerType>> bannerTypesByCampaignIds) {
        Set<Long> campaignIds = campaignRepository.getExistingCampaignIds(shard, clientId,
                bannerTypesByCampaignIds.keySet());
        Set<BannersBannerType> bannerTypes = EntryStream.of(bannerTypesByCampaignIds)
                .filterKeys(campaignIds::contains)
                .values()
                .flatMap(Collection::stream)
                .toSet();
        return bannerRelationsRepository.getLastChangedBannerIdsWithCampaignIds(shard, campaignIds, bannerTypes);
    }

    public Map<Long, Boolean> getHasRunningUnmoderatedBannersByCampaignId(ClientId clientId,
                                                                          Collection<Long> campaignIds) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        Set<Long> campaignIdsWithRunningUnmoderatedBanners =
                bannerRelationsRepository.getHasRunningUnmoderatedBannersByCampaignId(shard, campaignIds);
        return listToMap(campaignIds, identity(), campaignIdsWithRunningUnmoderatedBanners::contains);
    }

    /**
     * Возвращает флаг возможности удаления объявлений
     *
     * @param shard    шард клиента
     * @param clientId идентификатор клиента
     * @param banners  список объявлений
     * @return {@link Map&lt;Long, Boolean&gt;} с соответствием ID объявления и флагом возможности удаления
     */
    public Map<Long, Boolean> getCanBeDeletedBanners(int shard,
                                                     ClientId clientId,
                                                     List<BannerWithSystemFields> banners) {
        if (banners.isEmpty()) {
            return emptyMap();
        }

        Predicate<Long> canDeleteBannersPredicate =
                deleteBannerValidationService.getCanDeleteBannersPredicate(shard, clientId, banners);
        return listToMap(banners, Banner::getId, banner -> canDeleteBannersPredicate.test(banner.getId()));
    }

    /**
     * Возвращает флаг возможности удаления объявлений
     *
     * @see #getCanBeDeletedBanners
     */
    public Map<Long, Boolean> getCanBeDeletedBannersByIds(int shard, ClientId clientId, Collection<Long> bannerIds) {
        List<BannerWithSystemFields> banners =
                bannerTypedRepository.getStrictlyFullyFilled(shard, bannerIds, BannerWithSystemFields.class);
        return getCanBeDeletedBanners(shard, clientId, banners);
    }

    /**
     * Получить главный баннер по каждой из групп, чьи ID переданы в {@code adGroupIds}.
     * <p>
     * Главный баннер тут -- это первый неграфический при сортировке по {@link TextBanner#ID}
     */
    public Map<Long, BannerWithSystemFields> getMainBannerByAdGroupIds(ClientId clientId,
                                                                       Collection<Long> adGroupIds) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        return bannerRelationsRepository.getMainBannerByAdGroupIds(shard, adGroupIds);
    }

    /**
     * Изменить отображаемый домен для всех баннеров кампании. Разрешено только для ролей {@link RbacRole#SUPER},
     * {@link ru.yandex.direct.rbac.RbacRole#SUPPORT}, {@link ru.yandex.direct.rbac.RbacRole#PLACER},
     * {@link ru.yandex.direct.rbac.RbacRole#MANAGER}. Домен должен быть валиден.
     * При несоответствии роли домен просто не будет изменен.
     * При невалидном домене будет выброшено исключение {@link IllegalArgumentException}.
     */
    public void changeCampaignBannersDomains(long operatorUid, long campaignId, String newDomain) {
        if (!rbacService.getUidRole(operatorUid).isInternal()) {
            return;
        }

        newDomain = toUnicodeDomain(newDomain);

        int shard = shardHelper.getShardByCampaignId(campaignId);
        Domain newDomainModel = getOrCreateDomain(shard, newDomain);
        List<Long> bannerIds = bannerDomainRepository.changeCampaignBannersDomains(shard, campaignId, newDomainModel);

        if (trustedRedirectsService.isDomainTrusted(newDomain)) {
            bannerCommonRepository.addToRedirectCheckQueue(shard, bannerIds);
        }
        domainService.updateFilterDomain(shard, singletonList(newDomain));
    }


    /**
     * Для текстового домена добавляет домен в таблицу domains и возвращает модель {@link Domain}
     */
    private Domain getOrCreateDomain(int shard, String newDomain) {
        domainService.addDomains(shard, singletonList(newDomain));
        return domainRepository.getDomains(shard, singletonList(newDomain)).get(0);
    }

    /**
     * Массовое удаление объявлений для клиента.
     * Считается успешным, если удалено хотя бы одно объявление.
     *
     * @param clientId id клиента
     * @param adIds    удаляемые id сайтлинк сетов
     * @return результат удаления сайтлинк сетов
     */
    public MassResult<Long> deleteBannersPartial(long operatorUid, ClientId clientId, List<Long> adIds) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        return bannerDeleteOperationFactory
                .createBannerDeleteOperation(shard, clientId, operatorUid, adIds, Applicability.PARTIAL)
                .prepareAndApply();
    }

    /**
     * Получение кол-ва существующих баннеров по кампаниям
     */
    public Map<Long, Integer> getBannersCounterByCampaigns(ClientId clientId, Collection<Long> campaignIds) {
        return getBannersCounterByCampaigns(clientId, campaignIds, null);
    }

    /**
     * Получение кол-ва существующих баннеров по кампаниям
     */
    public Map<Long, Integer> getBannersCounterByCampaigns(ClientId clientId, Collection<Long> campaignIds,
                                                           @Nullable AdsCountCriteria countCriteria) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        return bannerRelationsRepository.getBannersCounterByCampaigns(shard, campaignIds, countCriteria);
    }

    /**
     * Возвращает данные о количестве баннеров в указанных группах
     */
    public Map<Long, Integer> getBannerQuantitiesByAdGroupIds(ClientId clientId, Collection<Long> adGroupIds) {
        return getBannerQuantitiesByAdGroupIds(clientId, adGroupIds, null);
    }

    /**
     * Возвращает данные о количестве баннеров в указанных группах
     */
    public Map<Long, Integer> getBannerQuantitiesByAdGroupIds(ClientId clientId, Collection<Long> adGroupIds,
                                                              @Nullable AdsCountCriteria countCriteria) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        Map<Long, Integer> result = bannerRelationsRepository.getBannerQuantitiesByAdGroupIds(shard, adGroupIds,
                countCriteria);
        // заполняем 0 значения для групп, которые не удалось заполнить из БД
        adGroupIds.forEach(id -> result.putIfAbsent(id, 0));
        return result;
    }

    /**
     * Получение баннеров с пикселями по списку групп объявлений
     *
     * @param adGroupIds - id групп (любых типов)
     */
    public Map<Long, List<BannerWithPixels>> getBannersWithPixelsByAdGroups(int shard, ClientId clientId,
                                                                            Collection<Long> adGroupIds) {
        List<BannerWithPixels> bannersByGroupIds = bannerTypedRepository.getBannersByGroupIds(shard, adGroupIds,
                clientId, BannerWithPixels.class);
        return StreamEx.of(bannersByGroupIds)
                .collect(groupingBy(banner -> ((BannerWithAdGroupId) banner).getAdGroupId()));
    }

    public Map<Long, List<Long>> getCreativeIdToBidsByCampaignId(ClientId clientId, Long cid) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        return bannerCommonRepository.getCreativeIdToBidsByCampaignId(shard, cid);
    }
}
