package ru.yandex.direct.core.entity.banner.service;

import java.time.LocalDateTime;
import java.util.Comparator;
import java.util.List;
import java.util.function.Consumer;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.aggregatedstatuses.repository.AggregatedStatusesRepository;
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.core.entity.banner.model.BannerWithSystemFields;
import ru.yandex.direct.core.entity.banner.repository.BannerCommonRepository;
import ru.yandex.direct.core.entity.banner.repository.BannerModerationRepository;
import ru.yandex.direct.core.entity.banner.repository.BannerModifyRepository;
import ru.yandex.direct.core.entity.banner.repository.BannerTypedRepository;
import ru.yandex.direct.core.entity.banner.service.validation.SuspendResumeBannerValidationService;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.operation.Operation;
import ru.yandex.direct.operation.creator.IgnoreDuplicatesOperationCreator;
import ru.yandex.direct.operation.creator.OperationCreator;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.result.Result;

import static ru.yandex.direct.core.entity.banner.service.validation.defects.BannerDefects.adMoreThanOnceInRequest;

@Service
@ParametersAreNonnullByDefault
public class BannerSuspendResumeService {
    private final ShardHelper shardHelper;
    private final AdGroupRepository adGroupRepository;
    private final CampaignRepository campaignRepository;
    private final BannerCommonRepository bannerCommonRepository;
    private final BannerTypedRepository typedRepository;
    private final BannerModifyRepository modifyRepository;
    private final BannerModerationRepository moderationRepository;
    private final AggregatedStatusesRepository aggregatedStatusesRepository;
    private final SuspendResumeBannerValidationService validationService;
    private final DslContextProvider ppcDslContextProvider;

    @Autowired
    public BannerSuspendResumeService(ShardHelper shardHelper,
                                      AdGroupRepository adGroupRepository,
                                      CampaignRepository campaignRepository,
                                      BannerCommonRepository bannerCommonRepository,
                                      BannerTypedRepository typedRepository,
                                      BannerModifyRepository modifyRepository,
                                      BannerModerationRepository moderationRepository,
                                      AggregatedStatusesRepository aggregatedStatusesRepository,
                                      SuspendResumeBannerValidationService validationService,
                                      DslContextProvider ppcDslContextProvider) {
        this.shardHelper = shardHelper;
        this.adGroupRepository = adGroupRepository;
        this.campaignRepository = campaignRepository;
        this.bannerCommonRepository = bannerCommonRepository;
        this.typedRepository = typedRepository;
        this.modifyRepository = modifyRepository;
        this.moderationRepository = moderationRepository;
        this.aggregatedStatusesRepository = aggregatedStatusesRepository;
        this.validationService = validationService;
        this.ppcDslContextProvider = ppcDslContextProvider;
    }

    public MassResult<Long> suspendResumeBanners(ClientId clientId, Long operatorUid,
                                                 List<ModelChanges<BannerWithSystemFields>> bannerChanges,
                                                 boolean resume) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        Comparator<ModelChanges<BannerWithSystemFields>> comparator = Comparator.comparing(ModelChanges::getId);
        LocalDateTime updateBefore = LocalDateTime.now();
        OperationCreator<ModelChanges<BannerWithSystemFields>, Operation<Long>> operationCreator =
                changes -> new BannersSuspendResumeOperation(
                        shard, operatorUid, changes, validationService, adGroupRepository, bannerCommonRepository,
                        typedRepository, modifyRepository, moderationRepository,
                        campaignRepository, aggregatedStatusesRepository,
                        clientId, resume, updateBefore, ppcDslContextProvider);
        Consumer<Result<Long>> resultVisitor = result -> result.getValidationResult()
                .addWarning(adMoreThanOnceInRequest());

        return new IgnoreDuplicatesOperationCreator<>(operationCreator, comparator, resultVisitor)
                .create(bannerChanges)
                .prepareAndApply();
    }
}
