package ru.yandex.direct.core.entity.banner.service;

import java.util.List;
import java.util.Objects;
import java.util.function.BiPredicate;

import com.google.common.collect.ImmutableList;
import one.util.streamex.StreamEx;

import ru.yandex.direct.core.entity.banner.model.BannerWithBody;
import ru.yandex.direct.core.entity.banner.model.BannerWithTitle;
import ru.yandex.direct.core.entity.banner.model.BannerWithTitleExtension;
import ru.yandex.direct.core.entity.banner.model.old.OldBannerWithBody;
import ru.yandex.direct.core.entity.banner.model.old.OldBannerWithTitle;
import ru.yandex.direct.core.entity.banner.model.old.OldBannerWithTitleExtension;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.Model;
import ru.yandex.direct.model.ModelProperty;

import static ru.yandex.direct.common.util.TextUtils.spaceCleaner;

public class BannerTextChangeUtil {


    private BannerTextChangeUtil() {
        // only for static members
    }

    private static final TextModificationChecker<BannerWithTitle> NEW_TITLE_CHECKER =
            defaultChecker(BannerWithTitle.class, BannerWithTitle.TITLE);

    private static final TextModificationChecker<BannerWithTitleExtension> NEW_TITLE_EXTENSION_CHECKER =
            defaultChecker(BannerWithTitleExtension.class, BannerWithTitleExtension.TITLE_EXTENSION);

    private static final TextModificationChecker<BannerWithBody> NEW_BODY_CHECKER =
            defaultChecker(BannerWithBody.class, BannerWithBody.BODY);

    private static final List<TextModificationChecker> ALL_CHECKERS =
            ImmutableList.of(
                    // старый баннер
                    defaultChecker(OldBannerWithTitle.class, OldBannerWithTitle.TITLE),
                    defaultChecker(OldBannerWithTitleExtension.class, OldBannerWithTitleExtension.TITLE_EXTENSION),
                    defaultChecker(OldBannerWithBody.class, OldBannerWithBody.BODY),

                    // новый баннер
                    NEW_TITLE_CHECKER,
                    NEW_TITLE_EXTENSION_CHECKER,
                    NEW_BODY_CHECKER
            );

    public static boolean isBannerTitleChangedSignificantly(AppliedChanges<BannerWithTitle> bannerChanges) {
        return NEW_TITLE_CHECKER.isTextPropertyChangedSignificantly(bannerChanges);
    }

    public static boolean isBannerTitleExtensionChangedSignificantly(AppliedChanges<BannerWithTitleExtension> bannerChanges) {
        return NEW_TITLE_EXTENSION_CHECKER.isTextPropertyChangedSignificantly(bannerChanges);
    }

    public static boolean isBannerBodyChangedSignificantly(AppliedChanges<BannerWithBody> bannerChanges) {
        return NEW_BODY_CHECKER.isTextPropertyChangedSignificantly(bannerChanges);
    }

    public static <T extends Model> boolean isBannerTextChangedSignificantly(AppliedChanges<T> bannerChanges) {
        return StreamEx.of(ALL_CHECKERS)
                .filter(checker -> checker.isApplicable(bannerChanges))
                .anyMatch(checker -> checker.isTextPropertyChangedSignificantly(bannerChanges));
    }

    private static class TextModificationChecker<T extends Model> {

        private Class<T> applicableClass;
        private ModelProperty<? super T, String> propertyToCheck;
        private BiPredicate<String, String> checkerFunction;

        TextModificationChecker(Class<T> applicableClass, ModelProperty<? super T, String> propertyToCheck,
                                BiPredicate<String, String> checkerFunction) {
            this.applicableClass = applicableClass;
            this.propertyToCheck = propertyToCheck;
            this.checkerFunction = checkerFunction;
        }

        public boolean isApplicable(AppliedChanges<? extends Model> bannerChanges) {
            return applicableClass.isAssignableFrom(bannerChanges.getModel().getClass());
        }

        public boolean isTextPropertyChangedSignificantly(AppliedChanges<? extends T> bannerChanges) {
            AppliedChanges<T> castedChanges = bannerChanges.castModelUp(applicableClass);
            String oldValue = castedChanges.getOldValue(propertyToCheck);
            String newValue = castedChanges.getNewValue(propertyToCheck);
            return checkerFunction.test(oldValue, newValue);
        }
    }

    public static <T extends Model> TextModificationChecker<T> defaultChecker(
            Class<T> classToCheck, ModelProperty<? super T, String> propertyToCheck) {
        return new TextModificationChecker<>(classToCheck, propertyToCheck,
                BannerTextChangeUtil::isTextChangedSignificantly);
    }

    private static boolean isTextChangedSignificantly(String oldText, String newText) {
        return !Objects.equals(spaceCleaner(oldText), spaceCleaner(newText));
    }
}
