package ru.yandex.direct.core.entity.banner.service;

import java.time.Duration;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.apache.commons.collections4.ListUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.db.PpcProperty;
import ru.yandex.direct.common.db.PpcPropertyNames;
import ru.yandex.direct.core.entity.banner.container.BannerRepositoryContainer;
import ru.yandex.direct.core.entity.banner.model.BannerPrice;
import ru.yandex.direct.core.entity.banner.model.BannerWithPrice;
import ru.yandex.direct.core.entity.banner.model.BannerWithSystemFields;
import ru.yandex.direct.core.entity.banner.model.BannerWithTurboApp;
import ru.yandex.direct.core.entity.banner.model.TextBanner;
import ru.yandex.direct.core.entity.banner.model.TurboAppInfo;
import ru.yandex.direct.core.entity.banner.model.TurboAppMetaContent;
import ru.yandex.direct.core.entity.banner.repository.BannerRepository;
import ru.yandex.direct.core.entity.banner.type.turboapp.BannerWithHrefAndPriceAndTurboAppUtils;
import ru.yandex.direct.core.entity.banner.type.turboapp.TurboAppsInfoRepository;
import ru.yandex.direct.core.entity.bs.resync.queue.model.BsResyncItem;
import ru.yandex.direct.core.entity.bs.resync.queue.model.BsResyncPriority;
import ru.yandex.direct.core.entity.bs.resync.queue.service.BsResyncService;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithTurboApp;
import ru.yandex.direct.core.entity.campaign.repository.CampaignModifyRepository;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.campaign.repository.CampaignTypedRepository;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.turboapps.client.TurboAppsClient;
import ru.yandex.direct.turboapps.client.TurboAppsClientException;
import ru.yandex.direct.turboapps.client.model.TurboAppInfoRequest;
import ru.yandex.direct.turboapps.client.model.TurboAppInfoResponse;

import static java.util.function.Function.identity;
import static java.util.stream.Collectors.toList;
import static ru.yandex.direct.core.entity.banner.repository.filter.BannerFilterFactory.bannerCampaignIdFilter;
import static ru.yandex.direct.core.entity.banner.service.BannerUtils.getValueIfAssignable;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Service
public class BannerTurboAppService {

    private static final Logger logger = LoggerFactory.getLogger(BannerTurboAppService.class);

    private static final int LOGGING_MAX_AMOUNT = 100;

    private final TurboAppsInfoRepository turboAppsInfoRepository;
    private final TurboAppsClient turboappsClient;
    private final CampaignRepository campaignRepository;
    private final CampaignTypedRepository campaignTypedRepository;
    private final CampaignModifyRepository campaignModifyRepository;
    private final BannerRepository bannerRepository;
    private final FeatureService featureService;
    private final BsResyncService bsResyncService;
    private final PpcProperty<Boolean> updateTurboAppsOnViewProperty;

    @Autowired
    public BannerTurboAppService(TurboAppsInfoRepository turboAppsInfoRepository,
                                 TurboAppsClient turboappsClient,
                                 CampaignRepository campaignRepository,
                                 CampaignTypedRepository campaignTypedRepository,
                                 CampaignModifyRepository campaignModifyRepository,
                                 BannerRepository bannerRepository,
                                 FeatureService featureService,
                                 BsResyncService bsResyncService,
                                 PpcPropertiesSupport ppcPropertiesSupport) {
        this.turboAppsInfoRepository = turboAppsInfoRepository;
        this.turboappsClient = turboappsClient;
        this.campaignRepository = campaignRepository;
        this.campaignTypedRepository = campaignTypedRepository;
        this.campaignModifyRepository = campaignModifyRepository;
        this.bannerRepository = bannerRepository;
        this.featureService = featureService;
        this.bsResyncService = bsResyncService;

        updateTurboAppsOnViewProperty = ppcPropertiesSupport
                .get(PpcPropertyNames.UPDATE_TURBO_APPS_ON_VIEW, Duration.ofMinutes(1));
    }

    /**
     * Отвязывает турбо-аппы всех баннеров кампаний и добавляет затронутые баннеры в bsResyncQueue
     */
    public void detachCampaignTurboApps(int shard, Collection<Long> campaignIds) {
        logger.info("Detaching turbo-apps for {} cids", campaignIds.size());

        Map<Long, Long> campaignIdsByBannerIds =
                bannerRepository.relations.getBannerIdsMapByCampaignIds(shard, campaignIds);

        var bidsWithDetachedTurboApps = detachBannerTurboApps(shard, campaignIdsByBannerIds.keySet());

        var detachResyncItems = mapList(bidsWithDetachedTurboApps,
                bid -> createBsResyncItem(bid, campaignIdsByBannerIds.get(bid)));

        bsResyncService.addObjectsToResync(detachResyncItems);
    }

    /**
     * Обновляет турбо-аппы на всех баннерах переданных кампаний и добавляет затронутые баннеры в bsResyncQueue
     */
    public void updateTurboAppsByCampaignsId(int shard, Collection<Long> campaignIds) {
        Map<ClientId, List<Long>> cidsByClientIds = filterCidsByFeature(shard, campaignIds);
        logger.info("Updating turbo-apps for {} clients", cidsByClientIds.size());

        var updateResyncItems = EntryStream.of(cidsByClientIds).flatMapKeyValue((clientId, clientCids) -> {
            logger.info("Updating turbo-apps in {} campaigns for client {}", clientCids.size(), clientId);

            List<TextBanner> banners = bannerRepository.type.getSafely(shard, bannerCampaignIdFilter(campaignIds),
                    TextBanner.class);

            Set<Long> updated = listToSet(updateNewBannerTurboApps(shard, clientId, banners));

            return StreamEx.of(banners)
                    .filter(banner -> updated.contains(banner.getId()))
                    .map(banner -> createBsResyncItem(banner.getId(), banner.getCampaignId()));
        }).toList();

        bsResyncService.addObjectsToResync(updateResyncItems);
    }

    private BsResyncItem createBsResyncItem(Long bid, Long cid) {
        return new BsResyncItem(BsResyncPriority.UPDATE_BANNER_WITH_ADDITIONS, cid, bid, null);
    }

    /**
     * Возвращает мета-контент турбо-аппов, привязаннык к баннерам.
     * <p>
     * Если включена пропертя UPDATE_TURBO_APPS_ON_VIEW,
     * ходит в ручку и обновляет привязанные турбо-аппы в базе.
     *
     * @return map[bid -> turbo-app meta content]
     */
    public Map<Long, TurboAppMetaContent> updateAndGetTurboApps(int shard, ClientId clientId,
                                                                Map<Long, Long> cidsByBannerIds) {
        Set<Long> campaignsWithTurboAppsEnables =
                campaignRepository.getCampaignsWithTurboAppsEnables(shard, listToSet(cidsByBannerIds.values()));
        Set<Long> bannerIds = cidsByBannerIds.keySet();
        List<Long> bannersWithTurboAppsAllowed =
                filterList(bannerIds, bid -> campaignsWithTurboAppsEnables.contains(cidsByBannerIds.get(bid)));

        if (updateTurboAppsOnViewProperty.getOrDefault(false)) {
            List<TextBanner> banners = bannerRepository.type.getSafely(shard, bannersWithTurboAppsAllowed,
                    TextBanner.class);

            Set<Long> updated = listToSet(updateNewBannerTurboApps(shard, clientId, banners));
            bannerRepository.common.resetStatusBsSyncedByIds(shard, updated);
        }

        return getTurboAppMetaContent(shard, bannersWithTurboAppsAllowed);
    }

    /**
     * Обновляет информацию о турбо-аппах баннеров.
     * <ol>
     *     <li>Если href был удален, турбоапп удаляется</li>
     *     <li>Для остальных идем в ручку</li>
     *     <li>Если информации о турбо-аппе нету, то он удаляется</li>
     *     <li>Иначе вставляем турбо-апп и соответствующий {@link BannerWithTurboApp}</li>
     *     <li>Если турбо-апп уже был в базе, обновляет content и metaContent</li>
     * </ol>
     *
     * @return Список затронутых баннеров
     */
    public List<Long> updateNewBannerTurboApps(int shard, ClientId clientId, List<TextBanner> banners) {
        if (banners.isEmpty()) {
            return Collections.emptyList();
        }

        Map<Long, String> bannerHrefByBid = StreamEx.of(banners)
                .filter(banner -> banner.getHref() != null)
                .toMap(TextBanner::getId, TextBanner::getHref);

        Map<Long, TurboAppInfoResponse> turboappsClientResponse = getTurboAppsInfo(bannerHrefByBid);

        List<Long> bannerIds = mapList(banners, TextBanner::getId);

        // Удаляем если не вернулся турбоапп в ответе ручки (в частности, если нету хрефа)
        List<Long> toDelete = StreamEx.of(bannerIds)
                .remove(turboappsClientResponse::containsKey)
                .toList();
        var bidsWithDeletedTurboApps = detachBannerTurboApps(shard, toDelete);

        // Обновляем(или добавляем) если ручка вернула результат
        List<BannerWithSystemFields> toUpdate = StreamEx.of(banners)
                .filter(banner -> turboappsClientResponse.containsKey(banner.getId()))
                .select(BannerWithSystemFields.class)
                .toList();
        var bidsWithUpdatedTurboApps = updateNewBannerTurboApps(shard, clientId, toUpdate,
                turboappsClientResponse);

        return ListUtils.union(bidsWithDeletedTurboApps, bidsWithUpdatedTurboApps);
    }

    public List<Long> updateNewBannerTurboApps(int shard, ClientId clientId, List<BannerWithSystemFields> banners,
                                               Map<Long, TurboAppInfoResponse> turboAppsInfoResponse) {

        var turboAppInfoByAppId = upsertTurboApps(shard, clientId, turboAppsInfoResponse.values());

        List<AppliedChanges<BannerWithSystemFields>> appliedChangesList = banners.stream()
                .filter(x -> x instanceof BannerWithTurboApp)
                .map(x -> (BannerWithTurboApp) x)
                .map(banner -> {
                    return createModelChanges(banner, turboAppsInfoResponse, turboAppInfoByAppId)
                            .applyTo(banner)
                            .castModelUp(BannerWithSystemFields.class);
                })
                .filter(AppliedChanges::hasActuallyChangedProps)
                .collect(toList());

        bannerRepository.modification.update(new BannerRepositoryContainer(shard), appliedChangesList);
        return mapList(appliedChangesList, x -> x.getModel().getId());
    }

    private ModelChanges<BannerWithTurboApp> createModelChanges(BannerWithTurboApp banner,
                                                                Map<Long, TurboAppInfoResponse> turboAppsInfoResponse,
                                                                Map<Long, TurboAppInfo> turboAppInfoByAppId) {

        Long bannerId = banner.getId();
        TurboAppInfoResponse turboAppInfoResponse = turboAppsInfoResponse.get(bannerId);

        TurboAppInfo turboAppInfo = turboAppInfoByAppId.get(turboAppInfoResponse.getAppId());
        Long turboAppInfoId = ifNotNull(turboAppInfo, TurboAppInfo::getTurboAppInfoId);

        BannerPrice bannerPrice = getValueIfAssignable(banner, BannerWithPrice.BANNER_PRICE);

        return new ModelChanges<>(bannerId, BannerWithTurboApp.class)
                .process(turboAppInfoId, BannerWithTurboApp.TURBO_APP_INFO_ID)
                .process(BannerWithHrefAndPriceAndTurboAppUtils.countBannerTurboAppType(bannerPrice),
                        BannerWithTurboApp.TURBO_APP_TYPE)
                .process(turboAppInfoResponse.getContent(), BannerWithTurboApp.TURBO_APP_CONTENT);
    }

    /**
     * Добавляет турбо-аппы, либо обновляет контент уже существующих
     *
     * @return map[app id -> turbo app info], удаленные/добавленные турбо-аппы
     */
    public Map<Long, TurboAppInfo> upsertTurboApps(int shard, ClientId clientId,
                                                   Collection<TurboAppInfoResponse> turboApps) {

        Map<Long, TurboAppInfo> turboAppInfoByAppId = StreamEx.of(turboApps)
                .distinct(TurboAppInfoResponse::getAppId)
                .toMap(TurboAppInfoResponse::getAppId, info -> new TurboAppInfo()
                        .withTurboAppId(info.getAppId())
                        .withClientId(clientId.asLong())
                        .withContent(info.getMetaContent()));

        List<TurboAppInfo> existedTurboAppsInfo = turboAppsInfoRepository.getTurboAppInfoByTurboAppIds(shard,
                turboAppInfoByAppId.keySet(), List.of(clientId));

        enrichTurboAppInfo(turboAppInfoByAppId, existedTurboAppsInfo);

        // Обновляем/добавляем turboAppInfoByAppId
        List<TurboAppInfo> newOrUpdatedTurboApps
                = filterExistedTurboApps(turboAppInfoByAppId.values(), existedTurboAppsInfo);
        turboAppsInfoRepository.addOrUpdateTurboAppInfo(shard, newOrUpdatedTurboApps);

        return turboAppInfoByAppId;
    }

    /**
     * Отвязывает турбо-аппы баннеров (удаляет записи из banner_turbo_apps)
     *
     * @return список bid, у которых были удалены турбо-аппы
     */
    public List<Long> detachBannerTurboApps(int shard, Collection<Long> bannerIds) {
        if (bannerIds.isEmpty()) {
            return Collections.emptyList();
        }

        Set<Long> existedBidsWithTurboApps = bannerRepository.turboApps.getExistedBidsWithTurboApps(shard, bannerIds);

        var filteredBids = filterList(bannerIds, existedBidsWithTurboApps::contains);
        bannerRepository.turboApps.deleteBannerTurboApps(shard, filteredBids);
        return filteredBids;
    }

    private Map<Long, TurboAppMetaContent> getTurboAppMetaContent(int shard, List<Long> bannerIds) {
        var turboApps = turboAppsInfoRepository.getTurboAppInfoByBannerIds(shard, bannerIds);
        return EntryStream.of(turboApps)
                .mapValues(TurboAppInfo::getContent)
                .mapValues(turboappsClient::tryParseMetaContent)
                .mapValues(metaContent -> ifNotNull(metaContent, mc -> new TurboAppMetaContent()
                        .withName(mc.getName())
                        .withDescription(mc.getDescription())
                        .withIconUrl(mc.getIconUrl())))
                .toMap();
    }

    /**
     * Возвращает информацию о турбо-аппах баннеров по их href.
     *
     * @param bannerHrefByBannerId map[key -> href]
     * @return map[key -> turbo_app_info]
     */
    public Map<Long, TurboAppInfoResponse> getTurboAppsInfo(Map<Long, String> bannerHrefByBannerId) {
        try {
            return turboappsClient.getTurboApps(EntryStream.of(bannerHrefByBannerId)
                    .mapKeyValue(TurboAppInfoRequest::new)
                    .toList());
        } catch (TurboAppsClientException e) {
            String bannersString = EntryStream.of(bannerHrefByBannerId)
                    .limit(LOGGING_MAX_AMOUNT)
                    .mapKeyValue((bid, href) -> String.format("{%d, %s}", bid, href))
                    .joining(", ");
            logger.error("Failed to get turbo apps for banners {}", bannersString, e);
            return Collections.emptyMap();
        }
    }

    /**
     * Добавляем информацию о существующих turboAppInfoId
     */
    private void enrichTurboAppInfo(Map<Long, TurboAppInfo> turboAppsInfoByBannerIds,
                                    List<TurboAppInfo> existedTurboAppsInfo) {
        Map<Long, List<TurboAppInfo>> existedTurboAppsByTurboAppIds = StreamEx.of(existedTurboAppsInfo)
                .groupingBy(TurboAppInfo::getTurboAppId);

        turboAppsInfoByBannerIds.forEach((bid, turbo) -> {
            Long turboAppId = turbo.getTurboAppId();
            if (existedTurboAppsByTurboAppIds.containsKey(turboAppId)) {
                StreamEx.of(existedTurboAppsByTurboAppIds.get(turboAppId))
                        .findFirst(t -> t.getClientId().equals(turbo.getClientId()))
                        .ifPresent(t -> turbo.setTurboAppInfoId(t.getTurboAppInfoId()));
            }
        });
    }

    /**
     * Оставляем только новые, либо изменившиеся турбо-аппы
     */
    private List<TurboAppInfo> filterExistedTurboApps(Collection<TurboAppInfo> turboAppsInfo,
                                                      List<TurboAppInfo> existedTurboAppsInfo) {
        Map<Long, TurboAppInfo> existedInfoByIds = listToMap(existedTurboAppsInfo, TurboAppInfo::getTurboAppInfoId);

        return StreamEx.of(turboAppsInfo)
                .remove(toInsert -> toInsert.equals(existedInfoByIds.get(toInsert.getTurboAppInfoId())))
                .toList();
    }

    public void setHasTurboAppForCampaigns(int shard, Collection<Long> campaignIds, boolean hasTurboApp) {
        var typedCampaigns = campaignTypedRepository.getTypedCampaigns(shard, campaignIds);

        var changes = StreamEx.of(typedCampaigns)
                .select(CampaignWithTurboApp.class)
                .distinct(CampaignWithTurboApp::getId)
                .map(campaign -> new ModelChanges<>(campaign.getId(), CampaignWithTurboApp.class)
                        .process(hasTurboApp, CampaignWithTurboApp.HAS_TURBO_APP)
                        .applyTo(campaign))
                .toList();

        if (!changes.isEmpty()) {
            campaignModifyRepository.updateCampaignsTable(shard, changes);
        }
    }

    /**
     * @return {@code map[client id, list[cid]]} &mdash; кампании клиентов,
     * у которых есть фича {@code turbo_app_allowed}
     */
    private Map<ClientId, List<Long>> filterCidsByFeature(int shard, Collection<Long> toUpdate) {
        Map<Long, ClientId> clientIdsForCids = EntryStream
                .of(campaignRepository.getClientIdsForCids(shard, toUpdate))
                .mapValues(ClientId::fromLong)
                .toMap();

        Set<ClientId> allClients = listToSet(clientIdsForCids.values());
        Map<ClientId, Boolean> isFeatureEnabled =
                featureService.isEnabledForClientIdsOnlyFromDb(allClients, FeatureName.TURBO_APP_ALLOWED.getName());

        return StreamEx.of(clientIdsForCids.keySet())
                .mapToEntry(clientIdsForCids::get, identity())
                .filterKeys(isFeatureEnabled::get)
                .grouping();
    }
}
