package ru.yandex.direct.core.entity.banner.service;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.regex.Matcher;

import one.util.streamex.StreamEx;

import ru.yandex.direct.core.entity.banner.model.Banner;
import ru.yandex.direct.core.entity.banner.model.BannerWithCampaignId;
import ru.yandex.direct.core.entity.banner.model.BannerWithOrganization;
import ru.yandex.direct.core.entity.banner.model.BannerWithSystemFields;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.Model;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.model.ModelProperty;
import ru.yandex.direct.utils.CommonUtils;

import static com.google.common.base.Preconditions.checkState;
import static java.util.stream.Collectors.toList;
import static ru.yandex.direct.core.entity.banner.service.validation.BannerLettersConstants.TEMPLATE_PATTERN;
import static ru.yandex.direct.dbutil.SqlUtils.ID_NOT_SET;

public class BannerUtils {
    private BannerUtils() {
    }

    public static <T extends BannerWithCampaignId> Map<Long, List<Long>> getCampaignIdToBannerIds(
            Collection<T> banners,
            Predicate<T> predicate) {
        return StreamEx.of(banners)
                .filter(predicate)
                .mapToEntry(BannerWithCampaignId::getCampaignId, BannerWithCampaignId::getId)
                .grouping();
    }

    public static <T extends BannerWithOrganization> List<Long> getValidPermalinkIds(
            Collection<T> banners) {
        return StreamEx.of(banners)
                .map(BannerWithOrganization::getPermalinkId)
                .filter(CommonUtils::isValidId)
                .toList();
    }

    public static <B extends BannerWithSystemFields, T extends Banner> Optional<ModelChanges<T>> ifModelTypeIs(
            ModelChanges<B> mc,
            Class<T> type) {
        return Optional.of(mc).filter(modelChangesOfType(type)).map(newModelChangesToType(type));
    }

    @SuppressWarnings("unchecked")
    public static <B extends BannerWithSystemFields, T extends Banner> Function<ModelChanges<B>, ModelChanges<T>> newModelChangesToType(
            Class<T> targetClass) {
        return mc -> {
            checkState(targetClass.isAssignableFrom(mc.getModelType()), "Filter model changes before mapping!");
            return (ModelChanges<T>) mc;
        };
    }

    private static <B extends BannerWithSystemFields, T extends Banner> Predicate<ModelChanges<B>> modelChangesOfType(
            Class<T> filterType) {
        return mc -> filterType.isAssignableFrom(mc.getModelType());
    }

    @SuppressWarnings("unchecked")
    public static <A extends Banner, B extends Banner> List<AppliedChanges<B>> selectAppliedChanges(
            Collection<AppliedChanges<A>> appliedChanges, Class<B> filterType) {
        return (List) appliedChanges.stream()
                .filter(ac -> filterType.isAssignableFrom(ac.getModel().getClass()))
                .collect(toList());
    }

    public static boolean bannerNeverSentToBs(BannerWithSystemFields banner) {
        return Objects.equals(banner.getBsBannerId(), ID_NOT_SET);
    }

    /**
     * Достать значение modelProperty, если у баннера есть такое поле.
     * Иначе null.
     */
    public static <T extends Model, V> V getValueIfAssignable(Banner banner, ModelProperty<T, V> modelProperty) {

        if (banner == null || !modelProperty.getModelClass().isAssignableFrom(banner.getClass())) {
            return null;
        }

        return modelProperty.get((T) banner);
    }

    public static int templateCountIn(String text) {
        Matcher matcher = TEMPLATE_PATTERN.matcher(text);
        int matchCount = 0;
        while (matcher.find()) {
            matchCount++;
        }

        return matchCount;
    }
}
