package ru.yandex.direct.core.entity.banner.service;

import java.util.List;
import java.util.Map;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.banner.container.BannerAdditionalActionsContainer;
import ru.yandex.direct.core.entity.banner.container.BannerRepositoryContainer;
import ru.yandex.direct.core.entity.banner.container.BannersAddOperationContainerImpl;
import ru.yandex.direct.core.entity.banner.model.BannerWithAdGroupId;
import ru.yandex.direct.core.entity.banner.service.execution.BannersAddExecutionGrutService;
import ru.yandex.direct.core.entity.banner.service.execution.BannersAddExecutionMysqlService;
import ru.yandex.direct.dbutil.sharding.ShardHelper;

import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Service
public class BannersAddExecutionFacade {

    private static final Logger logger = LoggerFactory.getLogger(BannersAddExecutionFacade.class);

    private final BannersAddExecutionMysqlService bannersAddExecutionMysqlService;
    private final BannersAddExecutionGrutService bannersAddExecutionGrutService;
    private final ShardHelper shardHelper;

    @Autowired
    public BannersAddExecutionFacade(BannersAddExecutionMysqlService bannersAddExecutionMysqlService,
                                     BannersAddExecutionGrutService bannersAddExecutionGrutService,
                                     ShardHelper shardHelper) {
        this.bannersAddExecutionMysqlService = bannersAddExecutionMysqlService;
        this.bannersAddExecutionGrutService = bannersAddExecutionGrutService;
        this.shardHelper = shardHelper;
    }

    public List<Long> execute(List<BannerWithAdGroupId> validBannersToApply,
                              BannersAddOperationContainerImpl operationContainer,
                              BannerRepositoryContainer repositoryContainer,
                              BannerAdditionalActionsContainer additionalActionsContainer) {
        switch (operationContainer.getDatabaseMode()) {
            case ONLY_MYSQL:
                return bannersAddExecutionMysqlService.execute(validBannersToApply, operationContainer,
                        repositoryContainer,
                        additionalActionsContainer);
            case ONLY_GRUT:
                //generate bannerIds in mysql
                var ids = shardHelper.generateBannerIds(mapList(validBannersToApply,
                                BannerWithAdGroupId::getAdGroupId))
                        .iterator();
                validBannersToApply.forEach(banner -> banner.withId(ids.next()));

                return bannersAddExecutionGrutService.execute(validBannersToApply, operationContainer,
                        repositoryContainer,
                        additionalActionsContainer);
            case MYSQL_AND_GRUT:
                //сохраняем сначала в mysql, чтобы получить идентификаторы
                var bannerIds = bannersAddExecutionMysqlService.execute(validBannersToApply, operationContainer,
                        repositoryContainer,
                        additionalActionsContainer);
                try {

                    //сохраняем в грут с идентификаторами из mysql
                    bannersAddExecutionGrutService.execute(validBannersToApply, operationContainer, repositoryContainer,
                            additionalActionsContainer);
                } catch (RuntimeException ex) {
                    //не показываем пользователю ошибку, т.к. в mysql сохранились корректные данные
                    logger.error(String.format("error in banner grut execute for bannerIds %s", bannerIds), ex);
                }
                return bannerIds;
            default:
                throw new IllegalArgumentException("Unsupported database mode: " + operationContainer.getDatabaseMode());
        }
    }

    public void afterExecution(Map<Integer, BannerWithAdGroupId> validModelsMapToApply,
                               BannersAddOperationContainerImpl operationContainer) {
        switch (operationContainer.getDatabaseMode()) {
            case ONLY_MYSQL:
                bannersAddExecutionMysqlService.afterExecution(validModelsMapToApply, operationContainer);
            case ONLY_GRUT:
                bannersAddExecutionGrutService.afterExecution(validModelsMapToApply, operationContainer);
            case MYSQL_AND_GRUT:
                bannersAddExecutionGrutService.afterExecution(validModelsMapToApply, operationContainer);
                bannersAddExecutionMysqlService.afterExecution(validModelsMapToApply, operationContainer);
        }
    }
}
