package ru.yandex.direct.core.entity.banner.service;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.core.entity.adgroup.model.AdGroupForBannerOperation;
import ru.yandex.direct.core.entity.adgroup.model.ContentPromotionAdgroupType;
import ru.yandex.direct.core.entity.banner.container.BannerAdditionalActionsContainer;
import ru.yandex.direct.core.entity.banner.container.BannerRepositoryContainer;
import ru.yandex.direct.core.entity.banner.container.BannersAddOperationContainerImpl;
import ru.yandex.direct.core.entity.banner.container.BannersAddOperationContainerService;
import ru.yandex.direct.core.entity.banner.model.BannerWithAdGroupId;
import ru.yandex.direct.core.entity.banner.model.BannerWithSitelinks;
import ru.yandex.direct.core.entity.banner.model.BannerWithVcard;
import ru.yandex.direct.core.entity.banner.service.moderation.ModerationMode;
import ru.yandex.direct.core.entity.banner.service.type.add.BannerAddOperationTypeSupportFacade;
import ru.yandex.direct.core.entity.banner.service.validation.BannerValidationInfo;
import ru.yandex.direct.core.entity.banner.service.validation.type.add.BannerAddValidationTypeSupportFacade;
import ru.yandex.direct.core.entity.client.repository.ClientRepository;
import ru.yandex.direct.core.entity.sitelink.model.SitelinkSet;
import ru.yandex.direct.core.entity.user.repository.UserRepository;
import ru.yandex.direct.core.entity.vcard.model.Vcard;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.operation.Applicability;
import ru.yandex.direct.operation.OperationsUtils;
import ru.yandex.direct.operation.add.AbstractAddOperation;
import ru.yandex.direct.operation.add.ModelsPreValidatedStep;
import ru.yandex.direct.operation.add.ModelsValidatedStep;
import ru.yandex.direct.rbac.RbacRole;
import ru.yandex.direct.rbac.RbacService;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static com.google.common.base.Preconditions.checkState;
import static ru.yandex.direct.core.entity.banner.service.BannerLogUtils.logBannersCount;
import static ru.yandex.direct.operation.Applicability.isPartial;
import static ru.yandex.direct.validation.result.ValidationResult.getValidItems;

@ParametersAreNonnullByDefault
public class BannersAddOperation extends AbstractAddOperation<BannerWithAdGroupId, Long> {

    private static final Logger logger = LoggerFactory.getLogger(BannersAddOperation.class);

    private final BannersAddOperationContainerService bannersOperationContainerService;
    private final BannerAddValidationTypeSupportFacade validationTypeSupportFacade;
    private final BannerAddOperationTypeSupportFacade addOperationTypeSupportFacade;
    private final BannersAddExecutionFacade bannersAddExecutionService;

    private final BannersAddOperationContainerImpl operationContainer;
    private final BannerRepositoryContainer repositoryContainer;
    private final BannerAdditionalActionsContainer additionalActionsContainer;


    public BannersAddOperation(Applicability applicability, boolean partOfComplexOperation,
                               List<BannerWithAdGroupId> bannersToAdd,
                               UserRepository userRepository,
                               ClientRepository clientRepository,
                               BannersAddOperationContainerService bannersOperationContainerService,
                               BannersAddExecutionFacade bannersAddExecutionService,
                               BannerAddValidationTypeSupportFacade validationTypeSupportFacade,
                               BannerAddOperationTypeSupportFacade addOperationTypeSupportFacade,
                               RbacService rbacService,
                               int shard, ClientId clientId, Long operatorUid,
                               ModerationMode moderationMode,
                               boolean isCopyAction,
                               boolean isFromApi,
                               boolean isUcPreValidation,
                               DatabaseMode databaseMode,
                               RbacRole operatorRole,
                               Set<String> clientEnabledFeatures) {
        super(applicability, bannersToAdd);

        this.bannersOperationContainerService = bannersOperationContainerService;
        this.bannersAddExecutionService = bannersAddExecutionService;
        this.validationTypeSupportFacade = validationTypeSupportFacade;
        this.addOperationTypeSupportFacade = addOperationTypeSupportFacade;

        Long uid = userRepository.getChiefUidByClientId(shard, clientId.asLong());
        Long clientRegionId = clientRepository.getCountryRegionIdByClientId(shard, clientId).orElse(null);
        var client = clientRepository.get(shard, List.of(clientId)).get(0);

        this.operationContainer = new BannersAddOperationContainerImpl(shard, operatorUid, operatorRole, clientId, uid,
                rbacService.getChiefByClientId(clientId), clientRegionId, clientEnabledFeatures, moderationMode,
                operatorRole.isInternal(), partOfComplexOperation, true);
        operationContainer.setClient(client);
        operationContainer.setFromApi(isFromApi);
        operationContainer.setCopy(isCopyAction);
        operationContainer.setUcPreValidation(isUcPreValidation);
        operationContainer.setDatabaseMode(databaseMode);
        this.repositoryContainer = new BannerRepositoryContainer(shard);
        this.additionalActionsContainer = new BannerAdditionalActionsContainer(clientId, clientRegionId);
    }

    /**
     * Устанавливает всем баннерам id группы.
     * Баннерам можно не проставлять adGroupId в случае, если гарантируется, что их добавление будет отменено.
     *
     * @param adGroupIdsByIndex мапа (индекс баннера) -> (adGroupId)
     */
    public void setAdGroupIds(Map<Integer, Long> adGroupIdsByIndex) {
        checkIsNotExecuted();
        for (int i = 0; i < getModels().size(); i++) {
            Long adGroupId = adGroupIdsByIndex.get(i);
            if (adGroupId != null) {
                getModels().get(i).setAdGroupId(adGroupId);
            } else {
                checkState(isPartial(applicability), "adGroupId for banners[%s] is not defined", i);
            }
        }
    }

    /**
     * Выставляет указанным баннерам id визитки.
     *
     * @param vcardByIndex мапа (индекс баннера) -> (vcard)
     */
    public void setVcards(Map<Integer, Vcard> vcardByIndex) {
        checkIsNotExecuted();
        var vcardIdToData = EntryStream.of(vcardByIndex)
                .nonNullValues()
                .mapToKey((k, v) -> v.getId())
                .append(operationContainer.getVcardIdToData())
                .distinctKeys()
                .toMap();
        operationContainer.setVcardIdToData(vcardIdToData);

        Map<Integer, Long> vcardIdsByIndex = new HashMap<>();
        vcardByIndex.forEach((k, v) -> vcardIdsByIndex.put(k, v != null ? v.getId() : null));
        modifyProperty(BannerWithVcard.VCARD_ID, vcardIdsByIndex);
    }

    /**
     * Выставляет указанным баннерам id набора сайтлинков.
     *
     * @param sitelinkSetsByIndex мапа (индекс баннера) -> (sitelinkSet)
     */
    public void setSitelinkSets(Map<Integer, SitelinkSet> sitelinkSetsByIndex) {
        checkIsNotExecuted();

        var sitelinkIdToData = EntryStream.of(sitelinkSetsByIndex)
                .nonNullValues()
                .mapToKey((k, v) -> v.getId())
                .append(operationContainer.getSitelinkSets())
                .distinctKeys()
                .toMap();
        operationContainer.setSitelinkSets(new ArrayList<>(sitelinkIdToData.values()));

        Map<Integer, Long> sitelinkIdsByIndex = new HashMap<>();
        sitelinkSetsByIndex.forEach((k, v) -> sitelinkIdsByIndex.put(k, v != null ? v.getId() : null));
        modifyProperty(BannerWithSitelinks.SITELINKS_SET_ID, sitelinkIdsByIndex);
    }

    /**
     * Сохранение информации о баннерах, приходящей из комплексной операции
     * Необходимо для корректной валидации
     *
     * @param bannersValidationInfoMap мапа индексов баннеры в информацию о них
     */
    public void setBannerValidationInfo(Map<Integer, BannerValidationInfo> bannersValidationInfoMap) {
        checkIsNotPrepared();
        operationContainer.setBannersValidationInfoMap(bannersValidationInfoMap);
    }

    private void checkIsNotExecuted() {
        checkState(!isExecuted(), "operation is already executed!");
    }

    public void setAdGroupsInfo(Map<Integer, AdGroupForBannerOperation> adGroups) {
        checkIsNotPrepared();
        bannersOperationContainerService.initializeAdGroupsAndCampaignMapsInContainers(operationContainer, adGroups);
        operationContainer.setAdGroupExternal(true);
    }

    /**
     * Сохранение информации о баннерах, приходящей из комплексной операции обновления контента
     * Необходимо для корректной валидации
     *
     * @param contentPromotionAdgroupTypeMap мапа индексов баннеров в информацию о них
     */
    public void setContentPromotionAdGroupTypeMap(
            Map<Integer, ContentPromotionAdgroupType> contentPromotionAdgroupTypeMap) {
        checkIsNotPrepared();
        operationContainer.setIndexToContentPromotionAdgroupTypeMap(() -> contentPromotionAdgroupTypeMap);
    }

    private void checkIsNotPrepared() {
        checkState(!isPrepared(), "operation is already prepared!");
    }

    @Override
    protected ValidationResult<List<BannerWithAdGroupId>, Defect> preValidate(List<BannerWithAdGroupId> models) {
        logBannersCount(logger, "All banners in preValidate {}", models);
        bannersOperationContainerService.fillContainers(operationContainer, models);

        var vr = new ValidationResult<>(models, Defect.class);
        validationTypeSupportFacade.preValidate(operationContainer, vr);
        return vr;
    }

    @Override
    protected void onPreValidated(ModelsPreValidatedStep<BannerWithAdGroupId> modelsPreValidatedStep) {
        List<BannerWithAdGroupId> preValidItems = getValidItems(modelsPreValidatedStep.getValidationResult());
        addOperationTypeSupportFacade.onPreValidated(operationContainer, preValidItems);
    }

    @Override
    protected void validate(ValidationResult<List<BannerWithAdGroupId>, Defect> preValidationResult) {
        validationTypeSupportFacade.validate(operationContainer, preValidationResult);
    }

    @Override
    protected void onModelsValidated(ModelsValidatedStep<BannerWithAdGroupId> modelsValidatedStep) {
        var validModelsMap = modelsValidatedStep.getValidModelsMap();
        var validModels = new ArrayList<>(validModelsMap.values());
        logBannersCount(logger, modelsValidatedStep.getModels(), validModelsMap.values());
        addOperationTypeSupportFacade.onModelsValidated(operationContainer, validModels);
    }

    @Override
    protected void beforeExecution(Map<Integer, BannerWithAdGroupId> validModelsMapToApply) {
        var validBannersToApply = new ArrayList<>(validModelsMapToApply.values());
        logBannersCount(logger, "Valid banners in beforeExecution: {}", validBannersToApply);
        addOperationTypeSupportFacade.beforeExecution(operationContainer, validBannersToApply);
    }

    @Override
    protected Map<Integer, Long> execute(Map<Integer, BannerWithAdGroupId> validModelsMapToApply) {
        return OperationsUtils.applyForMapValues(validModelsMapToApply, this::execute);
    }

    private List<Long> execute(List<BannerWithAdGroupId> validBannersToApply) {
        addOperationTypeSupportFacade.addToAdditionalActionsContainer(additionalActionsContainer,
                operationContainer, validBannersToApply);

        return bannersAddExecutionService.execute(validBannersToApply, operationContainer,
                repositoryContainer, additionalActionsContainer);
    }

    @Override
    protected void onExecuted(Map<Integer, BannerWithAdGroupId> validModelsMapToApply) {
        bannersAddExecutionService.afterExecution(validModelsMapToApply, operationContainer);
    }

}
