package ru.yandex.direct.core.entity.banner.service;

import java.time.LocalDateTime;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import org.jooq.DSLContext;
import org.jooq.TransactionalRunnable;

import ru.yandex.direct.core.aggregatedstatuses.repository.AggregatedStatusesRepository;
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.core.entity.banner.container.BannerRepositoryContainer;
import ru.yandex.direct.core.entity.banner.model.BannerWithSystemFields;
import ru.yandex.direct.core.entity.banner.model.InternalBanner;
import ru.yandex.direct.core.entity.banner.model.NewStatusImageModerate;
import ru.yandex.direct.core.entity.banner.model.PerformanceBanner;
import ru.yandex.direct.core.entity.banner.model.StatusBannerImageModerate;
import ru.yandex.direct.core.entity.banner.model.TextBanner;
import ru.yandex.direct.core.entity.banner.repository.BannerModerationRepository;
import ru.yandex.direct.core.entity.banner.repository.BannerModifyRepository;
import ru.yandex.direct.core.entity.banner.repository.BannerTypedRepository;
import ru.yandex.direct.core.entity.banner.service.moderation.BannerModerationHelper;
import ru.yandex.direct.core.entity.banner.service.validation.ArchiveUnarchiveBannerValidationService;
import ru.yandex.direct.core.entity.banner.type.image.BannerImageRepository;
import ru.yandex.direct.core.entity.campaign.model.CampaignStatusModerate;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.operation.Applicability;
import ru.yandex.direct.operation.update.AppliedChangesValidatedStep;
import ru.yandex.direct.operation.update.ExecutionStep;
import ru.yandex.direct.operation.update.SimpleAbstractUpdateOperation;
import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.tracing.TraceProfile;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.core.entity.banner.service.validation.BannerConstants.NEW_SENSITIVE_PROPERTIES;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Операция для архивации/разархивации баннеров. Является частным случаем операции Update.
 */
public class BannersArchiveUnarchiveOperation extends SimpleAbstractUpdateOperation<BannerWithSystemFields, Long> {
    private final ArchiveUnarchiveBannerValidationService validationService;
    private final AdGroupRepository adGroupRepository;
    private final BannerTypedRepository typedRepository;
    private final BannerModifyRepository modifyRepository;
    private final BannerModerationRepository bannerModerationRepository;
    private final CampaignRepository campaignRepository;
    private final BannerImageRepository imageRepository;
    private final AggregatedStatusesRepository aggregatedStatusesRepository;
    private final int shard;
    private final Long operatorUid;
    private final ClientId clientId;
    private final boolean archive;
    private final LocalDateTime updateBefore;
    private final DslContextProvider ppcDslContextProvider;
    private final BannerModerationHelper moderationHelper;
    private final Set<String> featuresEnabledForClient;
    private final BannerRepositoryContainer bannerRepositoryContainer;

    private TransactionalRunnable afterExecute;

    @SuppressWarnings("checkstyle:parameternumber")
    public BannersArchiveUnarchiveOperation(int shard, Long operatorUid,
                                            List<ModelChanges<BannerWithSystemFields>> modelChanges,
                                            ArchiveUnarchiveBannerValidationService validationService,
                                            AdGroupRepository adGroupRepository,
                                            BannerTypedRepository typedRepository,
                                            BannerModifyRepository modifyRepository,
                                            BannerModerationRepository bannerModerationRepository,
                                            CampaignRepository campaignRepository,
                                            AggregatedStatusesRepository aggregatedStatusesRepository,
                                            ClientId clientId,
                                            boolean archive,
                                            LocalDateTime updateBefore,
                                            DslContextProvider ppcDslContextProvider,
                                            BannerModerationHelper moderationHelper,
                                            Set<String> featuresEnabledForClient,
                                            BannerImageRepository imageRepository) {
        super(Applicability.PARTIAL, modelChanges, id -> new TextBanner().withId(id),
                NEW_SENSITIVE_PROPERTIES);
        this.validationService = validationService;
        this.adGroupRepository = adGroupRepository;
        this.shard = shard;
        this.operatorUid = operatorUid;
        this.typedRepository = typedRepository;
        this.modifyRepository = modifyRepository;
        this.bannerModerationRepository = bannerModerationRepository;
        this.campaignRepository = campaignRepository;
        this.aggregatedStatusesRepository = aggregatedStatusesRepository;
        this.clientId = clientId;
        this.archive = archive;
        this.updateBefore = updateBefore;
        this.ppcDslContextProvider = ppcDslContextProvider;
        this.moderationHelper = moderationHelper;
        this.featuresEnabledForClient = featuresEnabledForClient;
        this.imageRepository = imageRepository;

        this.afterExecute = conf -> {
        };

        bannerRepositoryContainer = new BannerRepositoryContainer(shard);
    }

    @Override
    protected ValidationResult<List<ModelChanges<BannerWithSystemFields>>, Defect> validateModelChangesBeforeApply(
            ValidationResult<List<ModelChanges<BannerWithSystemFields>>, Defect> preValidateResult,
            Map<Long, BannerWithSystemFields> models) {
        try (TraceProfile ignore = Trace.current()
                .profile("bannersArchiveUnarchiveOperation:validateModelChangesBeforeApply")) {
            return validationService.validateBanners(shard, preValidateResult, models, archive);
        }
    }

    @Override
    protected ValidationResult<List<ModelChanges<BannerWithSystemFields>>, Defect> validateModelChanges(
            List<ModelChanges<BannerWithSystemFields>> modelChanges) {
        try (TraceProfile ignore = Trace.current().profile("bannersArchiveUnarchiveOperation:validateModelChanges")) {
            return validationService.validateChanges(clientId, operatorUid, modelChanges);
        }
    }

    @Override
    protected Collection<BannerWithSystemFields> getModels(Collection<Long> ids) {
        try (TraceProfile ignore = Trace.current().profile("bannersArchiveUnarchiveOperation:getModels")) {
            return typedRepository.getStrictlyFullyFilled(shard, ids, BannerWithSystemFields.class);
        }
    }

    @Override
    protected void onAppliedChangesValidated(AppliedChangesValidatedStep<BannerWithSystemFields> step) {
        try (TraceProfile ignore = Trace.current()
                .profile("bannersArchiveUnarchiveOperation:onAppliedChangesValidated")) {
            Collection<AppliedChanges<BannerWithSystemFields>> changes = filterList(step.getValidAppliedChanges(),
                    AppliedChanges::hasActuallyChangedProps);
            moderationHelper.setAdditionalFlags(changes, archive);
            afterExecute = createAfterExecuteFlagsUpdate(changes);
        }
    }

    /**
     * Создает {@link TransactionalRunnable}, выполняемый после архивации или разархивации баннеров.
     */
    private TransactionalRunnable createAfterExecuteFlagsUpdate(Collection<AppliedChanges<BannerWithSystemFields>> changes) {
        List<Long> bannerIds = mapList(changes, ac -> ac.getModel().getId());

        if (archive) {
            return conf -> {
                DSLContext context = conf.dsl();
                aggregatedStatusesRepository.markAdStatusesAsObsolete(context, updateBefore, bannerIds);
            };
        }
        Set<Long> adGroupsWithCondition = adGroupRepository.getAdGroupIdsWithConditions(shard,
                mapList(changes, ac -> ac.getModel().getAdGroupId()));
        if (adGroupsWithCondition.isEmpty()) {
            // Обновление дополнительных флагов производится только для групп с условиями показа
            return conf -> {
                DSLContext context = conf.dsl();
                aggregatedStatusesRepository.markAdStatusesAsObsolete(context, updateBefore, bannerIds);
            };
        }
        Collection<AppliedChanges<BannerWithSystemFields>> changesWithConditions = filterList(changes,
                ac -> adGroupsWithCondition.contains(ac.getModel().getAdGroupId()));

        Set<Long> adGroupsToModerate = moderationHelper.collectArchivedAdGroups(adGroupsWithCondition, shard);
        Set<Long> campaignsToModerate = moderationHelper.collectCampaignsToModerate(adGroupsWithCondition, shard);
        Set<Long> imagesToModerate = moderationHelper.collectBannersWithImagesToModerate(changesWithConditions);
        Set<Long> bannersWithBannerImagesToModerate = moderationHelper
                .collectBannersWithBannerImagesToModerate(changesWithConditions);
        Set<Long> bannersToDeleteMinusGeo = listToSet(changesWithConditions, ac -> ac.getModel().getId());

        Set<Long> skipUpdateFlagsBannerIds = changes.stream()
                .map(AppliedChanges::getModel)
                .filter(m -> m instanceof PerformanceBanner || m instanceof InternalBanner)
                .map(BannerWithSystemFields::getId)
                .collect(Collectors.toSet());
        List<AppliedChanges<BannerWithSystemFields>> bannersToUpdateFlags =
                filterList(changesWithConditions, p -> !skipUpdateFlagsBannerIds.contains(p.getModel().getId()));

        moderationHelper.setFlagsForBannersWithShowConditions(bannersToUpdateFlags);

        return conf -> {
            DSLContext context = conf.dsl();
            bannerModerationRepository.deleteMinusGeo(context, bannersToDeleteMinusGeo);
            campaignRepository.setStatusModerateIfNewOrNo(context, campaignsToModerate, CampaignStatusModerate.READY);
            adGroupRepository.dropStatusModerateExceptTypesWithoutModeration(context, adGroupsToModerate);
            imageRepository.setImagesStatusModerate(context, imagesToModerate, NewStatusImageModerate.READY);
            imageRepository.setBannerImagesStatusModerate(context, bannersWithBannerImagesToModerate,
                    StatusBannerImageModerate.READY);

            aggregatedStatusesRepository.markAdStatusesAsObsolete(context, updateBefore, bannerIds);
        };
    }

    @Override
    protected List<Long> execute(List<AppliedChanges<BannerWithSystemFields>> changesList) {
        try (TraceProfile ignore = Trace.current().profile("bannersArchiveUnarchiveOperation:execute")) {
            modifyRepository.update(bannerRepositoryContainer,
                    filterList(changesList, AppliedChanges::hasActuallyChangedProps));
            return mapList(changesList, a -> a.getModel().getId());
        }
    }

    @Override
    protected void afterExecution(ExecutionStep<BannerWithSystemFields> executionStep) {
        try (TraceProfile ignore = Trace.current().profile("bannersArchiveUnarchiveOperation:afterExecution")) {
            ppcDslContextProvider.ppc(shard).transaction(afterExecute);
        }
    }
}
