package ru.yandex.direct.core.entity.banner.service;

import java.time.LocalDateTime;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import one.util.streamex.StreamEx;

import ru.yandex.direct.core.aggregatedstatuses.repository.AggregatedStatusesRepository;
import ru.yandex.direct.core.entity.StatusBsSynced;
import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.adgroup.model.StatusAutobudgetShow;
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.core.entity.banner.container.BannerRepositoryContainer;
import ru.yandex.direct.core.entity.banner.model.BannerStatusModerate;
import ru.yandex.direct.core.entity.banner.model.BannerWithSystemFields;
import ru.yandex.direct.core.entity.banner.model.InternalBanner;
import ru.yandex.direct.core.entity.banner.model.StubBanner;
import ru.yandex.direct.core.entity.banner.repository.BannerCommonRepository;
import ru.yandex.direct.core.entity.banner.repository.BannerModerationRepository;
import ru.yandex.direct.core.entity.banner.repository.BannerModifyRepository;
import ru.yandex.direct.core.entity.banner.repository.BannerTypedRepository;
import ru.yandex.direct.core.entity.banner.service.validation.SuspendResumeBannerValidationService;
import ru.yandex.direct.core.entity.campaign.model.StatusAutobudgetForecast;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.operation.Applicability;
import ru.yandex.direct.operation.update.AppliedChangesValidatedStep;
import ru.yandex.direct.operation.update.ExecutionStep;
import ru.yandex.direct.operation.update.SimpleAbstractUpdateOperation;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static java.util.Collections.disjoint;
import static java.util.Collections.emptyList;
import static java.util.stream.Collectors.toList;
import static ru.yandex.direct.core.entity.banner.service.BannerUtils.bannerNeverSentToBs;
import static ru.yandex.direct.core.entity.banner.service.validation.BannerConstants.NEW_SENSITIVE_PROPERTIES;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

public class BannersSuspendResumeOperation extends SimpleAbstractUpdateOperation<BannerWithSystemFields, Long> {
    private final SuspendResumeBannerValidationService validationService;
    private final AdGroupRepository adGroupRepository;
    private final BannerCommonRepository bannerCommonRepository;
    private final BannerTypedRepository typedRepository;
    private final BannerModifyRepository modifyRepository;
    private final BannerModerationRepository moderationRepository;
    private final CampaignRepository campaignRepository;
    private final AggregatedStatusesRepository aggregatedStatusesRepository;
    private final int shard;
    private final Long operatorUid;
    private final ClientId clientId;
    private final boolean resume;
    private final LocalDateTime updateBefore;
    private final DslContextProvider dslContextProvider;
    private final BannerRepositoryContainer bannerRepositoryContainer;

    public BannersSuspendResumeOperation(int shard, Long operatorUid,
                                         List<ModelChanges<BannerWithSystemFields>> modelChanges,
                                         SuspendResumeBannerValidationService validationService,
                                         AdGroupRepository adGroupRepository,
                                         BannerCommonRepository bannerCommonRepository,
                                         BannerTypedRepository typedRepository,
                                         BannerModifyRepository modifyRepository,
                                         BannerModerationRepository moderationRepository,
                                         CampaignRepository campaignRepository,
                                         AggregatedStatusesRepository aggregatedStatusesRepository,
                                         ClientId clientId, boolean resume, LocalDateTime updateBefore,
                                         DslContextProvider dslContextProvider) {
        super(Applicability.PARTIAL, modelChanges, id -> new StubBanner().withId(id),
                NEW_SENSITIVE_PROPERTIES);
        this.validationService = validationService;
        this.adGroupRepository = adGroupRepository;
        this.shard = shard;
        this.operatorUid = operatorUid;
        this.bannerCommonRepository = bannerCommonRepository;
        this.typedRepository = typedRepository;
        this.modifyRepository = modifyRepository;
        this.moderationRepository = moderationRepository;
        this.campaignRepository = campaignRepository;
        this.aggregatedStatusesRepository = aggregatedStatusesRepository;
        this.clientId = clientId;
        this.resume = resume;
        this.updateBefore = updateBefore;
        this.dslContextProvider = dslContextProvider;

        bannerRepositoryContainer = new BannerRepositoryContainer(shard);
    }

    @Override
    protected List<Long> execute(List<AppliedChanges<BannerWithSystemFields>> changesList) {
        modifyRepository.update(bannerRepositoryContainer,
                filterList(changesList, AppliedChanges::hasActuallyChangedProps));
        return mapList(changesList, a -> a.getModel().getId());
    }

    @Override
    protected ValidationResult<List<ModelChanges<BannerWithSystemFields>>, Defect> validateModelChangesBeforeApply(
            ValidationResult<List<ModelChanges<BannerWithSystemFields>>, Defect> preValidateResult,
            Map<Long, BannerWithSystemFields> models) {

        return validationService.validateBanners(bannerRepositoryContainer.getShard(), preValidateResult, models,
                resume);
    }

    @Override
    protected ValidationResult<List<ModelChanges<BannerWithSystemFields>>, Defect> validateModelChanges(
            List<ModelChanges<BannerWithSystemFields>> modelChanges) {
        return validationService.validateChanges(clientId, operatorUid, modelChanges);
    }

    @Override
    public void onAppliedChangesValidated(AppliedChangesValidatedStep<BannerWithSystemFields> step) {
        step.getValidAppliedChangesWithIndex().forEach((idx, ac) -> {
            if (ac.hasActuallyChangedProps()) {
                ac.modifyIf(BannerWithSystemFields.STATUS_MODERATE, BannerStatusModerate.NEW, Objects::isNull);
                ac.modify(BannerWithSystemFields.STATUS_BS_SYNCED, StatusBsSynced.NO);
                ac.modify(BannerWithSystemFields.LAST_CHANGE, LocalDateTime.now());
            }
        });
    }

    @Override
    protected void afterExecution(ExecutionStep<BannerWithSystemFields> executionStep) {
        Collection<AppliedChanges<BannerWithSystemFields>> changes = executionStep.getAppliedChangesForExecution();
        List<BannerWithSystemFields> changedBanners = changes.stream()
                .filter(AppliedChanges::hasActuallyChangedProps)
                .map(AppliedChanges::getModel)
                .collect(toList());

        Set<Long> internalBannerIds = StreamEx.of(changedBanners)
                .select(InternalBanner.class)
                .map(InternalBanner::getId)
                .toSet();

        List<Long> bannerIds = mapList(changedBanners, BannerWithSystemFields::getId);
        Set<Long> adGroupIds = listToSet(changedBanners, BannerWithSystemFields::getAdGroupId);
        Set<Long> campaignIds = listToSet(changedBanners, BannerWithSystemFields::getCampaignId);

        Map<Long, List<Long>> bannerMinusGeo = moderationRepository.getBannersMinusGeo(shard, bannerIds);
        Map<Long, List<Long>> adGroupsGeo = listToMap(adGroupRepository.getAdGroups(shard, adGroupIds), AdGroup::getId,
                AdGroup::getGeo);
        Set<Long> adGroupsForBsSync = getAdGroupsForBsSync(bannerMinusGeo, adGroupsGeo, changes, resume);

        dslContextProvider.ppc(shard).transaction(conf -> {
            campaignRepository.setStatusAutobudgetForecast(shard, campaignIds, StatusAutobudgetForecast.NEW);
            adGroupRepository.updateStatusAutoBudgetShow(shard, adGroupIds, StatusAutobudgetShow.YES);
            adGroupRepository.updateLastChange(shard, adGroupIds);
            adGroupRepository.updateStatusBsSynced(shard, adGroupsForBsSync, StatusBsSynced.NO);
            aggregatedStatusesRepository.markAdStatusesAsObsolete(shard, updateBefore, bannerIds);

            // для внутренней рекламы сбрасываем статус остановленности баннера урл-мониторингом
            // даже когда остановили баннер, чтобы зафиксировать остановленность вручную
            bannerCommonRepository.dropInternalBannersStoppedByUrlMonitoring(shard, internalBannerIds);
        });
    }

    /**
     * Выбирает группы объявлений, таргетинг которых был затронут остановкой/запуском баннеров либо
     * группы, баннеры в которых ни разу не отправлялись в БК.
     *
     * @param bannerMinusGeo Минус-гео баннеров по ID баннера
     * @param adGroupsGeo    Таргетинг групп по ID группы
     * @param changes        Изменения баннеров
     * @param resume         true если нужно возобновить показы, false для остановки
     * @return список ID групп для отправки в БК
     */
    public static Set<Long> getAdGroupsForBsSync(Map<Long, List<Long>> bannerMinusGeo,
                                                 Map<Long, List<Long>> adGroupsGeo,
                                                 Collection<AppliedChanges<BannerWithSystemFields>> changes,
                                                 boolean resume) {
        return changes.stream()
                .filter(AppliedChanges::hasActuallyChangedProps)
                .map(AppliedChanges::getModel)
                .filter(banner -> adGroupTargetingAffected(bannerMinusGeo, adGroupsGeo, banner)
                        || (resume && bannerNeverSentToBs(banner)))
                .map(BannerWithSystemFields::getAdGroupId)
                .collect(Collectors.toSet());
    }

    /**
     * Проверяет затрагивается ли таргетинг группы остановкой/запуском баннера.
     *
     * @param bannerMinusGeo Минус-гео баннеров по ID баннера
     * @param adGroupsGeo    Таргетинг групп по ID группы
     * @param banner         Баннер
     * @return true если затрагивается, иначе false
     */
    private static boolean adGroupTargetingAffected(Map<Long, List<Long>> bannerMinusGeo,
                                                    Map<Long, List<Long>> adGroupsGeo,
                                                    BannerWithSystemFields banner) {
        List<Long> a = Optional.ofNullable(adGroupsGeo.get(banner.getAdGroupId())).orElse(emptyList());
        List<Long> b = Optional.ofNullable(bannerMinusGeo.get(banner.getId())).orElse(emptyList());
        return !disjoint(a, b);
    }

    @Override
    protected Collection<BannerWithSystemFields> getModels(Collection<Long> ids) {
        return typedRepository.getStrictlyFullyFilled(shard, ids, BannerWithSystemFields.class);
    }
}
