package ru.yandex.direct.core.entity.banner.service;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.IdentityHashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.BiFunction;
import java.util.function.Function;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.core.entity.banner.container.BannerAdditionalActionsContainer;
import ru.yandex.direct.core.entity.banner.container.BannerRepositoryContainer;
import ru.yandex.direct.core.entity.banner.container.BannersUpdateOperationContainer;
import ru.yandex.direct.core.entity.banner.container.BannersUpdateOperationContainerImpl;
import ru.yandex.direct.core.entity.banner.container.BannersUpdateOperationContainerService;
import ru.yandex.direct.core.entity.banner.model.Banner;
import ru.yandex.direct.core.entity.banner.model.BannerWithSitelinks;
import ru.yandex.direct.core.entity.banner.model.BannerWithVcard;
import ru.yandex.direct.core.entity.banner.repository.BannerRepositoryConstants;
import ru.yandex.direct.core.entity.banner.repository.BannerTypedRepository;
import ru.yandex.direct.core.entity.banner.service.moderation.ModerationMode;
import ru.yandex.direct.core.entity.banner.service.type.update.BannerUpdateOperationTypeSupportFacade;
import ru.yandex.direct.core.entity.banner.service.validation.BannerValidationInfo;
import ru.yandex.direct.core.entity.banner.service.validation.BannersUpdateValidationService;
import ru.yandex.direct.core.entity.banner.service.validation.type.update.BannerUpdateValidationTypeSupportFacade;
import ru.yandex.direct.core.entity.client.repository.ClientRepository;
import ru.yandex.direct.core.entity.sitelink.model.SitelinkSet;
import ru.yandex.direct.core.entity.user.repository.UserRepository;
import ru.yandex.direct.core.entity.vcard.model.Vcard;
import ru.yandex.direct.dbschema.ppc.enums.BannersBannerType;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.model.ModelProperty;
import ru.yandex.direct.model.ModelWithId;
import ru.yandex.direct.operation.Applicability;
import ru.yandex.direct.operation.update.AppliedChangesValidatedStep;
import ru.yandex.direct.operation.update.ChangesAppliedStep;
import ru.yandex.direct.operation.update.ExecutionStep;
import ru.yandex.direct.operation.update.ModelChangesValidatedStep;
import ru.yandex.direct.operation.update.SimpleAbstractUpdateOperation;
import ru.yandex.direct.rbac.RbacRole;
import ru.yandex.direct.rbac.RbacService;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.constraint.CommonConstraints;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static com.google.common.base.Preconditions.checkState;
import static ru.yandex.direct.core.entity.banner.service.BannerLogUtils.logBannersCount;
import static ru.yandex.direct.core.entity.banner.service.BannerLogUtils.logBannersCountModelChanges;
import static ru.yandex.direct.core.entity.banner.service.BannerLogUtils.logModelChangesIds;
import static ru.yandex.direct.core.entity.banner.service.validation.BannerConstants.NEW_SENSITIVE_PROPERTIES;
import static ru.yandex.direct.multitype.service.validation.type.ValidationTypeSupportUtils.filterValidSubResults;
import static ru.yandex.direct.operation.update.ModelStubProvider.getModelStub;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@ParametersAreNonnullByDefault
public class BannersUpdateOperation<T extends Banner> extends SimpleAbstractUpdateOperation<T, Long> {
    private static final Logger logger = LoggerFactory.getLogger(BannersUpdateOperation.class);

    private final BannersUpdateOperationContainerImpl operationContainer;
    private final BannerRepositoryContainer repositoryContainer;
    private final BannerAdditionalActionsContainer additionalActionsContainer;

    private final BannerTypedRepository typedRepository;
    private final BannerUpdateOperationTypeSupportFacade operationTypeSupportFacade;
    private final BannerUpdateValidationTypeSupportFacade validationTypeSupportFacade;
    private final BannersUpdateExecutionFacade bannersUpdateExecutionService;
    private final BannersUpdateOperationContainerService bannersOperationContainerService;
    private final BannersUpdateValidationService validationService;
    private final Class<T> clazz;
    private final Function<BannersUpdateOperationContainer, Constraint<ModelChanges<T>, Defect>>
            modelChangesIsApplicableConstraintCreator;

    private final IdentityHashMap<ModelChanges<Banner>, Integer> modelChangesToIndexMap;

    public BannersUpdateOperation(Applicability applicability,
                                  boolean partOfComplexOperation,
                                  List<ModelChanges<T>> modelChanges,
                                  ModerationMode moderationMode,
                                  RbacService rbacService,
                                  ClientRepository clientRepository,
                                  UserRepository userRepository,
                                  BannerTypedRepository typedRepository,
                                  BannersUpdateExecutionFacade bannersUpdateExecutionService,
                                  BannerUpdateOperationTypeSupportFacade operationTypeSupportFacade,
                                  BannerUpdateValidationTypeSupportFacade validationTypeSupportFacade,
                                  BannersUpdateOperationContainerService bannersOperationContainerService,
                                  BannersUpdateValidationService validationService,
                                  Long operatorUid,
                                  RbacRole operatorRole,
                                  ClientId clientId,
                                  int shard,
                                  boolean isFromApi,
                                  boolean isUcPreValidation,
                                  DatabaseMode databaseMode,
                                  Set<String> clientEnabledFeatures,
                                  Class<T> clazz,
                                  @Nullable Function<BannersUpdateOperationContainer,
                                          Constraint<ModelChanges<T>, Defect>>
                                          modelChangesIsApplicableConstraintCreator,
                                  Map<ModelProperty<? super T, ?>, BiFunction<Object, Object, Boolean>>
                                          customModelEquals) {

        super(applicability, modelChanges, getModelStub(clazz, BannerRepositoryConstants.BANNER_CLASS_TO_TYPE.keySet()),
                NEW_SENSITIVE_PROPERTIES, customModelEquals);
        this.typedRepository = typedRepository;
        this.bannersUpdateExecutionService = bannersUpdateExecutionService;
        this.operationTypeSupportFacade = operationTypeSupportFacade;
        this.validationTypeSupportFacade = validationTypeSupportFacade;
        this.bannersOperationContainerService = bannersOperationContainerService;
        this.validationService = validationService;
        this.clazz = clazz;
        this.modelChangesIsApplicableConstraintCreator = Optional.ofNullable(modelChangesIsApplicableConstraintCreator)
                .orElse(container -> CommonConstraints.success());

        Long clientRegionId = clientRepository.getCountryRegionIdByClientId(shard, clientId)
                .orElse(null);

        Long uid = userRepository.getChiefUidByClientId(shard, clientId.asLong());
        var client = clientRepository.get(shard, List.of(clientId)).get(0);

        operationContainer = new BannersUpdateOperationContainerImpl(shard, operatorUid, operatorRole, clientId,
                uid, rbacService.getChiefByClientId(clientId), clientRegionId, clientEnabledFeatures,
                moderationMode, operatorRole.isInternal(), partOfComplexOperation, true);
        operationContainer.setClient(client);
        operationContainer.setFromApi(isFromApi);
        operationContainer.setUcPreValidation(isUcPreValidation);
        operationContainer.setDatabaseMode(databaseMode);
        repositoryContainer = new BannerRepositoryContainer(shard);
        additionalActionsContainer = new BannerAdditionalActionsContainer(clientId, clientRegionId);
        bannersOperationContainerService.fillContainers(operationContainer, modelChanges);

        modelChangesToIndexMap = EntryStream.of(modelChanges)
                .mapValues(mc -> mc.castModelUp(Banner.class))
                .invert()
                .toCustomMap(IdentityHashMap::new);
    }

    /**
     * Выставляет указанным баннерам id визитки.
     *
     * @param vcardByIndex мапа (индекс баннера) -> (vcard)
     */
    public void setVcards(Map<Integer, Vcard> vcardByIndex) {
        checkExecutionState();
        var vcardIdToData = EntryStream.of(vcardByIndex)
                .nonNullValues()
                .mapToKey((k, v) -> v.getId())
                .append(operationContainer.getVcardIdToData())
                .distinctKeys()
                .toMap();
        operationContainer.setVcardIdToData(vcardIdToData);


        Map<Integer, Long> vcardIdsByIndex = new HashMap<>();
        vcardByIndex.forEach((k, v) -> vcardIdsByIndex.put(k, v != null ? v.getId() : null));
        modifyProperty(BannerWithVcard.VCARD_ID, vcardIdsByIndex);
    }

    /**
     * Выставляет указанным баннерам id набора сайтлинков.
     *
     * @param sitelinkSetsByIndex мапа (индекс баннера) -> (sitelinkSet)
     */
    public void setSitelinkSets(Map<Integer, SitelinkSet> sitelinkSetsByIndex) {
        checkExecutionState();

        var sitelinkIdToData = EntryStream.of(sitelinkSetsByIndex)
                .nonNullValues()
                .mapToKey((k, v) -> v.getId())
                .append(operationContainer.getSitelinkSets())
                .distinctKeys()
                .toMap();
        operationContainer.setSitelinkSets(new ArrayList<>(sitelinkIdToData.values()));

        Map<Integer, Long> sitelinkIdsByIndex = new HashMap<>();
        sitelinkSetsByIndex.forEach((k, v) -> sitelinkIdsByIndex.put(k, v != null ? v.getId() : null));

        modifyProperty(BannerWithSitelinks.SITELINKS_SET_ID, sitelinkIdsByIndex);
    }

    /**
     * Сохранение информации о баннерах, приходящей из комплексной операции
     * Необходимо для корректной валидации
     *
     * @param bannersValidationInfoMap мапа индексов баннеры в информацию о них
     */
    public void setBannerValidationInfo(Map<Integer, BannerValidationInfo> bannersValidationInfoMap) {
        operationContainer.setBannersValidationInfoMap(bannersValidationInfoMap);
    }

    private void checkExecutionState() {
        checkState(!isExecuted(), "operation is already executed!");
    }

    @Override
    protected ValidationResult<List<ModelChanges<T>>, Defect> validateModelChanges(
            List<ModelChanges<T>> modelChanges) {
        logModelChangesIds(logger, "Banner ids in validateModelChanges: {}", modelChanges);
        Collection<Long> modelChangesBannerIds = mapList(modelChanges, ModelChanges::getId);
        Map<Long, BannersBannerType> clientBannerIdsWithType = typedRepository.getClientBannerIdsWithType(
                operationContainer.getShard(), operationContainer.getClientId(), modelChangesBannerIds);
        fillContainerBannerType(clientBannerIdsWithType, modelChanges);

        ValidationResult<List<ModelChanges<T>>, Defect> vr = validationService.validateModelChanges(operationContainer,
                modelChanges, modelChangesBannerIds, clientBannerIdsWithType.keySet(),
                modelChangesIsApplicableConstraintCreator);
        var vrWithValidSubResults = filterValidSubResults(vr);
        validationTypeSupportFacade.preValidate(operationContainer, vrWithValidSubResults);
        return vr;
    }

    @Override
    protected Collection<T> getModels(Collection<Long> ids) {
        List<T> banners = typedRepository.getStrictlyFullyFilled(operationContainer.getShard(), ids, clazz);
        Map<Long, T> bannersMap = listToMap(banners, ModelWithId::getId);
        IdentityHashMap<Banner, Integer> bannerToIndexMap = EntryStream.of(modelChangesToIndexMap)
                .mapKeys(mc -> (Banner) bannersMap.get(mc.getId()))
                .nonNullKeys()
                .toCustomMap(IdentityHashMap::new);
        operationContainer.setBannerToIndexMap(bannerToIndexMap);
        return banners;
    }

    private void fillContainerBannerType(Map<Long, BannersBannerType> clientBannerIdsWithType,
                                         List<ModelChanges<T>> modelChanges) {
        for (ModelChanges<T> mc : modelChanges) {
            BannersBannerType bannerType = clientBannerIdsWithType.get(mc.getId());
            if (bannerType != null) {
                operationContainer.setBannerType(mc.getId(), bannerType);
            }
        }
    }

    @Override
    protected void onModelChangesValidated(
            ModelChangesValidatedStep<T> modelChangesValidatedStep) {
        Collection<ModelChanges<T>> modelChanges =
                modelChangesValidatedStep.getValidModelChanges();
        operationTypeSupportFacade.onModelChangesValidated(operationContainer, modelChanges);
    }

    @Override
    protected ValidationResult<List<ModelChanges<T>>, Defect> validateModelChangesBeforeApply(
            ValidationResult<List<ModelChanges<T>>, Defect> preValidateResult,
            Map<Long, T> models) {

        bannersOperationContainerService.fillContainerOnModelChangesBeforeApply(
                operationContainer,
                preValidateResult,
                models
        );
        validationTypeSupportFacade.validateBeforeApply(operationContainer, preValidateResult, models);
        return preValidateResult;
    }

    @Override
    protected void onChangesApplied(ChangesAppliedStep<T> changesAppliedStep) {
        List<AppliedChanges<T>> validAppliedChanges =
                new ArrayList<>(changesAppliedStep.getAppliedChangesForValidModelChanges());
        operationTypeSupportFacade.onChangesApplied(operationContainer, validAppliedChanges);

        bannersOperationContainerService.fillContainerOnChangesApplied(operationContainer, validAppliedChanges);
    }

    @Override
    protected ValidationResult<List<T>, Defect> validateAppliedChanges(
            ValidationResult<List<T>, Defect> validationResult,
            Map<Integer, AppliedChanges<T>> appliedChangesForValidModelChanges) {
        validationTypeSupportFacade.validate(operationContainer, validationResult,
                appliedChangesForValidModelChanges);
        return validationResult;
    }

    @Override
    protected void onAppliedChangesValidated(
            AppliedChangesValidatedStep<T> appliedChangesValidatedStep) {
        var validAppliedChanges = new ArrayList<>(appliedChangesValidatedStep.getValidAppliedChanges());

        logBannersCountModelChanges(logger,
                appliedChangesValidatedStep.getModelChanges(),
                appliedChangesValidatedStep.getValidModelChanges());

        operationTypeSupportFacade.onAppliedChangesValidated(operationContainer, validAppliedChanges);
    }

    @Override
    protected void beforeExecution(ExecutionStep<T> executionStep) {
        var appliedChangesForExecution = new ArrayList<>(executionStep.getAppliedChangesForExecution());
        logBannersCount(logger, "Valid banners in beforeExecution: {}",
                mapList(appliedChangesForExecution, AppliedChanges::getModel));

        operationTypeSupportFacade.beforeExecution(appliedChangesForExecution, operationContainer);
    }

    @Override
    protected List<Long> execute(List<AppliedChanges<T>> applicableAppliedChanges) {
        return bannersUpdateExecutionService.execute(applicableAppliedChanges, operationContainer, repositoryContainer,
                additionalActionsContainer);
    }

    @Override
    protected void afterExecution(ExecutionStep<T> executionStep) {
        var appliedChangesForExecution = new ArrayList<>(executionStep.getAppliedChangesForExecution());
        operationTypeSupportFacade.afterExecution(appliedChangesForExecution, operationContainer);
    }
}
