package ru.yandex.direct.core.entity.banner.service;

import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.BiFunction;
import java.util.function.Function;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.banner.container.BannersUpdateOperationContainer;
import ru.yandex.direct.core.entity.banner.container.BannersUpdateOperationContainerService;
import ru.yandex.direct.core.entity.banner.model.Banner;
import ru.yandex.direct.core.entity.banner.model.BannerWithSystemFields;
import ru.yandex.direct.core.entity.banner.repository.BannerTypedRepository;
import ru.yandex.direct.core.entity.banner.service.moderation.ModerationMode;
import ru.yandex.direct.core.entity.banner.service.type.update.BannerUpdateOperationTypeSupportFacade;
import ru.yandex.direct.core.entity.banner.service.validation.BannersUpdateValidationService;
import ru.yandex.direct.core.entity.banner.service.validation.type.update.BannerUpdateValidationTypeSupportFacade;
import ru.yandex.direct.core.entity.client.repository.ClientRepository;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.user.repository.UserRepository;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.model.ModelProperty;
import ru.yandex.direct.operation.Applicability;
import ru.yandex.direct.rbac.RbacRole;
import ru.yandex.direct.rbac.RbacService;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.result.Defect;

import static java.util.Collections.emptyMap;

@Component
@ParametersAreNonnullByDefault
public class BannersUpdateOperationFactory {

    private final BannerTypedRepository typedRepository;

    private final BannerUpdateValidationTypeSupportFacade validationTypeSupportFacade;

    private final BannerUpdateOperationTypeSupportFacade updateOperationTypeSupportFacade;

    private final RbacService rbacService;

    private final ClientRepository clientRepository;

    private final UserRepository userRepository;

    private final BannersUpdateOperationContainerService bannersOperationContainerService;

    private final ShardHelper shardHelper;

    private final FeatureService featureService;

    private final BannersUpdateValidationService validationService;

    private final BannersUpdateExecutionFacade bannersUpdateExecutionService;

    @Autowired
    public BannersUpdateOperationFactory(BannerTypedRepository typedRepository,
                                         BannerUpdateValidationTypeSupportFacade validationTypeSupportFacade,
                                         BannerUpdateOperationTypeSupportFacade updateOperationTypeSupportFacade,
                                         BannersUpdateExecutionFacade bannersUpdateExecutionService,
                                         RbacService rbacService,
                                         ClientRepository clientRepository,
                                         UserRepository userRepository,
                                         BannersUpdateOperationContainerService bannersOperationContainerService,
                                         ShardHelper shardHelper,
                                         FeatureService featureService,
                                         BannersUpdateValidationService validationService) {
        this.typedRepository = typedRepository;
        this.validationTypeSupportFacade = validationTypeSupportFacade;
        this.updateOperationTypeSupportFacade = updateOperationTypeSupportFacade;
        this.bannersUpdateExecutionService = bannersUpdateExecutionService;
        this.rbacService = rbacService;
        this.clientRepository = clientRepository;
        this.userRepository = userRepository;
        this.bannersOperationContainerService = bannersOperationContainerService;
        this.shardHelper = shardHelper;
        this.featureService = featureService;
        this.validationService = validationService;
    }

    public BannersUpdateOperation<BannerWithSystemFields> createUpdateOperation(
            Applicability applicability,
            boolean partOfComplexOperation,
            ModerationMode moderationMode,
            List<ModelChanges<BannerWithSystemFields>> modelChanges,
            int shard,
            ClientId clientId,
            Long operatorUid,
            Set<String> clientEnabledFeatures,
            boolean isUcPreValidation) {
        return createUpdateOperation(applicability, partOfComplexOperation, moderationMode, modelChanges, shard,
                clientId, operatorUid, clientEnabledFeatures, BannerWithSystemFields.class, false,
                isUcPreValidation, DatabaseMode.ONLY_MYSQL, null);
    }

    public <T extends Banner> BannersUpdateOperation<T> createUpdateOperation(
            Applicability applicability,
            boolean partOfComplexOperation,
            ModerationMode moderationMode,
            List<ModelChanges<T>> modelChanges,
            int shard,
            ClientId clientId,
            Long operatorUid,
            Set<String> clientEnabledFeatures,
            Class<T> clazz,
            boolean isFromApi,
            boolean isUcPreValidation,
            DatabaseMode databaseMode,
            @Nullable Function<BannersUpdateOperationContainer, Constraint<ModelChanges<T>, Defect>>
                    modelChangesIsApplicableConstraintCreator) {
        return createUpdateOperation(applicability, partOfComplexOperation, moderationMode, modelChanges, shard,
                clientId, operatorUid, clientEnabledFeatures, clazz, isFromApi, isUcPreValidation, databaseMode,
                modelChangesIsApplicableConstraintCreator, emptyMap());
    }

    public <T extends Banner> BannersUpdateOperation<T> createUpdateOperation(
            Applicability applicability,
            boolean partOfComplexOperation,
            ModerationMode moderationMode,
            List<ModelChanges<T>> modelChanges,
            int shard,
            ClientId clientId,
            Long operatorUid,
            Set<String> clientEnabledFeatures,
            Class<T> clazz,
            boolean isFromApi,
            boolean isUcPreValidation,
            DatabaseMode databaseMode,
            @Nullable Function<BannersUpdateOperationContainer, Constraint<ModelChanges<T>, Defect>>
                    modelChangesIsApplicableConstraintCreator,
            Map<ModelProperty<? super T, ?>, BiFunction<Object, Object, Boolean>> customModelEquals) {
        RbacRole operatorRole = rbacService.getUidRole(operatorUid);

        return new BannersUpdateOperation<>(
                applicability,
                partOfComplexOperation,
                modelChanges,
                moderationMode,
                rbacService,
                clientRepository,
                userRepository,
                typedRepository,
                bannersUpdateExecutionService,
                updateOperationTypeSupportFacade,
                validationTypeSupportFacade,
                bannersOperationContainerService,
                validationService,
                operatorUid,
                operatorRole,
                clientId,
                shard,
                isFromApi,
                isUcPreValidation,
                databaseMode,
                clientEnabledFeatures,
                clazz,
                modelChangesIsApplicableConstraintCreator,
                customModelEquals);
    }

    public BannersUpdateOperation<BannerWithSystemFields> createPartialUpdateOperation(
            List<ModelChanges<BannerWithSystemFields>> modelChanges, Long operatorUid, ClientId clientId) {
        return createPartialUpdateOperation(ModerationMode.DEFAULT, modelChanges, operatorUid, clientId);
    }

    public <T extends Banner> BannersUpdateOperation<T> createPartialUpdateOperation(
            List<ModelChanges<T>> modelChanges, Long operatorUid, ClientId clientId, Class<T> clazz) {
        return createPartialUpdateOperation(ModerationMode.DEFAULT, modelChanges, operatorUid, clientId, clazz, null);
    }

    public <T extends Banner> BannersUpdateOperation<T> createPartialUpdateOperationWithCustomModelEquals(
            List<ModelChanges<T>> modelChanges, Long operatorUid, ClientId clientId, int shard, Class<T> clazz,
            Map<ModelProperty<? super T, ?>, BiFunction<Object, Object, Boolean>> customModelEquals
    ) {
        return createUpdateOperation(Applicability.PARTIAL, modelChanges, ModerationMode.DEFAULT,
                operatorUid, clientId, shard, clazz, null, customModelEquals);
    }

    public BannersUpdateOperation<BannerWithSystemFields> createPartialUpdateOperation(
            ModerationMode moderationMode, List<ModelChanges<BannerWithSystemFields>> modelChanges,
            Long operatorUid, ClientId clientId) {
        return createPartialUpdateOperation(moderationMode, modelChanges, operatorUid, clientId,
                BannerWithSystemFields.class, null);
    }

    public <T extends Banner> BannersUpdateOperation<T> createPartialUpdateOperation(
            ModerationMode moderationMode, List<ModelChanges<T>> modelChanges,
            Long operatorUid, ClientId clientId, Class<T> clazz) {
        return createPartialUpdateOperation(moderationMode, modelChanges, operatorUid, clientId,
                clazz, null);
    }

    public <T extends Banner> BannersUpdateOperation<T> createPartialUpdateOperation(
            ModerationMode moderationMode, List<ModelChanges<T>> modelChanges, Long operatorUid, ClientId clientId,
            Class<T> clazz,
            @Nullable Function<BannersUpdateOperationContainer, Constraint<ModelChanges<T>, Defect>>
                    modelChangesIsApplicableConstraintCreator) {
        return createUpdateOperation(Applicability.PARTIAL, modelChanges, moderationMode, operatorUid, clientId, clazz,
                modelChangesIsApplicableConstraintCreator, DatabaseMode.ONLY_MYSQL);
    }

    public <T extends Banner> BannersUpdateOperation<T> createPartialUpdateOperation(
            ModerationMode moderationMode, List<ModelChanges<T>> modelChanges, Long operatorUid, ClientId clientId,
            int shard, Class<T> clazz,
            @Nullable Function<BannersUpdateOperationContainer, Constraint<ModelChanges<T>, Defect>>
                    modelChangesIsApplicableConstraintCreator) {
        return createUpdateOperation(Applicability.PARTIAL, modelChanges, moderationMode, operatorUid, clientId, shard,
                clazz, modelChangesIsApplicableConstraintCreator, emptyMap());
    }

    public BannersUpdateOperation<BannerWithSystemFields> createFullUpdateOperation(
            List<ModelChanges<BannerWithSystemFields>> modelChanges, Long operatorUid, ClientId clientId,
            DatabaseMode databaseMode) {
        return createUpdateOperation(Applicability.FULL, modelChanges, ModerationMode.DEFAULT, operatorUid, clientId,
                databaseMode);
    }

    private BannersUpdateOperation<BannerWithSystemFields> createUpdateOperation(
            Applicability applicability,
            List<ModelChanges<BannerWithSystemFields>> modelChanges,
            ModerationMode moderationMode,
            Long operatorUid, ClientId clientId, DatabaseMode databaseMode) {
        return createUpdateOperation(applicability, modelChanges, moderationMode, operatorUid, clientId,
                BannerWithSystemFields.class, null, databaseMode);
    }

    public BannersUpdateOperation<BannerWithSystemFields> createUpdateOperation(
            List<ModelChanges<BannerWithSystemFields>> modelChanges,
            Long operatorUid,
            ClientId clientId,
            boolean isFromApi) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        var features = featureService.getEnabledForClientId(clientId);

        return createUpdateOperation(Applicability.PARTIAL, false, ModerationMode.DEFAULT,
                modelChanges, shard, clientId, operatorUid, features, BannerWithSystemFields.class, isFromApi,
                false, DatabaseMode.ONLY_MYSQL, null);
    }

    private <T extends Banner> BannersUpdateOperation<T> createUpdateOperation(
            Applicability applicability,
            List<ModelChanges<T>> modelChanges,
            ModerationMode moderationMode,
            Long operatorUid, ClientId clientId, Class<T> clazz,
            @Nullable Function<BannersUpdateOperationContainer, Constraint<ModelChanges<T>, Defect>>
                    modelChangesIsApplicableConstraintCreator,
            DatabaseMode databaseMode) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        var features = featureService.getEnabledForClientId(clientId);

        return createUpdateOperation(applicability, false, moderationMode, modelChanges, shard,
                clientId, operatorUid, features, clazz, false, false,
                databaseMode, modelChangesIsApplicableConstraintCreator);
    }

    private <T extends Banner> BannersUpdateOperation<T> createUpdateOperation(
            Applicability applicability,
            List<ModelChanges<T>> modelChanges,
            ModerationMode moderationMode,
            Long operatorUid, ClientId clientId, int shard, Class<T> clazz,
            @Nullable Function<BannersUpdateOperationContainer, Constraint<ModelChanges<T>, Defect>>
                    modelChangesIsApplicableConstraintCreator,
            Map<ModelProperty<? super T, ?>, BiFunction<Object, Object, Boolean>> customModelEquals) {
        var features = featureService.getEnabledForClientId(clientId);

        return createUpdateOperation(applicability, false, moderationMode, modelChanges, shard,
                clientId, operatorUid, features, clazz, false, false, DatabaseMode.ONLY_MYSQL,
                modelChangesIsApplicableConstraintCreator, customModelEquals);
    }

    public BannersUpdateOperation<BannerWithSystemFields> createPartialUpdateOperationAsPartOfComplexOperation(
            ModerationMode moderationMode, List<ModelChanges<BannerWithSystemFields>> banners,
            long operatorUid, ClientId clientId, boolean isUcPreValidation) {
        int shard = shardHelper.getShardByClientId(clientId);
        var enabledFeatures = featureService.getEnabledForClientId(clientId);
        return createUpdateOperation(
                Applicability.PARTIAL,
                true,
                moderationMode,
                banners,
                shard,
                clientId,
                operatorUid,
                enabledFeatures,
                isUcPreValidation
        );
    }
}
