package ru.yandex.direct.core.entity.banner.service;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;

import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.banner.container.ComplexBanner;
import ru.yandex.direct.core.entity.banner.model.Banner;
import ru.yandex.direct.core.entity.banner.model.BannerWithCreative;
import ru.yandex.direct.core.entity.banner.model.BannerWithSystemFields;
import ru.yandex.direct.core.entity.banner.repository.BannerRelationsRepository;
import ru.yandex.direct.core.entity.banner.repository.BannerTypedRepository;
import ru.yandex.direct.core.entity.creative.model.Creative;
import ru.yandex.direct.core.entity.creative.repository.CreativeRepository;
import ru.yandex.direct.core.entity.sitelink.model.SitelinkSet;
import ru.yandex.direct.core.entity.sitelink.repository.SitelinkSetRepository;
import ru.yandex.direct.core.entity.vcard.model.Vcard;
import ru.yandex.direct.core.entity.vcard.repository.VcardRepository;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;

import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;

@Service
public class ComplexBannerService {
    private final ShardHelper shardHelper;
    private final BannerTypedRepository bannerTypedRepository;
    private final BannerRelationsRepository bannerRelationsRepository;
    private final CreativeRepository creativeRepository;
    private final SitelinkSetRepository sitelinkSetRepository;
    private final VcardRepository vcardRepository;

    @Autowired
    public ComplexBannerService(ShardHelper shardHelper,
                                BannerTypedRepository bannerTypedRepository,
                                BannerRelationsRepository bannerRelationsRepository,
                                CreativeRepository creativeRepository,
                                SitelinkSetRepository sitelinkSetRepository,
                                VcardRepository vcardRepository) {
        this.shardHelper = shardHelper;
        this.bannerTypedRepository = bannerTypedRepository;
        this.bannerRelationsRepository = bannerRelationsRepository;
        this.creativeRepository = creativeRepository;
        this.sitelinkSetRepository = sitelinkSetRepository;
        this.vcardRepository = vcardRepository;
    }

    public List<ComplexBanner> getComplexBannersByAdGroupIds(ClientId clientId, long clientUid,
                                                             Collection<Long> adGroupIds) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);

        var banners = bannerTypedRepository.getBannersByGroupIds(shard, adGroupIds);
        Set<Long> bannerIds = listToSet(banners, Banner::getId);

        Map<Long, Long> creativeIdByBannerIdMap = StreamEx.of(banners)
                .select(BannerWithCreative.class)
                .mapToEntry(BannerWithCreative::getId, BannerWithCreative::getCreativeId)
                .nonNullValues()
                .toMap();
        List<Creative> creatives = creativeRepository.getCreatives(shard, clientId, creativeIdByBannerIdMap.values());
        Map<Long, Creative> creativeMap = listToMap(creatives, Creative::getId);

        Map<Long, Long> bannerIdsToVcardIds = vcardRepository.getBannerIdsToVcardIds(shard, clientUid, bannerIds, null);
        List<Vcard> vcards = vcardRepository.getVcards(shard, clientUid, bannerIdsToVcardIds.values());
        Map<Long, Vcard> vcardMap = listToMap(vcards, Vcard::getId);

        Map<Long, Long> bannerIdsToSitelinkSetIds = bannerRelationsRepository.getBannerIdsToSitelinkSetIds(shard,
                bannerIds);
        List<SitelinkSet> sitelinkSets = sitelinkSetRepository.get(shard, bannerIdsToSitelinkSetIds.values());
        Map<Long, SitelinkSet> sitelinkSetMap = listToMap(sitelinkSets, SitelinkSet::getId);

        List<ComplexBanner> complexBanners = new ArrayList<>();
        for (Banner banner : banners) {
            Long creativeId = creativeIdByBannerIdMap.get(banner.getId());
            Long vcardId = bannerIdsToVcardIds.get(banner.getId());
            Long sitelinkSetId = bannerIdsToSitelinkSetIds.get(banner.getId());

            ComplexBanner complexBanner = new ComplexBanner()
                    .withBanner((BannerWithSystemFields) banner)
                    .withCreative(creativeMap.get(creativeId))
                    .withSitelinkSet(sitelinkSetMap.get(sitelinkSetId))
                    .withVcard(vcardMap.get(vcardId));

            complexBanners.add(complexBanner);
        }

        return complexBanners;
    }
}
