package ru.yandex.direct.core.entity.banner.service;

import java.util.List;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.copyentity.CopyConfig;
import ru.yandex.direct.core.copyentity.CopyConfigBuilder;
import ru.yandex.direct.core.copyentity.CopyOperationFactory;
import ru.yandex.direct.core.copyentity.model.CopyCampaignFlags;
import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.banner.model.BannerWithAdGroupId;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.result.Result;
import ru.yandex.direct.result.ResultState;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.core.entity.banner.service.validation.defects.BannerDefects.errorsWhileCopyingBanners;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;

@Service
@ParametersAreNonnullByDefault
public class CopyBannerService {

    private final CopyOperationFactory copyOperationFactory;
    private final BannerService bannerService;

    @Autowired
    public CopyBannerService(
            CopyOperationFactory copyOperationFactory,
            BannerService bannerService
    ) {
        this.copyOperationFactory = copyOperationFactory;
        this.bannerService = bannerService;
    }

    public MassResult<Long> copyBanners(Long operatorUid,
                                        List<Long> bannerIds,
                                        ClientId clientFromId,
                                        ClientId clientToId,
                                        long adGroupToId) {
        return copy(operatorUid, bannerIds, clientFromId, clientToId, adGroupToId);
    }

    public MassResult<Long> copySameAdGroup(Long operatorUid, List<Long> bannerIds, ClientId clientFromId) {
        var config = copyConfig(operatorUid, bannerIds, clientFromId, clientFromId, null);
        return makeCopy(config);
    }

    private MassResult<Long> copy(Long operatorUid,
                                  List<Long> bannerIds,
                                  ClientId clientFromId,
                                  ClientId clientToId,
                                  long adGroupToId) {
        var config = copyConfig(operatorUid, bannerIds, clientFromId, clientToId, adGroupToId);
        return makeCopy(config);
    }

    private MassResult<Long> makeCopy(CopyConfig<BannerWithAdGroupId, Long> copyConfig) {
        var operation = copyOperationFactory.build(copyConfig);

        var result = operation.copy();
        result.logFailedResultsForMonitoring();

        ValidationResult<?, Defect> vr = result.getMassResult().getValidationResult();
        if (vr.hasAnyErrors()) {
            vr.addWarning(errorsWhileCopyingBanners());
        }
        List<Result<Long>> copiedEntityResults = result.getCopiedEntityResults(BannerWithAdGroupId.class);

        return new MassResult<Long>(copiedEntityResults, vr, ResultState.SUCCESSFUL);
    }

    private CopyConfig<BannerWithAdGroupId, Long> copyConfig(Long operatorUid,
                                                             List<Long> bannerIds,
                                                             ClientId clientFromId,
                                                             ClientId clientToId,
                                                             @Nullable Long adGroupToId) {
        var banners = bannerService.get(clientFromId, operatorUid, bannerIds);

        CopyConfigBuilder<BannerWithAdGroupId, Long> builder =
                new CopyConfigBuilder<>(clientFromId, clientToId, operatorUid, BannerWithAdGroupId.class, bannerIds);

        builder.withFlags(new CopyCampaignFlags.Builder()
                // https://st.yandex-team.ru/DIRECT-154930
                .withCopyStopped(true)
                .build());

        Set<Long> adGroupIds = listToSet(banners, BannerWithAdGroupId::getAdGroupId);
        adGroupIds.forEach(adGroupId ->
                builder.withParentIdMapping(AdGroup.class, adGroupId, nvl(adGroupToId, adGroupId)));

        return builder.build();
    }
}
