package ru.yandex.direct.core.entity.banner.service;

import java.time.format.DateTimeFormatter;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.base.Strings;
import com.google.common.collect.Iterables;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.banner.repository.BannerRelationsRepository;
import ru.yandex.direct.core.entity.banner.type.creative.BannerCreativeRepository;
import ru.yandex.direct.core.entity.campaign.model.Campaign;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.client.model.ClientMeasurerSettings;
import ru.yandex.direct.core.entity.client.model.ClientMeasurerSystem;
import ru.yandex.direct.core.entity.client.model.MediascopeClientMeasurerSettings;
import ru.yandex.direct.core.entity.client.service.ClientMeasurerSettingsService;
import ru.yandex.direct.core.entity.creative.model.Creative;
import ru.yandex.direct.core.entity.creative.repository.CreativeRepository;
import ru.yandex.direct.mediascope.MediascopeClient;
import ru.yandex.direct.mediascope.model.request.MediascopePosition;
import ru.yandex.direct.mediascope.model.request.MediascopePositionParam;

import static java.util.stream.Collectors.groupingBy;
import static java.util.stream.Collectors.toList;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.utils.JsonUtils.fromJson;

@Service
@ParametersAreNonnullByDefault
public class MediascopePositionService {
    private static final Logger logger = LoggerFactory.getLogger(MediascopePositionService.class);
    private static final int YANDEX_ID = 5;
    private static final DateTimeFormatter DATE_TIME_FORMATTER = DateTimeFormatter.ofPattern("yyyy-MM-dd");
    private static final int POSITIONS_BATCH_SIZE = 200;

    private final CampaignRepository campaignRepository;
    private final BannerRelationsRepository bannerRelationsRepository;
    private final BannerCreativeRepository bannerCreativeRepository;
    private final ClientMeasurerSettingsService clientMeasurerSettingsService;
    private final CreativeRepository creativeRepository;
    private final MediascopeClient mediascopeClient;

    public MediascopePositionService(
            CampaignRepository campaignRepository,
            BannerRelationsRepository bannerRelationsRepository,
            BannerCreativeRepository bannerCreativeRepository,
            ClientMeasurerSettingsService clientMeasurerSettingsService,
            CreativeRepository creativeRepository,
            MediascopeClient mediascopeClient
    ) {
        this.campaignRepository = campaignRepository;
        this.bannerRelationsRepository = bannerRelationsRepository;
        this.bannerCreativeRepository = bannerCreativeRepository;
        this.clientMeasurerSettingsService = clientMeasurerSettingsService;
        this.creativeRepository = creativeRepository;
        this.mediascopeClient = mediascopeClient;
    }

    public Map<Long, MediascopePosition> collectPositions(int shard, List<Long> bannerIds) {
        // собираем кампании
        Map<Long, Long> campaignIdsByBannerIds = bannerRelationsRepository
                .getCampaignIdsByBannerIdsForShard(shard, bannerIds);
        List<Long> campaignIds = campaignIdsByBannerIds.values().stream().distinct().collect(toList());
        List<Campaign> campaigns = campaignRepository.getCampaigns(shard, campaignIds);
        Map<Long, Campaign> campaignsMap = listToMap(campaigns, Campaign::getId);

        // собираем настройки клиентов в Mediascope
        List<Long> clientIds = mapList(campaigns, Campaign::getClientId);
        Map<Long, ClientMeasurerSettings> mediascopeSettingsByClientId =
                clientMeasurerSettingsService.getByClientIdsAndSystem(shard, clientIds,
                        ClientMeasurerSystem.MEDIASCOPE);

        // собираем креативы
        Map<Long, Long> creativeIdsForBannerIds =
                bannerCreativeRepository.getBannerIdToCreativeId(shard, bannerIds);
        List<Creative> creatives = creativeRepository.getCreatives(shard, creativeIdsForBannerIds.values());
        Map<Long, Creative> creativesMap = listToMap(creatives, Creative::getId);

        var positions = new HashMap<Long, MediascopePosition>();
        for (var bannerId : bannerIds) {

            var campaign = campaignsMap.get(campaignIdsByBannerIds.get(bannerId));
            if (campaign == null) {
                logger.error("Couldn't find campaign for banner " + bannerId);
                continue;
            }

            var clientMeasurerSettings = mediascopeSettingsByClientId.get(campaign.getClientId());
            var mediascopeSettings = parseMediascopeSettings(clientMeasurerSettings, bannerId);
            if (mediascopeSettings == null || Strings.isNullOrEmpty(mediascopeSettings.getTmsecPrefix())) {
                logger.error("Couldn't get tmsec_prefix for banner " + bannerId);
                continue;
            }

            var creativeId = creativeIdsForBannerIds.get(bannerId);
            var creative = creativesMap.get(creativeId);
            if (creative == null) {
                logger.error("Couldn't find creative for banner " + bannerId);
                continue;
            }

            String positionId = String.format(
                    "%s_%s-%s-%s", mediascopeSettings.getTmsecPrefix(), YANDEX_ID, campaign.getId(), bannerId);

            var position = new MediascopePosition()
                    .setPositionId(positionId)
                    .setPositionName(String.format("%s-%s", campaign.getName(), bannerId))
                    .setCreativeId(creativeId.toString())
                    .setCreativeUrl(creative.getPreviewUrl())
                    .setStartDate(campaign.getStartTime().format(DATE_TIME_FORMATTER))
                    .setFinishDate(ifNotNull(campaign.getFinishTime(), time -> time.format(DATE_TIME_FORMATTER)))
                    .setParams(List.of(
                            new MediascopePositionParam()
                                    .setParamName("campaign_name")
                                    .setParamValue(campaign.getName())))
                    .setAccessToken(mediascopeSettings.getAccessToken());

            positions.put(bannerId, position);
        }

        return positions;
    }

    public Map<String, String> sendPositions(Collection<MediascopePosition> mediascopePositions) {
        Map<String, List<MediascopePosition>> positionsByAccessToken =
                mediascopePositions.stream().collect(groupingBy(MediascopePosition::getAccessToken));

        var errorsMap = new HashMap<String, String>();
        for (var entry : positionsByAccessToken.entrySet()) {
            // шлём в Mediascope пачками на случай массовых изменений
            Iterables.partition(entry.getValue(), POSITIONS_BATCH_SIZE)
                    .forEach(batch -> {
                        var response = mediascopeClient.sendPositions(entry.getKey(), batch);
                        errorsMap.putAll(response.getErrors());
                    });
        }

        return errorsMap;
    }

    private MediascopeClientMeasurerSettings parseMediascopeSettings(
            @Nullable ClientMeasurerSettings clientMeasurerSettings, Long bannerId) {
        if (clientMeasurerSettings == null) {
            logger.warn("Couldn't find client measurer settings for banner " + bannerId);
            return null;
        }

        try {
            return fromJson(clientMeasurerSettings.getSettings(), MediascopeClientMeasurerSettings.class);
        } catch (IllegalArgumentException ex) {
            logger.error("Couldn't parse client measurer settings for banner " + bannerId);
            return null;
        }
    }
}
