package ru.yandex.direct.core.entity.banner.service;

import java.math.BigDecimal;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.base.Objects;
import com.google.common.collect.Sets;
import one.util.streamex.EntryStream;
import org.jooq.DSLContext;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.adgroup.model.PageBlock;
import ru.yandex.direct.core.entity.banner.container.ModerateBannerPagesSyncResult;
import ru.yandex.direct.core.entity.banner.container.OutdoorModerateBannerPagesUpdateParams;
import ru.yandex.direct.core.entity.banner.model.ModerateBannerPage;
import ru.yandex.direct.core.entity.banner.repository.BannerRepository;
import ru.yandex.direct.core.entity.banner.repository.ModerateBannerPagesRepository;
import ru.yandex.direct.core.entity.creative.model.Creative;
import ru.yandex.direct.core.entity.creative.model.VideoFormat;
import ru.yandex.direct.core.entity.creative.repository.CreativeRepository;
import ru.yandex.direct.core.entity.placements.model1.BlockSize;
import ru.yandex.direct.core.entity.placements.model1.OutdoorBlock;
import ru.yandex.direct.core.entity.placements.model1.PlacementBlock;
import ru.yandex.direct.core.entity.placements.model1.PlacementBlockKey;
import ru.yandex.direct.core.entity.placements.repository.PlacementBlockRepository;
import ru.yandex.direct.regions.GeoTree;
import ru.yandex.direct.regions.GeoTreeFactory;

import static com.google.common.base.Preconditions.checkState;
import static java.util.Collections.emptyMap;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toSet;

/**
 * Сервис для обновления записей отправки на внешнюю модерацию для outdoor баннеров и соответствующих пейджей.
 */
@Component
@ParametersAreNonnullByDefault
public class OutdoorModerateBannerPagesUpdater {
    private final ModerateBannerPagesRepository moderateBannerPagesRepository;
    private final PlacementBlockRepository blockRepository;
    private final CreativeRepository creativeRepository;
    private final BannerRepository bannerRepository;
    private final GeoTreeFactory geoTreeFactory;

    @Autowired
    public OutdoorModerateBannerPagesUpdater(
            ModerateBannerPagesRepository moderateBannerPagesRepository,
            PlacementBlockRepository blockRepository,
            CreativeRepository creativeRepository,
            BannerRepository bannerRepository,
            GeoTreeFactory geoTreeFactory) {
        this.moderateBannerPagesRepository = moderateBannerPagesRepository;
        this.blockRepository = blockRepository;
        this.creativeRepository = creativeRepository;
        this.bannerRepository = bannerRepository;
        this.geoTreeFactory = geoTreeFactory;
    }

    /**
     * 1. Определяем параметры синхронизации пейджей для каждого баннера
     * 2. Синхронизируем записи внешней модерации (moderate_banner_pages) - помечаем удаленными лишние, добавляем недостающие
     * 3. Сбрасываем statusBsSynced у баннеров
     */
    public void updateModerateBannerPages(Map<Long, OutdoorModerateBannerPagesUpdateParams> bannerIdToUpdateParams,
                                          DSLContext dslContext) {
        Map<Long, List<ModerateBannerPage>> bannerIdToModerateBannerPages =
                buildBannerToModerateBannerPages(dslContext, bannerIdToUpdateParams);
        ModerateBannerPagesSyncResult syncResult =
                moderateBannerPagesRepository.syncModerateBannerPages(dslContext, bannerIdToModerateBannerPages);

        Set<Long> affectedBannerIds = Sets.union(syncResult.getDeletedBannerIdToPageIds().keySet(),
                syncResult.getAddedBannerIdToPageIds().keySet());
        bannerRepository.common.resetStatusBsSyncedByIds(dslContext, affectedBannerIds);
    }

    /**
     * Отфильтровать и получить мапу bannerId -> newPageIds.
     * <p>
     * Первая фильтрация (фильтрация баннеров):
     * Считаем разницу между гео группы и минус гео баннеров (приходит из внутренней модерации)
     * Если минус гео баннеров полностью вычитает гео группы, то такой баннер не проходит фильтрацию (не добавим для
     * него записи в
     * moderate_banner_pages)
     * <p>
     * Вторая фильтрация (фильтрация pageIds):
     * Если длительность видео креатива баннера не подходит ни под одну из допустимых длительнеостей в блоке пейджа,
     * то такой pageId не проходит фильтрацию (не добавим для него записи в moderate_banner_pages со статусом READY).
     * <p>
     * Третья фильтрация (фильтрация pageIds):
     * Если ни одно из разрешений видео креатива баннера не подходит ни под одно разрешение блока пейджа,
     * то такой pageId не проходит фильтрацию (не добавим для него записи в moderate_banner_pages со статусом READY)
     *
     * @param dslContext             - контекст
     * @param bannerIdToUpdateParams - параметры изменения пейджей группы баннера
     * @return мапа bannerId -> список идентификаторов новых пейджей
     */
    private Map<Long, List<ModerateBannerPage>> buildBannerToModerateBannerPages(
            DSLContext dslContext,
            Map<Long, OutdoorModerateBannerPagesUpdateParams> bannerIdToUpdateParams) {

        Map<Long, Creative> bannerIdToCreative = getBannerCreatives(dslContext, bannerIdToUpdateParams.keySet());
        Map<Long, Map<Long, OutdoorBlock>> pageIdToBlocks = getPageIdToBlockIdData(bannerIdToUpdateParams.values());

        return EntryStream.of(bannerIdToUpdateParams)
                .mapToValue((bannerId, updateParams) -> {
                    List<Long> bannerMinusGeo = updateParams.getBannerMinusGeo();
                    List<Long> adGroupGeo = updateParams.getAdGroupGeo();
                    List<PageBlock> pageBlocks = updateParams.getAdGroupPageBlocks();
                    Long bannerVersion = updateParams.getBannerVersion();

                    if (isFullSubtraction(bannerMinusGeo, adGroupGeo)) {
                        return Collections.<ModerateBannerPage>emptyList();
                    }

                    Creative creative = bannerIdToCreative.get(bannerId);
                    Set<Resolution> creativeResolutions = getCreativeResolutions(creative);
                    Long creativeDuration = getCreativeDuration(creative);

                    return pageBlocks.stream()
                            .filter(pageBlock -> {
                                // фильтруем пейджблоки, неподходящие по длительности под длительность креатива
                                Long pageBlockDuration = getPageBlockDuration(pageBlock, pageIdToBlocks);
                                return pageBlockDuration.equals(creativeDuration);
                            })
                            .filter(pageBlock -> {
                                // фильтруем пейджблоки, неподходящие по разрешению ни под одно разрешение креатива
                                Resolution pageBlockResolution = getPageBlockResolution(pageBlock, pageIdToBlocks);
                                return creativeResolutions.contains(pageBlockResolution);
                            })
                            .map(PageBlock::getPageId)
                            .distinct()
                            .map(pageId -> new ModerateBannerPage()
                                    .withBannerId(bannerId)
                                    .withPageId(pageId)
                                    .withVersion(bannerVersion))
                            .collect(toList());
                }).toMap();
    }

    /**
     * Достать из бд креативовы баннера
     *
     * @param dslContext - контекст
     * @param bannerIds  - баннеры
     * @return key - bannerId, value - креатив баннера
     */
    private Map<Long, Creative> getBannerCreatives(DSLContext dslContext, Set<Long> bannerIds) {
        Map<Long, Creative> creativesByBannerIds = creativeRepository.getCreativesByBannerIds(dslContext, bannerIds);

        checkState(creativesByBannerIds.keySet().equals(bannerIds), "banners with id %s must have creatives",
                Sets.difference(bannerIds, creativesByBannerIds.keySet()));

        return creativesByBannerIds;
    }

    /**
     * Получить длительность креатива в округленных миллисекундах
     */
    private Long getCreativeDuration(Creative creative) {
        BigDecimal duration = creative.getAdditionalData().getDuration();
        return toRoundedMilliseconds(duration);
    }

    /**
     * Получить разрешения видео креатива outdoor баннера в формате {@link Resolution}
     */
    private static Set<Resolution> getCreativeResolutions(Creative creative) {
        return creative.getAdditionalData().getFormats().stream()
                .filter(x -> x.getHeight() != null && x.getWidth() != null)
                .map(Resolution::new)
                .collect(toSet());
    }


    /**
     * Достать из бд данные блоков пейджей.
     * <p>
     * Нельзя группировать только по pageId, т.к. для каждого баннера разрешены свои impId (blockId).
     *
     * @param updateParams - параметры изменения пейджей группы баннера
     * @return key - pageId, value - {key - blockId, value - данные блока}
     */
    private Map<Long, Map<Long, OutdoorBlock>> getPageIdToBlockIdData(
            Collection<OutdoorModerateBannerPagesUpdateParams> updateParams) {
        List<PlacementBlockKey> placementBlockKeys = updateParams.stream()
                .flatMap(ag -> ag.getAdGroupPageBlocks().stream())
                .map(x -> PlacementBlockKey.of(x.getPageId(), x.getImpId()))
                .collect(toList());

        List<PlacementBlock> placementBlocks = blockRepository.getPlacementBlocks(placementBlockKeys);
        List<OutdoorBlock> outdoorBlocks = placementBlocks.stream().map(OutdoorBlock.class::cast).collect(toList());

        Map<Long, Map<Long, OutdoorBlock>> pageIdToBlockIdData = new HashMap<>();
        for (OutdoorBlock block : outdoorBlocks) {
            Long pageId = block.getPageId();
            Long blockId = block.getBlockId();

            Map<Long, OutdoorBlock> blockIdToBlockData =
                    pageIdToBlockIdData.computeIfAbsent(pageId, x -> new HashMap<>());

            blockIdToBlockData.put(blockId, block);
        }
        return pageIdToBlockIdData;
    }

    /**
     * Для pageId + impId получить допустимое разрешение.
     *
     * @param pageBlock          -  pageId + impId (от пользователя)
     * @param pageIdToBlocksData - мапа key - pageId, value - {key - blockId, value - данные блока}
     * @return допустимое разрешение пейджблока
     */
    private static Resolution getPageBlockResolution(PageBlock pageBlock,
                                                     Map<Long, Map<Long, OutdoorBlock>> pageIdToBlocksData) {
        OutdoorBlock outdoorBlock = getOutdoorBlock(pageIdToBlocksData, pageBlock);
        return new Resolution(outdoorBlock.getResolution());
    }

    /**
     * Для pageId + impId получить поддерживаемую длительность видео.
     *
     * @param pageBlock          -  pageId + impId (от пользователя)
     * @param pageIdToBlocksData - мапа key - pageId, value - {key - blockId, value - данные блока}
     * @return допустимая длительность пейджблока
     */
    private static Long getPageBlockDuration(PageBlock pageBlock,
                                             Map<Long, Map<Long, OutdoorBlock>> pageIdToBlocksData) {
        OutdoorBlock outdoorBlock = getOutdoorBlock(pageIdToBlocksData, pageBlock);
        return toRoundedMilliseconds(outdoorBlock.getDuration());
    }

    /**
     * Достать нужный OutdoorBlock из мапы pageIdToBlocksData
     *
     * @param pageIdToBlocksData - мапа key - pageId, value - {key - blockId, value - данные блока}
     * @param pageBlockToSearch  - pageId + blockId
     * @return - искомый блок
     */
    private static OutdoorBlock getOutdoorBlock(Map<Long, Map<Long, OutdoorBlock>> pageIdToBlocksData,
                                                PageBlock pageBlockToSearch) {
        Long pageId = pageBlockToSearch.getPageId();
        Long blockId = pageBlockToSearch.getImpId();
        Map<Long, OutdoorBlock> blockIdToOutdoorBlock = pageIdToBlocksData.getOrDefault(pageId, emptyMap());
        OutdoorBlock outdoorBlock = blockIdToOutdoorBlock.get(blockId);
        checkState(outdoorBlock != null, "Not found data for page: %s block: %s", pageId, blockId);
        return outdoorBlock;
    }

    /**
     * 1) Округлить до одного знака после запятой
     * 2) Перевести в миллисекунды
     * 3.45 с. => 3500 мс.
     *
     * @param seconds - время в секундах (не целочисленное)
     * @return - миллисекунды
     */
    private static Long toRoundedMilliseconds(@Nullable Number seconds) {
        // Нет гарантий, что из ПИ придут заполненные данные
        if (seconds == null) {
            return null;
        }
        return Math.round(seconds.doubleValue() * 10) * 100;
    }

    /**
     * Проверка того, что минус гео баннеров полностью вычитает гео таргетинг группы.
     */
    private boolean isFullSubtraction(List<Long> bannerMinusGeo, List<Long> adGroupGeo) {
        return getGeoTree().excludeRegions(adGroupGeo, bannerMinusGeo).isEmpty();
    }

    private GeoTree getGeoTree() {
        return geoTreeFactory.getGlobalGeoTree();
    }

    /**
     * Общий тип разрешения для сравнения BlockSize и VideoFormat
     */
    private static class Resolution {
        private final int width;
        private final int height;

        Resolution(BlockSize blockSize) {
            this.width = blockSize.getWidth();
            this.height = blockSize.getHeight();
        }

        Resolution(VideoFormat videoFormat) {
            this.width = videoFormat.getWidth();
            this.height = videoFormat.getHeight();
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            Resolution that = (Resolution) o;
            return width == that.width &&
                    height == that.height;
        }

        @Override
        public int hashCode() {
            return Objects.hashCode(width, height);
        }
    }
}
