package ru.yandex.direct.core.entity.banner.service.execution

import org.springframework.stereotype.Service
import ru.yandex.direct.core.entity.banner.container.BannerAdditionalActionsContainer
import ru.yandex.direct.core.entity.banner.container.BannerRepositoryContainer
import ru.yandex.direct.core.entity.banner.container.BannersAddOperationContainerImpl
import ru.yandex.direct.core.entity.banner.model.BannerWithAdGroupId
import ru.yandex.direct.core.entity.banner.model.BannerWithBannerImage
import ru.yandex.direct.core.entity.banner.model.BannerWithCreative
import ru.yandex.direct.core.entity.banner.model.BannerWithSitelinks
import ru.yandex.direct.core.entity.banner.model.BannerWithSystemFields
import ru.yandex.direct.core.entity.banner.model.BannerWithTurboLanding
import ru.yandex.direct.core.entity.banner.model.BannerWithVcard
import ru.yandex.direct.core.entity.creative.model.Creative
import ru.yandex.direct.core.entity.uac.grut.GrutContext
import ru.yandex.direct.core.entity.uac.grut.GrutTransactionProvider
import ru.yandex.direct.core.grut.api.AdGroupGrut
import ru.yandex.direct.core.grut.api.AdGroupGrutApi
import ru.yandex.direct.core.grut.api.BannerGrut
import ru.yandex.direct.core.grut.api.BannerGrutApi
import ru.yandex.direct.core.grut.api.CampaignGrutApi
import ru.yandex.direct.core.grut.api.CampaignGrutModel
import ru.yandex.direct.core.grut.api.ClientGrutApi
import ru.yandex.direct.core.grut.api.ClientGrutModel
import ru.yandex.direct.core.grut.api.CreativeGrutApi
import ru.yandex.direct.core.grut.api.GrutApiBase.Companion.GRUT_CHANGE_OBJECTS_ATTEMPTS
import ru.yandex.direct.core.grut.api.GrutApiBase.Companion.GRUT_GET_OBJECTS_ATTEMPTS
import ru.yandex.direct.core.grut.api.VCardGrutApi
import ru.yandex.direct.dbutil.model.ClientId
import ru.yandex.direct.feature.FeatureName
import ru.yandex.grut.proto.transaction_context.TransactionContext.TTransactionContext

@Service
class BannersAddExecutionGrutService(
    grutContext: GrutContext,
    private val grutTransactionProvider: GrutTransactionProvider,
    private val grutStubObjectsService: GrutStubObjectsService,
) : BannersAddExecutionService {

    private val clientGrutApi = ClientGrutApi(grutContext)
    private val campaignGrutApi = CampaignGrutApi(grutContext)
    private val adGroupGrutApi = AdGroupGrutApi(grutContext)
    private val bannerGrutApi = BannerGrutApi(grutContext)
    private val vcardGrutApi = VCardGrutApi(grutContext)
    private val creativeGrutApi = CreativeGrutApi(grutContext)

    /**
     * Добавляет баннеры
     */
    override fun execute(
        validBannersToApply: List<BannerWithAdGroupId>,
        operationContainer: BannersAddOperationContainerImpl,
        repositoryContainer: BannerRepositoryContainer,
        additionalActionsContainer: BannerAdditionalActionsContainer
    ): List<Long> {

        val banners = validBannersToApply.map { it as BannerWithSystemFields }
        createClientIfNotExists(operationContainer)
        val directIdToCampaignId = createCampaignsIfNotExists(banners, operationContainer)
        createAdGroupsIfNotExists(banners, operationContainer, directIdToCampaignId)
        createVCardStubsIfNotExists(banners, operationContainer.clientId)
        createCreativeStubsIfNotExists(banners, operationContainer.clientId, operationContainer.creativeByIdMap)

        val idToSitelinkSet = operationContainer.sitelinkSets
        val idToTurboLanding = operationContainer.turboLandings ?: emptyMap()
        val bannersGrut = banners.map { banner ->
            val sitelinkSet = (banner as? BannerWithSitelinks)?.sitelinksSetId?.let { idToSitelinkSet[it] }
            val sitelinkTurboLandingIds: Set<Long> =
                sitelinkSet?.sitelinks?.mapNotNull { it.turboLandingId }?.toSet() ?: emptySet()
            val bannerTurboLandingId = (banner as? BannerWithTurboLanding)?.turboLandingId
            val allTurboLandingIds = if (bannerTurboLandingId != null) {
                setOf(bannerTurboLandingId) + sitelinkTurboLandingIds
            } else {
                sitelinkTurboLandingIds
            }
            val turbolandingsById = allTurboLandingIds.associateWith { idToTurboLanding[it]!! }

            val imageHash = (banner as? BannerWithBannerImage)?.imageHash
            val bannerImageFormat = if (imageHash != null) {
                operationContainer.bannerImageFormats[imageHash]
            } else {
                null
            }

            val enabledModerationInGrut = operationContainer.clientEnabledFeatures
                .contains(FeatureName.MODERATION_BANNERS_IN_GRUT_ENABLED.getName())
            BannerGrut(
                banner,
                directIdToCampaignId[banner.campaignId]!!,
                sitelinkSet,
                turbolandingsById,
                bannerImageFormat,
                skipModeration = enabledModerationInGrut.not(),
            )
        }

        // Указываем шард, чтобы он был виден в watch log'e
        val transactionContext = TTransactionContext.newBuilder().apply {
            shard = operationContainer.shard
        }.build()
        grutTransactionProvider.runInRetryableTransaction(GRUT_CHANGE_OBJECTS_ATTEMPTS, transactionContext) {
            bannerGrutApi.createObjects(bannersGrut)
        }

        return banners.map { it.id }
    }

    // Создаёт заглушки для тех визиток, которых ещё нет в груте
    // Это нужно для последующего создания баннеров: vcard_id должен указывать на существующий объект
    // А само тело объекта приедет по репликации
    private fun createVCardStubsIfNotExists(banners: List<BannerWithSystemFields>, clientId: ClientId) {
        val vcardIds = banners
            .mapNotNull { it as? BannerWithVcard }
            .mapNotNull { it.vcardId }

        grutStubObjectsService.createStubsForMissingVcards(clientId, vcardIds)
    }

    private fun createCreativeStubsIfNotExists(
        banners: List<BannerWithSystemFields>,
        clientId: ClientId,
        creativeByIdMap: MutableMap<Long, Creative>
    ) {
        val bannersWithCreatives = banners.mapNotNull { it as? BannerWithCreative }

        val creativeIds = bannersWithCreatives
            .mapNotNull { it.creativeId }

        grutStubObjectsService.createStubsForMissingCreatives(clientId, creativeIds, creativeByIdMap)
    }

    private fun createClientIfNotExists(operationContainer: BannersAddOperationContainerImpl) {
        grutTransactionProvider.runRetryable(GRUT_CHANGE_OBJECTS_ATTEMPTS) {
            val clientGrut = clientGrutApi.getClient(operationContainer.clientId.asLong())
            if (clientGrut == null) {
                clientGrutApi.createObjects(listOf(ClientGrutModel(operationContainer.client, listOf())))
            }
        }
    }

    private fun createCampaignsIfNotExists(
        banners: List<BannerWithSystemFields>,
        operationContainer: BannersAddOperationContainerImpl
    ): Map<Long, Long> {
        val directCampaignIds = banners.map { it.campaignId }
        return grutTransactionProvider.runRetryable(GRUT_CHANGE_OBJECTS_ATTEMPTS) {
            val directIdsToCampaignIds = campaignGrutApi.getCampaignIdsByDirectIds(directCampaignIds)

            val missingCampaignIds = directCampaignIds.filter { directIdsToCampaignIds[it] == null }.toSet()

            val newDirectIdsToCampaignIds = if (missingCampaignIds.isEmpty()) {
                directIdsToCampaignIds
            } else {
                val campaigns = banners
                    .filter { missingCampaignIds.contains(it.campaignId) }
                    .map { banner -> operationContainer.getCampaign(banner) }
                    .map { CampaignGrutModel(it, 0) }
                    .distinct()

                campaignGrutApi.createObjects(campaigns)
                campaignGrutApi.getCampaignIdsByDirectIds(directCampaignIds)
            }
            newDirectIdsToCampaignIds
        }
    }

    private fun createAdGroupsIfNotExists(
        banners: List<BannerWithSystemFields>,
        operationContainer: BannersAddOperationContainerImpl,
        directIdToCampaignId: Map<Long, Long>
    ) {
        val adGroupIds = banners.map { it.adGroupId }
        return grutTransactionProvider.runRetryable(GRUT_GET_OBJECTS_ATTEMPTS) {
            val existingAdGroupIds = adGroupGrutApi.getExistingObjects(adGroupIds).toSet()

            val missingAdGroups = adGroupIds.minus(existingAdGroupIds)

            val adGroups = banners
                .filter { missingAdGroups.contains(it.adGroupId) }
                .map { b ->
                    AdGroupGrut(
                        id = b.adGroupId,
                        orderId = directIdToCampaignId[b.campaignId]!!,
                        type = operationContainer.getAdGroup(b)!!.type,
                        regions = operationContainer.getAdGroup(b)!!.geo,
                        cryptaSegmentsMapping = emptyMap()
                    )
                }
                .distinct()

            adGroupGrutApi.createObjects(adGroups)
        }
    }

    override fun afterExecution(
        validModelsMapToApply: Map<Int, BannerWithAdGroupId>,
        operationContainer: BannersAddOperationContainerImpl
    ) {
    }
}
