package ru.yandex.direct.core.entity.banner.service.execution

import org.slf4j.LoggerFactory
import org.springframework.stereotype.Service
import ru.yandex.direct.core.entity.banner.container.BannersUpdateOperationContainerImpl
import ru.yandex.direct.core.entity.banner.model.Banner
import ru.yandex.direct.core.entity.banner.repository.type.BannerRepositoryTypeSupport
import ru.yandex.direct.core.entity.uac.grut.GrutContext
import ru.yandex.direct.core.entity.uac.grut.GrutTransactionProvider
import ru.yandex.direct.core.grut.api.BannerGrutApi
import ru.yandex.direct.core.grut.api.GrutApiBase
import ru.yandex.direct.core.grut.api.UpdatedObject
import ru.yandex.direct.model.AppliedChanges
import ru.yandex.direct.model.ModelProperty
import ru.yandex.grut.objects.proto.client.Schema
import ru.yandex.grut.proto.transaction_context.TransactionContext

/**
 * Контейнер, в который собираются изменения отдельных полей баннера
 * Хранит в себе модифицированный баннер и наборы изменённых полей
 */
data class BannerChangesContainer(
    val id: Long,
    val bannerBuilder: Schema.TBannerV2.Builder,
    val setPaths: HashSet<String> = HashSet(),
    val removePaths: HashSet<String> = HashSet(),
    var vcardIdToCheckIfExist: Long? = null,
    var creativeIdToCheckIfExist: Long? = null,
) {
    fun toUpdatedObject(): UpdatedObject {
        return UpdatedObject(
            bannerBuilder.meta.toByteString(),
            bannerBuilder.spec.toByteString(),
            setPaths.toList(),
            null,
            removePaths.toList()
        )
    }
}

@Service
class BannersUpdateExecutionGrutService(
    grutContext: GrutContext,
    private val grutTransactionProvider: GrutTransactionProvider,
    private val grutStubObjectsService: GrutStubObjectsService,
    private val typeSupports: List<BannerRepositoryTypeSupport<*>>
) {
    private val bannerGrutApi = BannerGrutApi(grutContext)

    companion object {
        private val logger = LoggerFactory.getLogger(BannersUpdateExecutionGrutService::class.java)
    }

    @Suppress("UNCHECKED_CAST")
    fun <T : Banner> execute(
        appliedChangesList: List<AppliedChanges<T>>,
        operationContainer: BannersUpdateOperationContainerImpl
    ): List<Long> {
        val banners = bannerGrutApi.getBannersByDirectId(appliedChangesList.map { it.model.id }.toSet())

        // Берём только те баннеры, которые есть в груте, остальное скипаем
        val filteredAppliedChangesList = filterAppliedChanges(appliedChangesList, banners)
        if (filteredAppliedChangesList.isEmpty()) {
            return emptyList()
        }

        val bannerIds = filteredAppliedChangesList.map { it.model.id }.toList()

        val bannersById = banners.associateBy { it.meta.directId }

        // Сконвертируем полученные из грута объекты в builder'ы, которые можно модифицировать
        val bannerContainersById = filteredAppliedChangesList.map {
            BannerChangesContainer(
                it.model.id,
                bannersById[it.model.id]!!.toBuilder()
            )
        }.associateBy { it.id }

        // Для каждого ModelProperty храним список AppliedChanges, в которых есть изменения этого поля
        val appliedChangesByModelProperty: Map<ModelProperty<*, *>, List<AppliedChanges<T>>> =
            filteredAppliedChangesList.flatMap { appliedChanges ->
                appliedChanges.propertiesForUpdate.map { Pair(it, appliedChanges) }
            }.groupBy({ it.first }, { it.second })

        val processedModelProperties = HashSet<ModelProperty<*, *>>()
        for (typeSupport in typeSupports) {
            val grutSupportedProperties = typeSupport.grutSupportedProperties
            if (grutSupportedProperties.isEmpty()) {
                continue
            }

            val affectedAppliedChangesSet = HashSet<AppliedChanges<T>>()
            for (grutSupportedProperty in grutSupportedProperties) {
                val affectedAppliedChangesList = appliedChangesByModelProperty[grutSupportedProperty]
                if (affectedAppliedChangesList != null) {
                    affectedAppliedChangesSet.addAll(affectedAppliedChangesList)
                }
            }

            if (affectedAppliedChangesSet.isNotEmpty()) {
                val bannersBuilderMap = affectedAppliedChangesSet
                    .map { bannerContainersById[it.model.id]!! }
                    .associateBy({ it.id }, { it.bannerBuilder })
                val modifiedPathsMap = typeSupport.applyToGrutObjects(
                    bannersBuilderMap,
                    affectedAppliedChangesSet as Collection<Nothing>,
                    operationContainer
                )

                for ((id, modifiedPaths) in modifiedPathsMap) {
                    val bannerChangesContainer = bannerContainersById[id]!!
                    bannerChangesContainer.setPaths.addAll(modifiedPaths.setPaths)
                    bannerChangesContainer.removePaths.addAll(modifiedPaths.removePaths)
                    if (modifiedPaths.vcardIdToCheckIfExist != null) {
                        bannerChangesContainer.vcardIdToCheckIfExist = modifiedPaths.vcardIdToCheckIfExist
                    }
                    if (modifiedPaths.creativeIdToCheckIfExist != null) {
                        bannerChangesContainer.creativeIdToCheckIfExist = modifiedPaths.creativeIdToCheckIfExist
                    }
                }
            }

            processedModelProperties.addAll(grutSupportedProperties)
        }

        // Залогируем поля, для которых не нашлось обработчиков
        val skippedModelProperties: Set<ModelProperty<*, *>> =
            appliedChangesByModelProperty.keys - processedModelProperties
        if (skippedModelProperties.isNotEmpty()) {
            logger.warn(
                "Skipped ${skippedModelProperties.size} model properties " +
                    "since there are no support: $skippedModelProperties"
            )
        }

        val bannerContainersWithChanges = bannerContainersById.values
            .filter { it.setPaths.isNotEmpty() || it.removePaths.isNotEmpty() }

        val bannersToUpdate: List<UpdatedObject> = bannerContainersWithChanges
            .map { it.toUpdatedObject() }.toList()

        if (bannersToUpdate.isEmpty()) {
            return bannerIds
        }

        // Указываем шард, чтобы он был виден в watch log'e
        val transactionContext = TransactionContext.TTransactionContext.newBuilder().apply {
            shard = operationContainer.shard
        }.build()

        // Предсоздаём заглушки для vcard'ов, если нужно
        val vcardIdsToCheckExist = bannerContainersWithChanges.mapNotNull { it.vcardIdToCheckIfExist }.toSet()
        if (vcardIdsToCheckExist.isNotEmpty()) {
            grutStubObjectsService.createStubsForMissingVcards(operationContainer.clientId, vcardIdsToCheckExist)
        }
        // Аналогично с креативами
        val creativeIdsToCheckExist = bannerContainersWithChanges.mapNotNull { it.creativeIdToCheckIfExist }.toSet()
        if (creativeIdsToCheckExist.isNotEmpty()) {
            grutStubObjectsService.createStubsForMissingCreatives(
                operationContainer.clientId, creativeIdsToCheckExist, operationContainer.creativeByIdMap
            )
        }

        // Сохранение собственно баннеров
        grutTransactionProvider.runInRetryableTransaction(
            GrutApiBase.GRUT_CHANGE_OBJECTS_ATTEMPTS,
            transactionContext
        ) {
            bannerGrutApi.updateBanners(bannersToUpdate)
        }

        // TODO : понять, что будем делать с additionalActions в груте

        return bannerIds
    }

    private fun <T : Banner> filterAppliedChanges(
        appliedChanges: List<AppliedChanges<T>>,
        bannersInGrut: List<Schema.TBannerV2>
    ): List<AppliedChanges<T>> {
        val existingBannerIds = bannersInGrut.map { it.meta.directId }.toSet()
        val (existing, notExisting) = appliedChanges.partition { it.model.id in existingBannerIds }
        if (notExisting.isNotEmpty()) {
            val missingIds = notExisting.map { it.model.id }.toList()
            logger.warn("Skip ${notExisting.size} banners that are not in GrUT: $missingIds")
        }
        return existing
    }
}
