package ru.yandex.direct.core.entity.banner.service.moderation;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;

import one.util.streamex.StreamEx;
import org.jooq.DSLContext;
import org.jooq.TransactionalRunnable;

import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.core.entity.banner.container.BannerRepositoryContainer;
import ru.yandex.direct.core.entity.banner.model.BannerWithSystemFields;
import ru.yandex.direct.core.entity.banner.model.TextBanner;
import ru.yandex.direct.core.entity.banner.repository.BannerModifyRepository;
import ru.yandex.direct.core.entity.banner.repository.BannerTypedRepository;
import ru.yandex.direct.core.entity.banner.service.moderation.type.BannerModerateOperationSupportFacade;
import ru.yandex.direct.core.entity.banner.service.validation.ModerateBannerValidationService;
import ru.yandex.direct.core.entity.campaign.model.CampaignStatusModerate;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.creative.model.Creative;
import ru.yandex.direct.core.entity.creative.model.CreativeType;
import ru.yandex.direct.core.entity.creative.model.StatusModerate;
import ru.yandex.direct.core.entity.creative.service.CreativeService;
import ru.yandex.direct.core.entity.moderation.repository.ModerationRepository;
import ru.yandex.direct.core.entity.user.repository.UserRepository;
import ru.yandex.direct.dbschema.ppc.enums.UsersOptionsStatuspostmoderate;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.operation.Applicability;
import ru.yandex.direct.operation.update.AppliedChangesValidatedStep;
import ru.yandex.direct.operation.update.ExecutionStep;
import ru.yandex.direct.operation.update.SimpleAbstractUpdateOperation;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.core.entity.banner.service.validation.BannerConstants.NEW_SENSITIVE_PROPERTIES;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

public class BannerModerateOperation extends SimpleAbstractUpdateOperation<BannerWithSystemFields, Long> {
    private final int shard;
    private final ClientId clientId;
    private final Long operatorUid;
    private final AdGroupRepository adGroupRepository;
    private final ModerationRepository moderationRepository;
    private final DslContextProvider dslContextProvider;
    private final CreativeService creativeService;
    private final CampaignRepository campaignRepository;
    private final UserRepository userRepository;
    private final ModerateBannerValidationService moderateBannerValidationService;
    private final BannerModerateOperationSupportFacade bannerModerateOperationSupportFacade;
    private final BannerTypedRepository typedRepository;
    private final BannerModifyRepository modifyRepository;
    private final BannerRepositoryContainer bannerRepositoryContainer;
    private final BannerModerationHelper bannerModerationHelper;
    private final BannerModerateOptions options;

    private TransactionalRunnable postExecute;

    public <B extends BannerWithSystemFields> BannerModerateOperation(
            List<Long> bannerIds,
            int shard, long operatorUid, ClientId clientId,
            AdGroupRepository adGroupRepository,
            ModerationRepository moderationRepository,
            DslContextProvider dslContextProvider,
            CreativeService creativeService,
            CampaignRepository campaignRepository,
            UserRepository userRepository,
            ModerateBannerValidationService moderateBannerValidationService,
            BannerModerateOperationSupportFacade bannerModerateOperationSupportFacade,
            BannerTypedRepository typedRepository,
            BannerModifyRepository modifyRepository,
            BannerModerationHelper bannerModerationHelper,
            BannerModerateOptions options) {
        super(Applicability.PARTIAL,
                mapList(bannerIds, id -> new ModelChanges<>(id, BannerWithSystemFields.class)),
                id -> new TextBanner().withId(id),
                NEW_SENSITIVE_PROPERTIES);
        this.shard = shard;
        this.clientId = clientId;
        this.operatorUid = operatorUid;
        this.adGroupRepository = adGroupRepository;
        this.moderationRepository = moderationRepository;
        this.dslContextProvider = dslContextProvider;
        this.bannerModerationHelper = bannerModerationHelper;
        this.creativeService = creativeService;
        this.campaignRepository = campaignRepository;
        this.userRepository = userRepository;
        this.moderateBannerValidationService = moderateBannerValidationService;
        this.bannerModerateOperationSupportFacade = bannerModerateOperationSupportFacade;
        this.typedRepository = typedRepository;
        this.modifyRepository = modifyRepository;
        this.options = options;

        bannerRepositoryContainer = new BannerRepositoryContainer(shard);
    }

    @Override
    protected ValidationResult<List<ModelChanges<BannerWithSystemFields>>, Defect> validateModelChanges(
            List<ModelChanges<BannerWithSystemFields>> modelChanges) {
        return moderateBannerValidationService.validateAccess(operatorUid, clientId, modelChanges);
    }

    @Override
    protected ValidationResult<List<ModelChanges<BannerWithSystemFields>>, Defect> validateModelChangesBeforeApply(
            ValidationResult<List<ModelChanges<BannerWithSystemFields>>, Defect> preValidateResult,
            Map<Long, BannerWithSystemFields> models) {
        return moderateBannerValidationService.validateModerate(shard, clientId, preValidateResult, models);
    }

    @Override
    protected void onAppliedChangesValidated(AppliedChangesValidatedStep<BannerWithSystemFields> step) {
        Collection<AppliedChanges<BannerWithSystemFields>> changes = step.getValidAppliedChanges();
        Map<Long, AppliedChanges<AdGroup>> adGroupChangesByBannerId = prepareAdGroupChangesByBannerId(changes);
        Map<Long, AppliedChanges<Creative>> creativeChangesByBannerId =
                bannerModerationHelper.prepareCreativeChangesByBannerId(shard, changes);

        changes.forEach(banner -> {
            long bannerId = banner.getModel().getId();
            bannerModerateOperationSupportFacade
                    .getModerateOperationSupport(banner)
                    .moderate(adGroupChangesByBannerId.get(bannerId), banner, creativeChangesByBannerId.get(bannerId),
                            options);
        });

        Set<Long> campaignsToModerate = listToSet(changes, ac -> ac.getModel().getCampaignId());
        campaignsToModerate.removeAll(campaignRepository.getCampaignsWithActiveBanners(shard, campaignsToModerate));

        Set<Long> bannersToPostModerate = getBannersForPostModerate(changes);
        List<Long> performanceCreativeIds = StreamEx.ofValues(creativeChangesByBannerId)
                .filter(c -> c.getModel().getType() == CreativeType.PERFORMANCE)
                .filter(c -> c.getModel().getStatusModerate() == StatusModerate.NEW
                        || c.getModel().getStatusModerate() == StatusModerate.ERROR)
                .map(c -> c.getModel().getId())
                .toList();

        postExecute = conf -> {
            DSLContext context = conf.dsl();
            adGroupRepository.updateAdGroups(shard, clientId, adGroupChangesByBannerId.values());
            campaignRepository.setStatusModerateIfNewOrNo(context, campaignsToModerate, CampaignStatusModerate.READY);
            moderationRepository.addPostModerate(context, bannersToPostModerate);
            creativeService.setGeoForUnmoderatedCreatives(shard, clientId, performanceCreativeIds);
            creativeService.sendCreativesToModeration(shard, performanceCreativeIds);
        };
    }

    /**
     * Собирает ID баннеров для добавления в таблицу ppc.post_moderate
     */
    private Set<Long> getBannersForPostModerate(Collection<AppliedChanges<BannerWithSystemFields>> changes) {
        Map<Long, UsersOptionsStatuspostmoderate> userPostModerateByCampaignId = userRepository
                .getUserStatusPostModerateByCampaignId(shard,
                        StreamEx.of(changes).map(ac -> ac.getModel().getCampaignId()).toSet());
        return StreamEx.of(changes)
                .map(AppliedChanges::getModel)
                .mapToEntry(b -> userPostModerateByCampaignId.get(b.getCampaignId()))
                .filterValues(status -> status == UsersOptionsStatuspostmoderate.Yes)
                .keys()
                .map(BannerWithSystemFields::getId)
                .toSet();
    }

    /**
     * Подготавливает {@link AppliedChanges} для групп баннеров и возвращает их в связке с ID баннера.
     * Для баннеров из разных групп возвращаются указатели на один объект
     */
    private Map<Long, AppliedChanges<AdGroup>> prepareAdGroupChangesByBannerId(
            Collection<AppliedChanges<BannerWithSystemFields>> changes) {
        Map<Long, AppliedChanges<AdGroup>> adGroupsById = StreamEx
                .of(adGroupRepository.getAdGroups(shard, mapList(changes, c -> c.getModel().getAdGroupId())))
                .toMap(AdGroup::getId, adGroup -> new ModelChanges<>(adGroup.getId(), AdGroup.class).applyTo(adGroup));

        return StreamEx.of(changes)
                .mapToEntry(c -> c.getModel().getId(), c -> adGroupsById.get(c.getModel().getAdGroupId()))
                .toMap();
    }

    @Override
    protected List<Long> execute(List<AppliedChanges<BannerWithSystemFields>> changes) {
        modifyRepository.update(bannerRepositoryContainer, changes);
        return mapList(changes, a -> a.getModel().getId());
    }

    @Override
    protected void afterExecution(ExecutionStep<BannerWithSystemFields> executionStep) {
        dslContextProvider.ppc(shard).transaction(postExecute);
    }

    @Override
    protected Collection<BannerWithSystemFields> getModels(Collection<Long> ids) {
        return typedRepository.getStrictlyFullyFilled(shard, ids, BannerWithSystemFields.class);
    }
}
