package ru.yandex.direct.core.entity.banner.service.moderation;

import java.util.Comparator;
import java.util.List;
import java.util.function.Consumer;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.core.entity.banner.repository.BannerModifyRepository;
import ru.yandex.direct.core.entity.banner.repository.BannerTypedRepository;
import ru.yandex.direct.core.entity.banner.service.moderation.type.BannerModerateOperationSupportFacade;
import ru.yandex.direct.core.entity.banner.service.validation.ModerateBannerValidationService;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.creative.service.CreativeService;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.moderation.repository.ModerationRepository;
import ru.yandex.direct.core.entity.moderation.repository.sending.remoderation.RemoderationFlagsRepository;
import ru.yandex.direct.core.entity.user.repository.UserRepository;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.operation.Operation;
import ru.yandex.direct.operation.creator.IgnoreDuplicatesOperationCreator;
import ru.yandex.direct.operation.creator.OperationCreator;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.result.Result;

import static ru.yandex.direct.core.entity.banner.service.validation.defects.BannerDefects.adMoreThanOnceInRequest;

@Service
@ParametersAreNonnullByDefault
public class BannerModerateService {
    private final ShardHelper shardHelper;

    private final AdGroupRepository adGroupRepository;
    private final ModerationRepository moderationRepository;
    private final DslContextProvider dslContextProvider;
    private final CampaignRepository campaignRepository;
    private final CreativeService creativeService;
    private final UserRepository userRepository;
    private final ModerateBannerValidationService moderateBannerValidationService;
    private final BannerModerateOperationSupportFacade bannerModerateOperationSupportFacade;
    private final BannerTypedRepository typedRepository;
    private final BannerModifyRepository modifyRepository;
    private final RemoderationFlagsRepository remoderationFlagsRepository;
    private final FeatureService featureService;
    private final BannerModerationHelper bannerModerationHelper;

    @Autowired
    public BannerModerateService(
            ShardHelper shardHelper,
            AdGroupRepository adGroupRepository,
            ModerationRepository moderationRepository,
            DslContextProvider dslContextProvider,
            CampaignRepository campaignRepository,
            CreativeService creativeService,
            UserRepository userRepository,
            ModerateBannerValidationService moderateBannerValidationService,
            BannerModerateOperationSupportFacade bannerModerateOperationSupportFacade,
            BannerTypedRepository typedRepository,
            BannerModifyRepository modifyRepository,
            RemoderationFlagsRepository remoderationFlagsRepository,
            FeatureService featureService, BannerModerationHelper bannerModerationHelper) {
        this.shardHelper = shardHelper;
        this.adGroupRepository = adGroupRepository;
        this.moderationRepository = moderationRepository;
        this.dslContextProvider = dslContextProvider;
        this.campaignRepository = campaignRepository;
        this.creativeService = creativeService;
        this.userRepository = userRepository;
        this.moderateBannerValidationService = moderateBannerValidationService;
        this.bannerModerateOperationSupportFacade = bannerModerateOperationSupportFacade;
        this.typedRepository = typedRepository;
        this.modifyRepository = modifyRepository;
        this.remoderationFlagsRepository = remoderationFlagsRepository;
        this.featureService = featureService;
        this.bannerModerationHelper = bannerModerationHelper;
    }

    public MassResult<Long> moderateBanners(
            ClientId clientId, long operatorUid, List<Long> bannerIds) {
        return moderateBanners(clientId, operatorUid, bannerIds, new BannerModerateOptions());
    }

    /**
     * Отправляет баннеры на модерацию по списку их ID.
     *
     * @param bannerIds Список ID баннеров для отправки на модерацию
     */
    public MassResult<Long> moderateBanners(
            ClientId clientId, long operatorUid, List<Long> bannerIds, BannerModerateOptions options) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        OperationCreator<Long, Operation<Long>> operationCreator = ids ->
                new BannerModerateOperation(ids,
                        shard,
                        operatorUid,
                        clientId,
                        adGroupRepository,
                        moderationRepository,
                        dslContextProvider,
                        creativeService,
                        campaignRepository,
                        userRepository,
                        moderateBannerValidationService,
                        bannerModerateOperationSupportFacade,
                        typedRepository,
                        modifyRepository,
                        bannerModerationHelper,
                        options
                        );

        Consumer<Result<Long>> resultVisitor = result -> result.getValidationResult()
                .addWarning(adMoreThanOnceInRequest());

        return new IgnoreDuplicatesOperationCreator<>(operationCreator, Comparator.naturalOrder(), resultVisitor)
                .create(bannerIds)
                .prepareAndApply();
    }

    /**
     * Отправляет объявления на перемодерацию по списку их ID.
     * Доступно клиентам для отклоненных по фиксированному списку причин объявлений
     *
     * @param bannerIds Список ID баннеров для отправки на перемодерацию
     */
    public MassResult<Long> remoderateBanners(ClientId clientId, long operatorUid, List<Long> bannerIds) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        OperationCreator<Long, Operation<Long>> operationCreator = ids ->
                new BannerRemoderateOperation(ids, shard, operatorUid, clientId,
                        dslContextProvider,
                        moderateBannerValidationService,
                        bannerModerateOperationSupportFacade,
                        typedRepository,
                        modifyRepository,
                        featureService.getEnabledForClientId(clientId),
                        remoderationFlagsRepository,
                        bannerModerationHelper);

        Consumer<Result<Long>> resultVisitor = result -> result.getValidationResult()
                .addWarning(adMoreThanOnceInRequest());

        return new IgnoreDuplicatesOperationCreator<>(operationCreator, Comparator.naturalOrder(), resultVisitor)
                .create(bannerIds)
                .prepareAndApply();
    }

}
