package ru.yandex.direct.core.entity.banner.service.moderation;

import java.time.LocalDateTime;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;

import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.StatusBsSynced;
import ru.yandex.direct.core.entity.banner.model.BannerDisplayHrefStatusModerate;
import ru.yandex.direct.core.entity.banner.model.BannerStatusModerate;
import ru.yandex.direct.core.entity.banner.model.BannerStatusPostModerate;
import ru.yandex.direct.core.entity.banner.model.BannerStatusSitelinksModerate;
import ru.yandex.direct.core.entity.banner.model.BannerVcardStatusModerate;
import ru.yandex.direct.core.entity.banner.model.BannerWithBannerImage;
import ru.yandex.direct.core.entity.banner.model.BannerWithCreative;
import ru.yandex.direct.core.entity.banner.model.BannerWithDisplayHref;
import ru.yandex.direct.core.entity.banner.model.BannerWithImage;
import ru.yandex.direct.core.entity.banner.model.BannerWithSitelinks;
import ru.yandex.direct.core.entity.banner.model.BannerWithSystemFields;
import ru.yandex.direct.core.entity.banner.model.BannerWithVcard;
import ru.yandex.direct.core.entity.banner.repository.BannerTypedRepository;
import ru.yandex.direct.core.entity.creative.model.Creative;
import ru.yandex.direct.core.entity.creative.repository.CreativeRepository;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelChanges;

import static java.util.function.Function.identity;
import static java.util.stream.Collectors.toSet;

@Component
public class BannerModerationHelper {

    private final BannerTypedRepository bannerTypedRepository;
    private final CreativeRepository creativeRepository;

    @Autowired
    public BannerModerationHelper(BannerTypedRepository bannerTypedRepository, CreativeRepository creativeRepository) {
        this.bannerTypedRepository = bannerTypedRepository;
        this.creativeRepository = creativeRepository;
    }

    /**
     * Выставляет дополнительные флаги, актуальные для всех изменяемых баннеров независимо от условий показа группы
     */
    public void setAdditionalFlags(Collection<AppliedChanges<BannerWithSystemFields>> changes, boolean archive) {
        LocalDateTime now = LocalDateTime.now();
        changes.forEach(ac -> {
            if (ac.hasActuallyChangedProps()) {
                ac.modify(BannerWithSystemFields.STATUS_BS_SYNCED, StatusBsSynced.NO);
                ac.modify(BannerWithSystemFields.LAST_CHANGE, now);
            }
            if (archive) {
                ac.modify(BannerWithSystemFields.STATUS_SHOW, false);
            }
        });
    }

    /**
     * Добавляет флаги, проставляемые только для баннеров из групп с условиями показа.
     */
    public void setFlagsForBannersWithShowConditions(Collection<AppliedChanges<BannerWithSystemFields>> changes) {
        changes.forEach(ac -> {
            ac.modify(BannerWithSystemFields.STATUS_MODERATE, BannerStatusModerate.READY);
            ac.modify(BannerWithSystemFields.STATUS_POST_MODERATE, BannerStatusPostModerate.NO);
            ac.modifyIf(BannerWithSitelinks.STATUS_SITELINKS_MODERATE, BannerStatusSitelinksModerate.READY,
                    BannerWithSitelinks.class, b -> b.getSitelinksSetId() != null);
            ac.modifyIf(BannerWithDisplayHref.DISPLAY_HREF_STATUS_MODERATE,
                    BannerDisplayHrefStatusModerate.READY, BannerWithDisplayHref.class,
                    b -> b.getDisplayHref() != null);
            ac.modifyIf(BannerWithVcard.VCARD_STATUS_MODERATE, BannerVcardStatusModerate.READY,
                    BannerWithVcard.class, b -> b.getVcardId() != null);
        });
    }

    /**
     * Возвращает список ID картинок баннеров (ТГО), которые присутствуют в измененных баннерах.
     */
    public Set<Long> collectBannersWithImagesToModerate(Collection<AppliedChanges<BannerWithSystemFields>> changes) {
        return StreamEx.of(changes)
                .map(AppliedChanges::getModel)
                .select(BannerWithImage.class)
                .filter(b -> b.getImageId() != null)
                .map(BannerWithImage::getImageId)
                .toSet();
    }

    /**
     * Возвращает список ID баннеров (ГО), которые надо отправить на модерацию
     */
    public Set<Long> collectBannersWithBannerImagesToModerate(
            Collection<AppliedChanges<BannerWithSystemFields>> changes) {
        return StreamEx.of(changes)
                .map(AppliedChanges::getModel)
                .select(BannerWithBannerImage.class)
                .map(BannerWithBannerImage::getId)
                .toSet();
    }

    /**
     * Возвращает список ID кампаний, которые нужно отправить на модерацию.
     * На модерацию отправляются кампании, все баннеры которых были заархивированы до вызова unarchive.
     */
    public Set<Long> collectCampaignsToModerate(Set<Long> adGroupIds, int shard) {
        if (adGroupIds.isEmpty()) {
            return Collections.emptySet();
        }

        List<BannerWithSystemFields> banners =
                bannerTypedRepository.getBannersByGroupIds(shard, adGroupIds, BannerWithSystemFields.class);

        Map<Long, List<BannerWithSystemFields>> bannersByCampaigns = StreamEx.of(banners)
                .groupingBy(BannerWithSystemFields::getCampaignId);

        return bannersByCampaigns.entrySet().stream()
                .filter(e -> e.getValue().stream().allMatch(BannerWithSystemFields::getStatusArchived))
                .map(Map.Entry::getKey)
                .collect(toSet());
    }

    /**
     * Возвращает список ID групп, в которых все баннеры заархивированы.
     */
    public Set<Long> collectArchivedAdGroups(Collection<Long> adGroups, int shard) {

        List<BannerWithSystemFields> banners =
                bannerTypedRepository.getBannersByGroupIds(shard, adGroups, BannerWithSystemFields.class);

        Map<Long, List<BannerWithSystemFields>> bannersByAdGroups = StreamEx.of(banners)
                .groupingBy(BannerWithSystemFields::getAdGroupId);

        return bannersByAdGroups.entrySet().stream()
                .filter(e -> e.getValue().stream().allMatch(BannerWithSystemFields::getStatusArchived))
                .map(Map.Entry::getKey)
                .collect(toSet());
    }

    /**
     * Подготавливает {@link AppliedChanges} для креативов и возвращает их в связке с ID баннера.
     */
    public Map<Long, AppliedChanges<Creative>> prepareCreativeChangesByBannerId(
            int shard, Collection<AppliedChanges<BannerWithSystemFields>> changes) {
        Map<Long, List<Long>> bannerIdsByCreativeIds = StreamEx.of(changes)
                .map(AppliedChanges::getModel)
                .select(BannerWithCreative.class)
                .filter(b -> b.getCreativeId() != null)
                .mapToEntry(BannerWithCreative::getCreativeId, BannerWithCreative::getId)
                .grouping();

        return StreamEx.of(creativeRepository.getCreatives(shard, bannerIdsByCreativeIds.keySet()))
                .map(c -> new ModelChanges<>(c.getId(), Creative.class).applyTo(c))
                .mapToEntry(c -> bannerIdsByCreativeIds.get(c.getModel().getId()), identity())
                .flatMapKeys(StreamEx::of)
                .toMap();
    }

}
