package ru.yandex.direct.core.entity.banner.service.moderation;

import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import one.util.streamex.StreamEx;
import org.jooq.TransactionalRunnable;

import ru.yandex.direct.core.entity.banner.container.BannerRepositoryContainer;
import ru.yandex.direct.core.entity.banner.model.BannerWithSystemFields;
import ru.yandex.direct.core.entity.banner.model.TextBanner;
import ru.yandex.direct.core.entity.banner.repository.BannerModifyRepository;
import ru.yandex.direct.core.entity.banner.repository.BannerTypedRepository;
import ru.yandex.direct.core.entity.banner.service.moderation.type.BannerModerateOperationSupportFacade;
import ru.yandex.direct.core.entity.banner.service.validation.ModerateBannerValidationService;
import ru.yandex.direct.core.entity.creative.model.Creative;
import ru.yandex.direct.core.entity.moderation.repository.sending.remoderation.RemoderationFlagsRepository;
import ru.yandex.direct.core.entity.moderation.repository.sending.remoderation.RemoderationType;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.operation.Applicability;
import ru.yandex.direct.operation.update.AppliedChangesValidatedStep;
import ru.yandex.direct.operation.update.ExecutionStep;
import ru.yandex.direct.operation.update.SimpleAbstractUpdateOperation;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.core.entity.banner.service.validation.BannerConstants.NEW_SENSITIVE_PROPERTIES;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

public class BannerRemoderateOperation extends SimpleAbstractUpdateOperation<BannerWithSystemFields, Long> {
    private final int shard;
    private final ClientId clientId;
    private final Long operatorUid;
    private final Set<String> featuresEnabledForClient;
    private final DslContextProvider dslContextProvider;
    private final RemoderationFlagsRepository remoderationFlagsRepository;
    private final ModerateBannerValidationService moderateBannerValidationService;
    private final BannerModerateOperationSupportFacade bannerModerateOperationSupportFacade;
    private final BannerTypedRepository typedRepository;
    private final BannerModifyRepository modifyRepository;
    private final BannerRepositoryContainer bannerRepositoryContainer;
    private final BannerModerationHelper bannerModerationHelper;

    private TransactionalRunnable postExecute;

    public <B extends BannerWithSystemFields> BannerRemoderateOperation(
            List<Long> bannerIds,
            int shard, long operatorUid, ClientId clientId,
            DslContextProvider dslContextProvider,
            ModerateBannerValidationService moderateBannerValidationService,
            BannerModerateOperationSupportFacade bannerModerateOperationSupportFacade,
            BannerTypedRepository typedRepository,
            BannerModifyRepository modifyRepository,
            Set<String> featuresEnabledForClient, RemoderationFlagsRepository remoderationFlagsRepository,
            BannerModerationHelper helper) {
        super(Applicability.PARTIAL,
                mapList(bannerIds, id -> new ModelChanges<>(id, BannerWithSystemFields.class)),
                id -> new TextBanner().withId(id),
                NEW_SENSITIVE_PROPERTIES);
        this.shard = shard;
        this.clientId = clientId;
        this.operatorUid = operatorUid;
        this.dslContextProvider = dslContextProvider;
        this.bannerModerationHelper = helper;
        this.remoderationFlagsRepository = remoderationFlagsRepository;
        this.moderateBannerValidationService = moderateBannerValidationService;
        this.bannerModerateOperationSupportFacade = bannerModerateOperationSupportFacade;
        this.typedRepository = typedRepository;
        this.modifyRepository = modifyRepository;
        this.featuresEnabledForClient = featuresEnabledForClient;

        bannerRepositoryContainer = new BannerRepositoryContainer(shard);
    }

    @Override
    protected ValidationResult<List<ModelChanges<BannerWithSystemFields>>, Defect> validateModelChanges(
            List<ModelChanges<BannerWithSystemFields>> modelChanges) {
        return moderateBannerValidationService.validateAccess(operatorUid, clientId, modelChanges);
    }

    @Override
    protected ValidationResult<List<ModelChanges<BannerWithSystemFields>>, Defect> validateModelChangesBeforeApply(
            ValidationResult<List<ModelChanges<BannerWithSystemFields>>, Defect> preValidateResult,
            Map<Long, BannerWithSystemFields> models) {
        preValidateResult = moderateBannerValidationService.validateRemoderationAllowed(operatorUid, clientId,
                preValidateResult, featuresEnabledForClient);
        return moderateBannerValidationService.validateReModerate(shard, clientId, preValidateResult, models);
    }

    @Override
    protected void onAppliedChangesValidated(AppliedChangesValidatedStep<BannerWithSystemFields> step) {
        Collection<AppliedChanges<BannerWithSystemFields>> changes = step.getValidAppliedChanges();
        Map<Long, AppliedChanges<Creative>> creativeChangesByBannerId =
                bannerModerationHelper.prepareCreativeChangesByBannerId(shard, changes);
        Map<Long, Set<RemoderationType>> remoderationTypesByBannerId = StreamEx.of(changes)
                .toMap(ac -> ac.getModel().getId(), ac -> new HashSet<>());
        changes.forEach(banner -> {
            long bannerId = banner.getModel().getId();
            bannerModerateOperationSupportFacade
                    .getModerateOperationSupport(banner)
                    .remoderate(banner, creativeChangesByBannerId.get(bannerId),
                            remoderationTypesByBannerId.get(bannerId));
            if (remoderationTypesByBannerId.get(bannerId).size() == 0) {
                remoderationTypesByBannerId.remove(bannerId);
            }
        });

        postExecute = conf -> remoderationFlagsRepository.addRemoderationFlags(conf, remoderationTypesByBannerId);
    }

    @Override
    protected List<Long> execute(List<AppliedChanges<BannerWithSystemFields>> changes) {
        modifyRepository.update(bannerRepositoryContainer, changes);
        return mapList(changes, a -> a.getModel().getId());
    }

    @Override
    protected void afterExecution(ExecutionStep<BannerWithSystemFields> executionStep) {
        dslContextProvider.ppc(shard).transaction(postExecute);
    }

    @Override
    protected Collection<BannerWithSystemFields> getModels(Collection<Long> ids) {
        return typedRepository.getStrictlyFullyFilled(shard, ids, BannerWithSystemFields.class);
    }
}
