package ru.yandex.direct.core.entity.banner.service.moderation;

import java.util.List;
import java.util.Map;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.jooq.DSLContext;

import ru.yandex.direct.core.entity.banner.container.BannerAdditionalActionsContainer;
import ru.yandex.direct.core.entity.banner.container.BannersUpdateOperationContainer;
import ru.yandex.direct.core.entity.banner.model.Banner;
import ru.yandex.direct.model.AppliedChanges;

import static java.util.function.Function.identity;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@ParametersAreNonnullByDefault
public class BannerWithChildrenModerationProvider<T extends Banner> {

    private BannerWithChildrenModerationProcessor<? extends T> defaultProcessor;
    private Map<? extends Class<? extends T>, ? extends BannerWithChildrenModerationProcessor<? extends T>>
            classToProcessorMap;

    public BannerWithChildrenModerationProvider(
            Class<? extends BannerWithChildrenModerationProcessor<? extends T>> defaultProcessorClass,
            List<? extends BannerWithChildrenModerationProcessor<? extends T>> processors) {
        this.defaultProcessor = processors.stream()
                .filter(p -> p.getClass().equals(defaultProcessorClass))
                .findAny()
                .orElseThrow(() -> new IllegalStateException("no default processor"));
        this.classToProcessorMap = StreamEx.of(processors)
                .remove(p -> p.getClass().equals(defaultProcessorClass))
                .mapToEntry(BannerWithChildrenModerationProcessor::getProcessedClass, identity())
                .toMap();
    }

    @SuppressWarnings("unchecked")
    public void process(
            DSLContext dsl,
            BannerAdditionalActionsContainer additionalActionsContainer,
            BannersUpdateOperationContainer updateContainer,
            List<AppliedChanges<T>> appliedChanges) {
        StreamEx.of(appliedChanges)
                .mapToEntry(change -> change.getModel().getClass(), identity())
                .collapseKeys()
                .mapKeys(this::getProcessor)
                .forKeyValue((processor, changes) -> {
                    var castChanges = mapList(changes, ch -> ch.castModelUp(processor.getProcessedClass()));
                    processor.process(
                            dsl, additionalActionsContainer,
                            updateContainer, castChanges);
                });
    }

    private BannerWithChildrenModerationProcessor getProcessor(Class<? extends Banner> clazz) {
        var processor = classToProcessorMap.get(clazz);
        return processor == null ? defaultProcessor : processor;
    }
}
