package ru.yandex.direct.core.entity.banner.service.text;

import java.util.Collection;
import java.util.IdentityHashMap;
import java.util.List;
import java.util.function.Function;

import com.google.common.collect.ImmutableList;
import one.util.streamex.StreamEx;

import static java.util.function.Function.identity;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Извлекает составной текст из произвольных объектов.
 * <p>
 * Например, если объект содержит несколько текстов, они будут извлечены
 * с помощью зарегистрированных экстракторов и конкатенированы с разделителем.
 * <p>
 * Конструктор принимает список экстракторов, каждый из которых умеет извлекать
 * текст из определенного типа объектов. Для каждого из переданных объектов
 * подбираются подходящие по типу экстракторы. Затем каждый извлекает текст
 * и извлеченные тексты конкатенируются с разделителем.
 */
@SuppressWarnings("unchecked")
public class CompoundTextExtractor {

    private static final CharSequence DEFAULT_DELIMITER = " ";

    private final List<TextExtractor> textExtractors;
    private final CharSequence delimiter;

    public CompoundTextExtractor(List<TextExtractor> textExtractors) {
        this.textExtractors = ImmutableList.copyOf(textExtractors);
        this.delimiter = DEFAULT_DELIMITER;
    }

    /**
     * В ответе присутствуют все объекты.
     * Для объектов, к которым не подошел ни один из зарегистрированных
     * экстракторов, будет возвращена пустая строка.
     */
    public <T> IdentityHashMap<T, String> extractTexts(Collection<T> objectsWithCompoundText) {
        // Ключами мап являются баннеры, а значениями - тексты.
        // В каждой мапе присутствуют баннеры, удовлетворяющие двум условиям:
        // - баннер подходит к соответствующему экстрактору
        // - тест не равен null
        //noinspection Convert2MethodRef
        List<IdentityHashMap> objectsToExtractedTextMaps = (List<IdentityHashMap>) StreamEx.of(textExtractors)
                .mapToEntry(identity(), textExtractor -> getSupportedObjects(textExtractor, objectsWithCompoundText))
                .mapKeyValue((textExtractor, supportedObjects) -> textExtractor.extractTexts((supportedObjects)))
                .toList();

        Function<Object, String> joinTextsFunction =
                object -> StreamEx.of(objectsToExtractedTextMaps)
                        .map(map -> map.get(object))
                        .nonNull()
                        .select(String.class)
                        .map(String::trim)
                        .joining(delimiter)
                        .trim();

        return StreamEx.of(objectsWithCompoundText)
                .mapToEntry(identity(), joinTextsFunction)
                .toCustomMap(IdentityHashMap::new);
    }

    public Collection<Class<?>> getClasses() {
        return mapList(textExtractors, TextExtractor::getSupportedType);
    }

    private <T> List<T> getSupportedObjects(TextExtractor textExtractor, Collection<T> objects) {
        return StreamEx.of(objects)
                .filter(object -> isObjectSupportedByTextExtractor(textExtractor, object))
                .toList();
    }

    private boolean isObjectSupportedByTextExtractor(TextExtractor textExtractor, Object object) {
        Class<?> objectClass = object.getClass();
        Class<?> stopType = textExtractor.getStopType();
        if (stopType != null && stopType.isAssignableFrom(objectClass)) {
            return false;
        }
        return textExtractor.getSupportedType().isAssignableFrom(objectClass);
    }
}
