package ru.yandex.direct.core.entity.banner.service.validation;

import java.util.Comparator;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;
import java.util.function.Supplier;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.banner.model.BannerStatusModerate;
import ru.yandex.direct.core.entity.banner.model.BannerWithHref;
import ru.yandex.direct.core.entity.banner.model.BannerWithSitelinks;
import ru.yandex.direct.core.entity.banner.model.BannerWithSystemFields;
import ru.yandex.direct.core.entity.banner.model.BannerWithTurboLanding;
import ru.yandex.direct.core.entity.campaign.model.Campaign;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.campaign.service.accesschecker.CampaignAccessDefects;
import ru.yandex.direct.core.entity.campaign.service.accesschecker.CampaignSubObjectAccessCheckerFactory;
import ru.yandex.direct.core.entity.campaign.service.accesschecker.CampaignSubObjectAccessConstraint;
import ru.yandex.direct.core.entity.campaign.service.validation.CampaignAccessType;
import ru.yandex.direct.core.entity.sitelink.model.SitelinkSet;
import ru.yandex.direct.core.entity.sitelink.repository.SitelinkSetRepository;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static java.util.function.Function.identity;
import static ru.yandex.direct.core.entity.banner.service.validation.defects.BannerDefects.adNotFound;
import static ru.yandex.direct.core.entity.banner.service.validation.defects.BannerDefects.alreadyArchived;
import static ru.yandex.direct.core.entity.banner.service.validation.defects.BannerDefects.archiveBannerShownInBs;
import static ru.yandex.direct.core.entity.banner.service.validation.defects.BannerDefects.archiveDraftBanner;
import static ru.yandex.direct.core.entity.banner.service.validation.defects.BannerDefects.archiveInArchivedCampaign;
import static ru.yandex.direct.core.entity.banner.service.validation.defects.BannerDefects.bannerHrefIsIncompatibleWithPayForConversionStrategy;
import static ru.yandex.direct.core.entity.banner.service.validation.defects.BannerDefects.bannerSitelinksHrefIsIncompatibleWithPayForConversionStrategy;
import static ru.yandex.direct.core.entity.banner.service.validation.defects.BannerDefects.isNotArchived;
import static ru.yandex.direct.core.entity.banner.service.validation.defects.BannerDefects.unarchiveInArchivedCampaign;
import static ru.yandex.direct.core.entity.banner.service.validation.defects.BannerDefects.unsupportedBannerType;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignDefects.archivedCampaignModification;
import static ru.yandex.direct.core.entity.sitelink.service.validation.SitelinkSetConstraints.isSitelinksSetContainsOnlyTurbolandings;
import static ru.yandex.direct.core.validation.defects.RightsDefects.noRights;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.filterAndMapToSet;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.validation.constraint.CollectionConstraints.unique;
import static ru.yandex.direct.validation.defect.CommonDefects.objectNotFound;

/**
 * Сервис валидации баннеров для {@link ru.yandex.direct.core.entity.banner.service.BannersArchiveUnarchiveOperation}
 */
@Service
public class ArchiveUnarchiveBannerValidationService {
    private static final Function<Long, Defect> AD_NOT_FOUND =
            objectId -> adNotFound();
    private static final CampaignAccessDefects CAMPAIGN_ACCESS_DEFECTS = new CampaignAccessDefects.Builder()
            .withTypeNotAllowable(AD_NOT_FOUND)
            .withNotVisible(AD_NOT_FOUND)
            .withTypeNotSupported(objectId -> unsupportedBannerType())
            .withNoRights(objectId -> noRights())
            .withArchivedModification(objectId -> archivedCampaignModification())
            .build();

    private final CampaignRepository campaignRepository;
    private final CampaignSubObjectAccessCheckerFactory accessCheckerFactory;
    private final SitelinkSetRepository sitelinkSetRepository;

    @Autowired
    public ArchiveUnarchiveBannerValidationService(CampaignRepository campaignRepository,
                                                   CampaignSubObjectAccessCheckerFactory accessCheckerFactory,
                                                   SitelinkSetRepository sitelinkSetRepository) {
        this.campaignRepository = campaignRepository;
        this.accessCheckerFactory = accessCheckerFactory;
        this.sitelinkSetRepository = sitelinkSetRepository;
    }

    /**
     * Предварительная валидация списка баннеров.
     *
     * @param preValidateResult результат предыдущей проверки
     * @param banners           полные модели баннеров по ID
     * @return Результат валидации списка.
     */
    public ValidationResult<List<ModelChanges<BannerWithSystemFields>>, Defect> validateBanners(
            int shard, ValidationResult<List<ModelChanges<BannerWithSystemFields>>, Defect> preValidateResult,
            Map<Long, BannerWithSystemFields> banners, boolean archive) {
        List<Campaign> campaigns = campaignRepository.getCampaigns(shard, mapList(banners.values(),
                BannerWithSystemFields::getCampaignId));

        Set<Long> campaignIdsWithPayForConversion =
                filterAndMapToSet(campaigns,
                        c -> nvl(c.getStrategy().getStrategyData().getPayForConversion(), false),
                        Campaign::getId);

        Supplier<Map<Long, BannerWithSystemFields>> bannersRelatingToCampaignWithPayForConversionByIdsSupplier =
                () ->
                        StreamEx.of(banners.values())
                                .filter(b -> campaignIdsWithPayForConversion.contains(b.getCampaignId()))
                                .toMap(BannerWithSystemFields::getId, identity());

        Supplier<Map<Long, SitelinkSet>> sitelinksSetsByBannerIdSupplier = () -> {
            Map<Long, Long> siteLinkSetIdsByBannerIds = StreamEx.of(banners.values())
                    .select(BannerWithSitelinks.class)
                    .filter(b -> b.getSitelinksSetId() != null)
                    .mapToEntry(BannerWithSitelinks::getId, BannerWithSitelinks::getSitelinksSetId)
                    .toMap();

            Map<Long, SitelinkSet> sitelinkSetsByIds = listToMap(sitelinkSetRepository.get(shard,
                    new HashSet<>(siteLinkSetIdsByBannerIds.values())),
                    SitelinkSet::getId);

            return EntryStream.of(siteLinkSetIdsByBannerIds)
                    .mapValues(sitelinkSetsByIds::get)
                    .toMap();
        };

        Map<Long, Campaign> campaignsById = listToMap(campaigns, Campaign::getId, identity());

        return new ListValidationBuilder<>(preValidateResult)
                .checkEach(bannerExists(banners), When.isValid())
                .checkEach(campaignExists(campaignsById, banners), When.isValid())
                .checkEach(campaignIsNotArchived(archive, banners, campaignsById), When.isValid())
                .checkEach(adIsNotDraft(banners), When.isValidAnd(When.isTrue(archive)))
                .checkEach(bannerIsNotShownInBs(banners), When.isValidAnd(When.isTrue(archive)))
                .weakCheckEach(adArchivedStatusCorrect(archive, banners), When.isValid())
                .getResult();
    }

    /**
     * Условие проверки для кампаний с включенной оплатой конверсий: баннер должен содержать только турбостраницу
     * или ссылку на турбостраницу
     */
    private Constraint<ModelChanges<BannerWithSystemFields>, Defect> bannerCompatibleWithPayForConverson(
            Map<Long, BannerWithSystemFields> bannersRelatesToCampaignWithPayForConversion) {

        return Constraint.fromPredicate(mc -> {
                    if (bannersRelatesToCampaignWithPayForConversion.containsKey(mc.getId())) {
                        BannerWithSystemFields banner = bannersRelatesToCampaignWithPayForConversion.get(mc.getId());
                        return banner instanceof BannerWithTurboLanding
                                && ((BannerWithTurboLanding) banner).getTurboLandingId() != null
                                && banner instanceof BannerWithHref
                                && ((BannerWithHref) banner).getHref() == null;
                    } else {
                        return true;
                    }
                },
                bannerHrefIsIncompatibleWithPayForConversionStrategy());
    }

    /**
     * Условие проверки для кампаний с включенной оплатой конверсий: сайтлинки должны содержать только турбостраницу
     * или ссылку на турбостраницу
     */
    private Constraint<ModelChanges<BannerWithSystemFields>, Defect> bannerSitelinksCompatibleWithPayForConverson(
            Map<Long, SitelinkSet> sitelikSetsByBannerId) {

        return Constraint.fromPredicate(mc -> {
                    if (sitelikSetsByBannerId.containsKey(mc.getId())) {
                        SitelinkSet sitelinksSet = sitelikSetsByBannerId.get(mc.getId());
                        return isSitelinksSetContainsOnlyTurbolandings(sitelinksSet);
                    } else {
                        return true;
                    }
                },
                bannerSitelinksHrefIsIncompatibleWithPayForConversionStrategy());
    }

    private Constraint<ModelChanges<BannerWithSystemFields>, Defect> campaignExists(Map<Long, Campaign> campaignsById,
                                                                                    Map<Long,
                                                                                            BannerWithSystemFields> banners) {
        return Constraint.fromPredicate(mc -> campaignsById.keySet().contains(banners.get(mc.getId()).getCampaignId()),
                objectNotFound());
    }

    private Constraint<ModelChanges<BannerWithSystemFields>, Defect> bannerExists(Map<Long,
            BannerWithSystemFields> banners) {
        return Constraint.fromPredicate(mc -> banners.keySet().contains(mc.getId()), objectNotFound());
    }

    private Constraint<ModelChanges<BannerWithSystemFields>, Defect> bannerIsNotShownInBs(Map<Long,
            BannerWithSystemFields> banners) {
        return Constraint.fromPredicate(mc -> (!banners.get(mc.getId()).getStatusShow()
                        || banners.get(mc.getId()).getBsBannerId() == 0),
                archiveBannerShownInBs());
    }

    /**
     * Возвращает условие проверки что баннер не является черновиком. Выполняется только для архивации.
     *
     * @param banners полные модели баннеров по ID
     * @return условие для проверки
     */
    private Constraint<ModelChanges<BannerWithSystemFields>, Defect> adIsNotDraft(Map<Long,
            BannerWithSystemFields> banners) {
        return Constraint.fromPredicate(mc -> banners.get(mc.getId()).getStatusModerate() != BannerStatusModerate.NEW,
                archiveDraftBanner());
    }

    /**
     * Возвращает условие проверки что статус архивации баннера подходит для операции
     *
     * @param archive true если баннер необходимо запустить, false если остановить
     * @param banners полные модели баннеров по ID
     * @return условие для проверки
     */
    private Constraint<ModelChanges<BannerWithSystemFields>, Defect> adArchivedStatusCorrect(boolean archive,
                                                                                             Map<Long,
                                                                                                     BannerWithSystemFields> banners) {
        return Constraint.fromPredicate(mc -> archive ^ banners.get(mc.getId()).getStatusArchived(),
                archive ? alreadyArchived() : isNotArchived());
    }

    /**
     * Возвращает условие проверки что соответствующая баннеру кампания не в архиве.
     *
     * @param archive       true если баннер необходимо запустить, false если остановить
     * @param banners       полные модели баннеров по ID
     * @param campaignsById кампании по ID
     * @return условие для проверки
     */
    private Constraint<ModelChanges<BannerWithSystemFields>, Defect> campaignIsNotArchived(
            boolean archive, Map<Long, BannerWithSystemFields> banners, Map<Long, Campaign> campaignsById) {
        return mc -> {
            if (!campaignsById.get(banners.get(mc.getId()).getCampaignId()).getStatusArchived()) {
                return null;
            }
            return archive
                    ? archiveInArchivedCampaign(banners.get(mc.getId()).getCampaignId())
                    : unarchiveInArchivedCampaign(banners.get(mc.getId()).getCampaignId());
        };
    }

    public ValidationResult<List<ModelChanges<BannerWithSystemFields>>, Defect> validateChanges(
            ClientId clientId, Long operatorUid, List<ModelChanges<BannerWithSystemFields>> modelChanges) {
        ListValidationBuilder<ModelChanges<BannerWithSystemFields>, Defect> lvb = ListValidationBuilder
                .of(modelChanges, Defect.class);
        lvb.weakCheckEach(unique(Comparator.comparing(ModelChanges::getId)), When.isValid());
        validateAccess(clientId, operatorUid, modelChanges, lvb);
        return lvb.getResult();
    }

    private void validateAccess(ClientId clientId, Long operatorUid,
                                List<ModelChanges<BannerWithSystemFields>> modelChanges,
                                ListValidationBuilder<ModelChanges<BannerWithSystemFields>, Defect> lvb) {
        CampaignSubObjectAccessConstraint constraint = accessCheckerFactory
                .newAdsChecker(operatorUid, clientId, mapList(modelChanges, ModelChanges::getId))
                .createValidator(CampaignAccessType.READ_WRITE, CAMPAIGN_ACCESS_DEFECTS)
                .getAccessConstraint();
        lvb.checkEach((Constraint<ModelChanges<BannerWithSystemFields>, Defect>) c -> constraint.apply(c.getId()),
                When.isValid());
    }
}
