package ru.yandex.direct.core.entity.banner.service.validation;

import java.util.Collection;
import java.util.Map;
import java.util.Set;

import one.util.streamex.StreamEx;

import ru.yandex.direct.core.entity.adgroup.model.AdGroupForBannerOperation;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.banner.model.Banner;
import ru.yandex.direct.core.entity.banner.model.BannerStatusModerate;
import ru.yandex.direct.core.entity.banner.model.BannerWithHrefAndTurboLanding;
import ru.yandex.direct.core.entity.banner.model.BannerWithInternalInfo;
import ru.yandex.direct.core.entity.banner.model.BannerWithModerationStatuses;
import ru.yandex.direct.core.entity.banner.model.BannerWithStatusArchived;
import ru.yandex.direct.core.entity.banner.model.BannerWithSystemFields;
import ru.yandex.direct.core.entity.banner.model.old.OldBanner;
import ru.yandex.direct.core.entity.banner.model.old.OldBannerStatusModerate;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.model.ModelProperty;
import ru.yandex.direct.model.ModelWithId;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.result.Defect;

import static com.google.common.base.Preconditions.checkState;
import static ru.yandex.direct.core.entity.banner.service.validation.BannerConstants.DEFAULT_MAX_BANNERS_IN_UAC_ADGROUP;
import static ru.yandex.direct.core.entity.banner.service.validation.BannerConstants.DEFAULT_MAX_BANNERS_IN_UAC_TEXT_ADGROUP;
import static ru.yandex.direct.core.entity.banner.service.validation.BannerConstants.MAX_BANNERS_IN_ADGROUP;
import static ru.yandex.direct.core.entity.banner.service.validation.BannerConstants.MAX_BANNERS_IN_INTERNAL_ADGROUP;
import static ru.yandex.direct.core.entity.banner.service.validation.BannerConstants.MAX_BANNERS_IN_INTERNAL_CAMPAIGN;
import static ru.yandex.direct.core.entity.banner.service.validation.BannerConstants.MAX_BANNERS_IN_UNIVERSAL_APP_CAMPAIGN;
import static ru.yandex.direct.core.entity.banner.service.validation.defects.BannerDefects.cannotUpdateArchivedAd;
import static ru.yandex.direct.core.entity.banner.service.validation.defects.BannerDefects.forbiddenToChange;
import static ru.yandex.direct.core.entity.banner.service.validation.defects.BannerDefects.inconsistentBannerType;
import static ru.yandex.direct.core.entity.banner.service.validation.defects.BannerDefects.inconsistentStateBannerTypeAndAdgroupType;
import static ru.yandex.direct.core.entity.banner.service.validation.defects.BannerDefects.maxBannersInAdGroup;
import static ru.yandex.direct.core.entity.banner.service.validation.defects.BannerDefects.maxBannersInInternalCampaign;
import static ru.yandex.direct.core.entity.banner.service.validation.defects.BannerDefects.maxBannersInUniversalAppCampaign;
import static ru.yandex.direct.core.entity.banner.service.validation.defects.BannerDefects.requiredButEmptyHrefOrTurbolandingId;
import static ru.yandex.direct.validation.builder.Constraint.fromPredicate;
import static ru.yandex.direct.validation.defect.CommonDefects.invalidValue;
import static ru.yandex.direct.validation.defect.CommonDefects.objectNotFound;
import static ru.yandex.misc.lang.StringUtils.isNotEmpty;

public class BannerConstraints {
    private BannerConstraints() {
    }

    public static <B extends OldBanner> Constraint<ModelChanges<B>, Defect> isNotArchivedOld(Map<Long, B> models) {
        return fromPredicate(mc -> {
            checkState(models.containsKey(mc.getId()), "validated banner is null");
            return !models.get(mc.getId()).getStatusArchived();
        }, cannotUpdateArchivedAd());
    }

    public static <B extends BannerWithStatusArchived> Constraint<ModelChanges<B>, Defect> isNotArchivedNew(Map<Long, B> models) {
        return fromPredicate(mc -> {
            checkState(models.containsKey(mc.getId()), "validated banner is null");
            return !models.get(mc.getId()).getStatusArchived();
        }, cannotUpdateArchivedAd());
    }

    public static <B extends OldBanner> Constraint<ModelChanges<B>, Defect> bannerStatusShowIsNotChangedOld() {
        return fromPredicate(mc -> !mc.isPropChanged(B.STATUS_SHOW), invalidValue());
    }

    public static Constraint<ModelChanges<BannerWithInternalInfo>, Defect> bannerTemplateIdIsNotChanged() {
        return fromPredicate(mc -> !mc.isPropChanged(BannerWithInternalInfo.TEMPLATE_ID), forbiddenToChange());
    }

    public static <B extends OldBanner> Constraint<ModelChanges<B>, Defect> bannerIsDraftOrNotChangedOld(
            Map<Long, B> unmodifiedModels) {

        return fromPredicate(
                mc -> {
                    var banner = unmodifiedModels.get(mc.getId());
                    return OldBannerStatusModerate.NEW.equals(banner.getStatusModerate()) || mc.getChangedPropsNames().isEmpty();
                },
                forbiddenToChange());
    }

    public static <B extends BannerWithModerationStatuses> Constraint<ModelChanges<B>, Defect> bannerIsDraftOrNotChanged(
            Map<Long, B> unmodifiedModels) {

        return fromPredicate(
                mc -> {
                    var banner = unmodifiedModels.get(mc.getId());
                    return BannerStatusModerate.NEW.equals(banner.getStatusModerate()) || !mc.isAnyPropChanged();
                },
                forbiddenToChange());
    }

    public static <B extends ModelWithId> Constraint<B, Defect> limitBannersInInternalCampaign(
            Integer campaignBannersCounter) {
        return fromPredicate(b -> campaignBannersCounter <= MAX_BANNERS_IN_INTERNAL_CAMPAIGN,
                maxBannersInInternalCampaign(MAX_BANNERS_IN_INTERNAL_CAMPAIGN));
    }

    public static <B extends ModelWithId> Constraint<B, Defect> limitBannersInUniversalAppCampaign(
            Integer campaignBannersCounter) {
        return fromPredicate(b -> campaignBannersCounter <= MAX_BANNERS_IN_UNIVERSAL_APP_CAMPAIGN,
                maxBannersInUniversalAppCampaign(MAX_BANNERS_IN_UNIVERSAL_APP_CAMPAIGN));
    }

    public static <B extends ModelWithId> Constraint<B, Defect> limitBannersInGroup(
            ModelProperty<? super B, Long> adGroupIdProperty,
            Map<Long, Long> groupBannersCounter,
            AdGroupForBannerOperation adGroup,
            boolean isUac,
            boolean isUcTextCampaign) {

        int maxBannersCount = getLimitBannersInGroup(adGroup, isUac, isUcTextCampaign);

        return fromPredicate(banner -> groupBannersCounter.get(adGroupIdProperty.get(banner)) <= maxBannersCount,
                maxBannersInAdGroup(maxBannersCount));
    }

    private static int getLimitBannersInGroup(
            AdGroupForBannerOperation adGroup,
            boolean isUac,
            boolean isUcTextCampaign) {
        return getLimitBannersInGroup(
                adGroup != null && adGroup.getType() == AdGroupType.INTERNAL,
                isUac,
                isUcTextCampaign
        );
    }

    /**
     * Вернуть разрешённое количество объявлений в группе в зависимости от типа группы или свойств кампании
     * @param isInternal — внутренняя ли группа (тип INTERNAL)
     * @param isUac — принадлежит кампании с source = uac
     * @param isUcTextCampaign — принадлежит текстовой кампании с source = uac
     * @return сколько можно иметь объявлений в такой группе
     */
    public static int getLimitBannersInGroup(
            boolean isInternal,
            boolean isUac,
            boolean isUcTextCampaign
    ) {
        if (isInternal) {
            return MAX_BANNERS_IN_INTERNAL_ADGROUP;
        } else if (isUcTextCampaign) {
            return DEFAULT_MAX_BANNERS_IN_UAC_TEXT_ADGROUP;
        } else if (isUac) {
            return DEFAULT_MAX_BANNERS_IN_UAC_ADGROUP;
        }
        return MAX_BANNERS_IN_ADGROUP;
    }

    /**
     * Проверка соотвествия типа баннера типу группы
     */
    public static <B extends Banner> Constraint<B, Defect> isBannerClassCorrespondTo(AdGroupType adGroupType) {
        return fromPredicate(b -> adGroupType == null ||
                        BannersAdGroupsTypes.isValidPair(b.getClass(), adGroupType),
                inconsistentStateBannerTypeAndAdgroupType());
    }

    /**
     * Проверка наличия хотя бы одного из двух полей: ссылка или турболендинг
     */
    public static <B extends BannerWithHrefAndTurboLanding> Constraint<B, Defect> hrefOrTurboIsSet() {
        return fromPredicate(
                b -> isNotEmpty(b.getHref()) || b.getTurboLandingId() != null,
                requiredButEmptyHrefOrTurbolandingId());
    }

    public static Constraint<Long, Defect> bannerExists(Set<Long> existingAdIds) {
        return fromPredicate(existingAdIds::contains, objectNotFound());
    }

    /**
     * Проверяет, что класс объявления является наследником определенного класса.
     * Например, TextBanner наследник BannerWithSitelinks.
     */
    public static Constraint<BannerWithSystemFields, Defect> bannerIsSubClassOf(Class<?> superClass) {
        return fromPredicate(banner -> superClass.isAssignableFrom(banner.getClass()), inconsistentBannerType());
    }

    /**
     * Проверяет, что класс объявления является наследником одного из списка класса.
     */
    public static Constraint<BannerWithSystemFields, Defect> bannerIsSubClassOfOneOf(Collection<Class<?>> superClasses) {
        return fromPredicate(banner -> StreamEx.of(superClasses)
                .anyMatch(aClass -> aClass.isAssignableFrom(banner.getClass())), inconsistentBannerType());
    }
}
